//
// uVGAIII is a class to drive 4D Systems TFT touch screens
//
// Copyright (C) <2010> Stephane ROCHON <stephane.rochon at free.fr>
//
// uVGAIII is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// uVGAIII is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with uVGAIII.  If not, see <http://www.gnu.org/licenses/>.

#include "mbed.h"
#include "uVGAIII.h"

#define ARRAY_SIZE(X) sizeof(X)/sizeof(X[0])

//**************************************************************************
void uVGAIII :: cls() {  // clear screen
    
    char command[2] = "";
    command[0] = ( CLS >> 8 ) & 0xFF;
    command[1] = CLS & 0xFF;
    writeCOMMAND(command, 2, 1);
    
#if DEBUGMODE
    pc.printf("Clear screen completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: change_color(int oldColor, int newColor) {   //  change all old color pixels to new color within the clipping window area
    char command[6] = "";
    
    command[0] = (CHANGECOLOR >> 8) & 0xFF; 
    command[1] = CHANGECOLOR & 0xFF;
    
    int red5   = (oldColor >> (16 + 3)) & 0x1F;              // get red on 5 bits
    int green6 = (oldColor >> (8 + 2))  & 0x3F;              // get green on 6 bits
    int blue5  = (oldColor >> (0 + 3))  & 0x1F;              // get blue on 5 bits
    
    command[2] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;  // first part of 16 bits color
    command[3] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color
    
    red5   = (newColor >> (16 + 3)) & 0x1F;              // get red on 5 bits
    green6 = (newColor >> (8 + 2))  & 0x3F;              // get green on 6 bits
    blue5  = (newColor >> (0 + 3))  & 0x1F;              // get blue on 5 bits

    command[4] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;  // first part of 16 bits color
    command[5] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color
    
    writeCOMMAND(command, 6, 1);

#if DEBUGMODE
    pc.printf("Change color completed.\n");    
#endif
}

//****************************************************************************************************
void uVGAIII :: background_color(int color) {            // set screen background color

    char command[4]= "";                                  // input color is in 24bits like 0xRRGGBB

    command[0] = ( BCKGDCOLOR >> 8 ) & 0xFF;
    command[1] = BCKGDCOLOR & 0xFF;
    
    int red5   = (color >> (16 + 3)) & 0x1F;              // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;              // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;              // get blue on 5 bits

    command[2] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;  // first part of 16 bits color
    command[3] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    //command[2] = 0x00;
    //command[3] = 0x10;

    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE    
    pc.printf("Set background color completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: screen_mode(char mode) {   // set the graphics orientation LANDSCAPE, LANDSCAPE_R, PORTRAIT, PORTRAIT_R
    char command[4]= "";

    command[0] = (SCREENMODE >> 8) & 0xFF;
    command[1] = SCREENMODE & 0xFF;
    command[2] = 0x00;
    command[3] = mode;

    switch (mode) {
        case LANDSCAPE :
            current_orientation = IS_LANDSCAPE;
            break;
        case LANDSCAPE_R :
            current_orientation = IS_LANDSCAPE;
            break;
        case PORTRAIT :
            current_orientation = IS_PORTRAIT;
            break;
        case PORTRAIT_R :
            current_orientation = IS_PORTRAIT;
            break;
    }

    writeCOMMAND(command, 4, 3);

#if DEBUGMODE
    pc.printf("Screen mode completed.\n");
#endif

    set_font(current_font);
}

//****************************************************************************************************
void uVGAIII :: circle(int x, int y , int radius, int color) {   // draw a circle in (x,y)
    char command[10]= "";

    command[0] = (CIRCLE >> 8) & 0xFF;
    command[1] = CIRCLE & 0xFF;

    command[2] = (x >> 8) & 0xFF;
    command[3] = x & 0xFF;

    command[4] = (y >> 8) & 0xFF;
    command[5] = y & 0xFF;

    command[6] = (radius >> 8) & 0xFF;
    command[7] = radius & 0xFF;

    int red5   = (color >> (16 + 3)) & 0x1F;              // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;              // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;              // get blue on 5 bits

    command[8] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;  // first part of 16 bits color
    command[9] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    writeCOMMAND(command, 10, 1);
    
#if DEBUGMODE
    pc.printf("Draw circle completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: filled_circle(int x, int y , int radius, int color) {   // draw a filled circle in (x,y)
    char command[10]= "";

    command[0] = (CIRCLE_F >> 8) & 0xFF;
    command[1] = CIRCLE_F & 0xFF;

    command[2] = (x >> 8) & 0xFF;
    command[3] = x & 0xFF;

    command[4] = (y >> 8) & 0xFF;
    command[5] = y & 0xFF;

    command[6] = (radius >> 8) & 0xFF;
    command[7] = radius & 0xFF;

    int red5   = (color >> (16 + 3)) & 0x1F;              // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;              // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;              // get blue on 5 bits

    command[8] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;  // first part of 16 bits color
    command[9] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    writeCOMMAND(command, 10, 1);
    
#if DEBUGMODE
    pc.printf("Draw filled circle completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: triangle(int x1, int y1 , int x2, int y2, int x3, int y3, int color) {   // draw a traingle
    char command[16]= "";

    command[0] = (TRIANGLE >> 8) & 0xFF;
    command[1] = TRIANGLE & 0xFF;

    command[2] = (x1 >> 8) & 0xFF;
    command[3] = x1 & 0xFF;

    command[4] = (y1 >> 8) & 0xFF;
    command[5] = y1 & 0xFF;

    command[6] = (x2 >> 8) & 0xFF;
    command[7] = x2 & 0xFF;

    command[8] = (y2 >> 8) & 0xFF;
    command[9] = y2 & 0xFF;

    command[10] = (x3 >> 8) & 0xFF;
    command[11] = x3 & 0xFF;

    command[12] = (y3 >> 8) & 0xFF;
    command[13] = y3 & 0xFF;

    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits

    command[14] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;  // first part of 16 bits color
    command[15] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    writeCOMMAND(command, 16, 1);
    
#if DEBUGMODE
    pc.printf("Draw triangle completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: filled_triangle(int x1, int y1 , int x2, int y2, int x3, int y3, int color) {   // draw a filled traingle
    char command[16]= "";

    command[0] = (TRIANGLE_F >> 8) & 0xFF;
    command[1] = TRIANGLE_F & 0xFF;

    command[2] = (x1 >> 8) & 0xFF;
    command[3] = x1 & 0xFF;

    command[4] = (y1 >> 8) & 0xFF;
    command[5] = y1 & 0xFF;

    command[6] = (x2 >> 8) & 0xFF;
    command[7] = x2 & 0xFF;

    command[8] = (y2 >> 8) & 0xFF;
    command[9] = y2 & 0xFF;

    command[10] = (x3 >> 8) & 0xFF;
    command[11] = x3 & 0xFF;

    command[12] = (y3 >> 8) & 0xFF;
    command[13] = y3 & 0xFF;

    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits

    command[14] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;  // first part of 16 bits color
    command[15] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    writeCOMMAND(command, 16, 1);
    
#if DEBUGMODE
    pc.printf("Draw filled triangle completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: line(int x1, int y1 , int x2, int y2, int color) {   // draw a line
    char command[12]= "";

    command[0] = (LINE >> 8) & 0xFF;
    command[1] = LINE & 0xFF;

    command[2] = (x1 >> 8) & 0xFF;
    command[3] = x1 & 0xFF;

    command[4] = (y1 >> 8) & 0xFF;
    command[5] = y1 & 0xFF;

    command[6] = (x2 >> 8) & 0xFF;
    command[7] = x2 & 0xFF;

    command[8] = (y2 >> 8) & 0xFF;
    command[9] = y2 & 0xFF;

    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits

    command[10] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;   // first part of 16 bits color
    command[11] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    writeCOMMAND(command, 12, 1);
    
#if DEBUGMODE
    pc.printf("Draw line completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: rectangle(int x1, int y1 , int x2, int y2, int color) {   // draw a rectangle
    char command[12]= "";

    command[0] = (RECTANGLE >> 8) & 0xFF;
    command[1] = RECTANGLE & 0xFF;

    command[2] = (x1 >> 8) & 0xFF;
    command[3] = x1 & 0xFF;

    command[4] = (y1 >> 8) & 0xFF;
    command[5] = y1 & 0xFF;

    command[6] = (x2 >> 8) & 0xFF;
    command[7] = x2 & 0xFF;

    command[8] = (y2 >> 8) & 0xFF;
    command[9] = y2 & 0xFF;

    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits

    command[10] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;   // first part of 16 bits color
    command[11] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    writeCOMMAND(command, 12, 1);
    
#if DEBUGMODE
    pc.printf("Draw rectangle completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: filled_rectangle(int x1, int y1 , int x2, int y2, int color) {   // draw a filled rectangle
    char command[12]= "";

    command[0] = (RECTANGLE_F >> 8) & 0xFF;
    command[1] = RECTANGLE_F & 0xFF;

    command[2] = (x1 >> 8) & 0xFF;
    command[3] = x1 & 0xFF;

    command[4] = (y1 >> 8) & 0xFF;
    command[5] = y1 & 0xFF;

    command[6] = (x2 >> 8) & 0xFF;
    command[7] = x2 & 0xFF;

    command[8] = (y2 >> 8) & 0xFF;
    command[9] = y2 & 0xFF;

    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits

    command[10] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;   // first part of 16 bits color
    command[11] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    writeCOMMAND(command, 12, 1);
    
#if DEBUGMODE
    pc.printf("Draw filled rectangle completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: ellipse(int x, int y , int radius_x, int radius_y, int color) {   // draw an ellipse
    char command[12]= "";

    command[0] = (ELLIPSE >> 8) & 0xFF;
    command[1] = ELLIPSE & 0xFF;

    command[2] = (x >> 8) & 0xFF;
    command[3] = x & 0xFF;

    command[4] = (y >> 8) & 0xFF;
    command[5] = y & 0xFF;

    command[6] = (radius_x >> 8) & 0xFF;
    command[7] = radius_x & 0xFF;

    command[8] = (radius_y >> 8) & 0xFF;
    command[9] = radius_y & 0xFF;

    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits

    command[10] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;   // first part of 16 bits color
    command[11] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    writeCOMMAND(command, 12, 1);
    
#if DEBUGMODE
    pc.printf("Draw ellipse completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: filled_ellipse(int x, int y , int radius_x, int radius_y, int color) {   // draw a filled ellipse
    char command[12] = "";

    command[0] = (ELLIPSE_F >> 8) & 0xFF;
    command[1] = ELLIPSE_F & 0xFF;

    command[2] = (x >> 8) & 0xFF;
    command[3] = x & 0xFF;

    command[4] = (y >> 8) & 0xFF;
    command[5] = y & 0xFF;

    command[6] = (radius_x >> 8) & 0xFF;
    command[7] = radius_x & 0xFF;

    command[8] = (radius_y >> 8) & 0xFF;
    command[9] = radius_y & 0xFF;

    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits

    command[10] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;   // first part of 16 bits color
    command[11] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    writeCOMMAND(command, 12, 1);
    
#if DEBUGMODE
    pc.printf("Draw filled ellipse completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: button(int state, int x, int y, int buttoncolor, int txtcolor, int font, int txtWidth, int txtHeight, char * text) {  // draw a button
    char command[1000] = "";
    int size = strlen(text);
    int i = 0;
    
    command[0] = (BUTTON >> 8) & 0xFF;
    command[1] = BUTTON & 0xFF;
    
    command[2] = (state >> 8) & 0xFF;
    command[3] = state & 0xFF;
    
    command[4] = (x >> 8) & 0xFF;
    command[5] = x & 0xFF;
    
    command[6] = (y >> 8) & 0xFF;
    command[7] = y & 0xFF;
    
    int red5   = (buttoncolor >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (buttoncolor >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (buttoncolor >> (0 + 3))  & 0x1F;               // get blue on 5 bits
    
    command[8] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;
    command[9] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;
    
    red5   = (txtcolor >> (16 + 3)) & 0x1F;               // get red on 5 bits
    green6 = (txtcolor >> (8 + 2))  & 0x3F;               // get green on 6 bits
    blue5  = (txtcolor >> (0 + 3))  & 0x1F;               // get blue on 5 bits
    
    command[10] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;
    command[11] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;
    
    command[12] = (font >> 8) & 0xFF;
    command[13] = font & 0xFF;
    
    command[14] = (txtWidth >> 8) & 0xFF;
    command[15] = txtWidth & 0xFF;
    
    command[16] = (txtHeight >> 8) & 0xFF;
    command[17] = txtHeight & 0xFF;
    
    for (i=0; i<size; i++) command[18+i] = text[i];
    
    command[18+size] = 0;
    
    writeCOMMAND(command, 19 + size, 1);
    
#if DEBUGMODE
    pc.printf("Draw button completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: panel(int state, int x, int y, int Width, int Height, int color) {    // draw a panel
    char command[14] = "";
    
    command[0] = (PANEL >> 8) & 0xFF;
    command[1] = PANEL & 0xFF;
    
    command[2] = (state >> 8) & 0xFF;
    command[3] = state & 0xFF;
    
    command[4] = (x >> 8) & 0xFF;
    command[5] = x & 0xFF;
    
    command[6] = (y >> 8) & 0xFF;
    command[7] = y & 0xFF;
    
    command[8] = (Width >> 8) & 0xFF;
    command[9] = Width & 0xFF;
    
    command[10] = (Height >> 8) & 0xFF;
    command[11] = Height & 0xFF;
    
    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits
    
    command[12] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;
    command[13] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;
    
    writeCOMMAND(command, 14, 1);
    
#if DEBUGMODE
    pc.printf("Draw panel completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: slider(char mode, int x1, int y1, int x2, int y2, int color, int scale, int value) { // draw a slider
    char command[18] = "";
    
    command[0] = (SLIDER >> 8) & 0xFF;
    command[1] = SLIDER & 0xFF;
    
    command[2] = 0x00;
    command[3] = mode & 0xFF;
    
    command[4] = (x1 >> 8) & 0xFF;
    command[5] = x1 & 0xFF;
    
    command[6] = (y1 >> 8) & 0xFF;
    command[7] = y1 & 0xFF;
    
    command[8] = (x2 >> 8) & 0xFF;
    command[9] = x2 & 0xFF;
    
    command[10] = (y2 >> 8) & 0xFF;
    command[11] = y2 & 0xFF;
    
    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits
    
    command[12] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;
    command[13] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;
    
    command[14] = (scale >> 8) & 0xFF;
    command[15] = scale & 0xFF;
    
    command[16] = (value) & 0xFF;
    command[17] = value & 0xFF;
    
    writeCOMMAND(command, 18, 1);
    
#if DEBUGMODE
    pc.printf("Draw slider completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: put_pixel(int x, int y, int color) {   // draw a pixel
    char command[8] = "";

    command[0] = (PUTPIXEL >> 8) & 0xFF;
    command[1] = PUTPIXEL & 0xFF;

    command[2] = (x >> 8) & 0xFF;
    command[3] = x & 0xFF;

    command[4] = (y >> 8) & 0xFF;
    command[5] = y & 0xFF;

    int red5   = (color >> (16 + 3)) & 0x1F;              // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;              // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;              // get blue on 5 bits

    command[6] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;  // first part of 16 bits color
    command[7] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color

    writeCOMMAND(command, 8, 1);
    
#if DEBUGMODE
    pc.printf("Put pixel completed.\n");
#endif
}

//******************************************************************************************************
int uVGAIII :: read_pixel(int x, int y) { // read screen info and populate data

    char command[6]= "";

    command[0] = (READPIXEL >> 8) & 0xFF;
    command[1] = READPIXEL & 0xFF;

    command[2] = (x >> 8) & 0xFF;
    command[3] = x & 0xFF;

    command[4] = (y >> 8) & 0xFF;
    command[5] = y & 0xFF;

    int i, temp = 0, color = 0, resp = 0;
    char response[3] = "";

    freeBUFFER();
    
#if DEBUGMODE
    pc.printf("\n");
    pc.printf("New COMMAND : 0x%02X%02X\n", command[0], command[1]);
#endif

    for (i = 0; i < 6; i++) {                   // send all chars to serial port
        writeBYTE(command[i]);
    }

    while (!_cmd.readable()) wait_ms(TEMPO);    // wait for screen answer

    while (_cmd.readable() && resp < ARRAY_SIZE(response)) {
        temp = _cmd.getc();
#if DEBUGMODE
        pc.printf("Response = 0x%02X\n",int(temp));
#endif
        response[resp++] = (char)temp;
    }

    color = ((response[1] << 8) + response[2]);
    
#if DEBUGMODE
    pc.printf("Read pixel completed.\n");
#endif

    return color; // WARNING : this is 16bits color, not 24bits... need to be fixed
}

//******************************************************************************************************
void uVGAIII :: screen_copy(int xs, int ys , int xd, int yd , int width, int height) {   //  copy an area of a screen from xs, ys of size given by width and height  parameters and pastes it to another location determined by xd, yd

    char command[14]= "";

    command[0] = (SCREENCOPY >> 8) & 0xFF;
    command[1] = SCREENCOPY & 0xFF;

    command[2] = (xs >> 8) & 0xFF;
    command[3] = xs & 0xFF;

    command[4] = (ys >> 8) & 0xFF;
    command[5] = ys & 0xFF;

    command[6] = (xd >> 8) & 0xFF;
    command[7] = xd & 0xFF;

    command[8] = (yd >> 8) & 0xFF;
    command[9] = yd & 0xFF;

    command[10] = (width >> 8) & 0xFF;
    command[11] = width & 0xFF;

    command[12] = (height >> 8) & 0xFF;
    command[13] = height & 0xFF;

    writeCOMMAND(command, 14, 1);
    
#if DEBUGMODE
    pc.printf("Screen copy completed.\n");
#endif
}

//******************************************************************************************************
void uVGAIII :: clipping(char value) {  //  To enable or disable the ability for Clippling to be used
    char command[4] = "";
    
    command[0] = (CLIPPING >> 8) & 0xFF;
    command[1] = CLIPPING & 0xFF;
    
    command[2] = 0x00;
    command[3] = value & 0xFF;
    
    writeCOMMAND(command, 4, 1);
    
#if DEBUGMODE
    pc.printf("Clipping completed.\n");
#endif    
}

//******************************************************************************************************
void uVGAIII :: set_clipping_win(int x1, int y1, int x2, int y2) {   //   Set the clipping window region
    char command[10] = "";
    
    command[0] = (SETCLIPWIN >> 8) & 0xFF;
    command[1] = SETCLIPWIN & 0xFF;
    
    command[2] = (x1 >> 8) & 0xFF;
    command[3] = x1 & 0xFF;
    
    command[4] = (y1 >> 8) & 0xFF;
    command[5] = y1 & 0xFF;

    command[6] = (x2 >> 8) & 0xFF;
    command[7] = x2 & 0xFF;

    command[8] = (y2 >> 8) & 0xFF;
    command[9] = y2 & 0xFF;
    
    writeCOMMAND(command, 10, 1);
    
#if DEBUGMODE
    pc.printf("Set clipping window completed.\n");    
#endif    
}

//******************************************************************************************************
void uVGAIII :: extend_clip_region() {      //    to force the clip region to the extent of the last text that was printed, or the last image that was shown
    char command[2] = "";
    
    command[0] = (EXTCLIPREG) & 0xFF;
    command[1] = EXTCLIPREG & 0xFF;    
    
    writeCOMMAND(command, 2, 1);
    
#if DEBUGMODE
    pc.printf("Extend clip region completed.\n");
#endif
}

//******************************************************************************************************
void uVGAIII :: move_origin(int xpos, int ypos) {   //   Move the origin to a new position
    char command[6] = "";
    
    command[0] = (MOVEORIGIN) & 0xFF;
    command[1] = MOVEORIGIN & 0xFF;    

    command[2] = (xpos >> 8) & 0xFF;
    command[3] = xpos & 0xFF;
    
    command[4] = (ypos >> 8) & 0xFF;
    command[5] = ypos & 0xFF;
    
    writeCOMMAND(command, 6, 1);
    
#if DEBUGMODE
    pc.printf("Move origin completed.\n");
#endif
}

//******************************************************************************************************
void uVGAIII :: line_pattern(int pattern) {    //   Set the line pattern
    char command[4] = "";                       //   If set to zero means lines are solid, else each '1' bit represents a pixel that is turned off
    
    command[0] = (LINEPATTERN) & 0xFF;
    command[1] = LINEPATTERN & 0xFF;    

    command[2] = (pattern >> 8) & 0xFF;
    command[3] = pattern & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Line pattern completed.\n");
#endif
}

//******************************************************************************************************
void uVGAIII :: transparency(char mode) {    //   set whether to enable the transparency
    char command[4] = "";
    
    command[0] = (TRANSPARENCY) & 0xFF;
    command[1] = TRANSPARENCY & 0xFF;    

    command[2] = 0x00;
    command[3] = mode & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Transparency completed.\n");
#endif
}

//******************************************************************************************************
void uVGAIII :: outline_color(int color) {   //  set the outline color for rectangles and circles
    char command[4] = "";
    
    command[0] = (OUTLINECOLOR) & 0xFF;
    command[1] = OUTLINECOLOR & 0xFF;    
    
    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits
    
    command[2] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;
    command[3] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Outline color completed.\n");
#endif
}

//******************************************************************************************************
void uVGAIII :: transparent_color(int color) {    // Set the color to be transparent
    char command[4] = "";
    
    command[0] = (TRANSPCOLOR) & 0xFF;
    command[1] = TRANSPCOLOR & 0xFF;    
    
    int red5   = (color >> (16 + 3)) & 0x1F;               // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;               // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;               // get blue on 5 bits
    
    command[2] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;
    command[3] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Transparent color completed.\n");
#endif
}

//******************************************************************************************************
void uVGAIII :: graphics_parameters(int function, int value) {     //Set graphics parameters
    char command[6] = "";
    
    command[0] = (PARAMETERS >> 8) & 0xFF;
    command[1] = PARAMETERS & 0xFF;
    
    command[2] = (function >> 8) & 0xFF;
    command[3] = function & 0xFF;
    
    command[4] = (value >> 8) & 0xFF;
    command[5] = value & 0xFF;
    
    writeCOMMAND(command, 6, 1); 
    
#if DEBUGMODE
    pc.printf("Graphics parameters completed.\n");
#endif
} 