//
// uVGAIII is a class to drive 4D Systems TFT touch screens
//
// Copyright (C) <2010> Stephane ROCHON <stephane.rochon at free.fr>
//
// uVGAIII is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// uVGAIII is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with uVGAIII.  If not, see <http://www.gnu.org/licenses/>.

#include "mbed.h"
#include "uVGAIII.h"

//****************************************************************************************************
void uVGAIII :: set_font(char mode) {          // set font size
    char command[4] = "";

    command[0] = (SETFONT >> 8) & 0xFF;
    command[1] = SETFONT & 0xFF;
    command[2] = 0;
    command[3] = mode;

    current_font = mode;

    if (current_orientation == IS_PORTRAIT) {
        current_w = SIZE_X;
        current_h = SIZE_Y;
    } else {
        current_w = SIZE_Y;
        current_h = SIZE_X;
    }

    switch (mode) {
        case FONT1 :
            current_fx = 7;
            current_fy = 8;
            break;
        case FONT2 :
            current_fx = 8;
            current_fy = 8;
            break;
        case FONT3 :
            current_fx = 8;
            current_fy = 12;
            break;
    }

    max_col = current_w / current_fx;
    max_row = current_h / current_fy;

    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Set font completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: char_width(char c) {          // Get the width in pixel units for a character
    char command[3] = "";
    
    command[0] = (CHARWIDTH >> 8) & 0xFF;
    command[1] = CHARWIDTH & 0xFF;
    
    command[2] = c;
    
    writeCOMMAND(command, 3, 3);
    
#if DEBUGMODE
    pc.printf("Character width completed.\n");
#endif    
}

//****************************************************************************************************
void uVGAIII :: char_height(char c) {          // Get the height in pixel units for a character
    char command[3] = "";
    
    command[0] = (CHARHEIGHT >> 8) & 0xFF;
    command[1] = CHARHEIGHT & 0xFF;
    
    command[2] = c;
    
    writeCOMMAND(command, 3, 3);
    
#if DEBUGMODE
    pc.printf("Character height completed.\n");
#endif    
}

//****************************************************************************************************
void uVGAIII :: text_opacity(char mode) {   // set text mode
    char command[4] = "";

    command[0] = (TEXTOPACITY >> 8) & 0xFF;
    command[1] = TEXTOPACITY & 0xFF;
    command[2] = 0x00;
    command[3] = mode;

    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Text opacity completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_width(int multiplier) {      // Set teh text width multiplier between 1 and 16
    char command[4] = "";
         
    command[0] = (TEXTWIDTH >> 8) & 0xFF;
    command[1] = TEXTWIDTH & 0xFF;
    
    command[2] = (multiplier >> 8) & 0xFF;
    command[3] = multiplier & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Text width completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_height(int multiplier) {     // Set teh text height multiplier between 1 and 16
    char command[4] = "";
         
    command[0] = (TEXTHEIGHT >> 8) & 0xFF;
    command[1] = TEXTHEIGHT & 0xFF;
    
    command[2] = (multiplier >> 8) & 0xFF;
    command[3] = multiplier & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Text width completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_y_gap(int pixelcount) {    //  set the pixel gap between characters(y-axis), and the gap is in pixel units
    char command[4] = "";
    
    command[0] = (TEXTYGAP >> 8) & 0xFF;
    command[1] = TEXTYGAP & 0xFF;
    
    command[2] = (pixelcount >> 8) & 0xFF;
    command[3] = pixelcount & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Text Y-gap completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_x_gap(int pixelcount) {     //  set the pixel gap between characters(x-axis), and the gap is in pixel units
    char command[4] = "";
    
    command[0] = (TEXTXGAP >> 8) & 0xFF;
    command[1] = TEXTXGAP & 0xFF;
    
    command[2] = (pixelcount >> 8) & 0xFF;
    command[3] = pixelcount & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Text X-gap completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_bold(char mode) {     //  set the Bold attribute for the text
    char command[4] = "";
    
    command[0] = (TEXTBOLD >> 8) & 0xFF;
    command[1] = TEXTBOLD & 0xFF;
    
    command[2] = 0x00;
    command[3] = mode & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Text bold completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_inverse(char mode) {    //  set the text foreground and background color to be inverse
    char command[4] = "";
    
    command[0] = (TEXTINVERSE >> 8) & 0xFF;
    command[1] = TEXTINVERSE & 0xFF;
    
    command[2] = 0x00;
    command[3] = mode & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Text inverse completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_italic(char mode) {    //  set the text to italic
    char command[4] = "";
    
    command[0] = (TEXTITALIC >> 8) & 0xFF;
    command[1] = TEXTITALIC & 0xFF;
    
    command[2] = 0x00;
    command[3] = mode & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Text italic completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_underline(char mode) {   // Set whether the text to be underlined
    char command[4] = "";                      // For text underline to work, text Y-gap must be set to at least 2
    
    command[0] = (TEXTUNDLINE >> 8) & 0xFF;
    command[1] = TEXTUNDLINE & 0xFF;
    
    command[2] = 0x00;
    command[3] = mode & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Text underline completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_attributes(int value) {   // Set text attributes: bold, italic, inverse, underlined
    char command[4] = "";                       // For text underline to work, text Y-gap must be set to at least 2
    
    command[0] = (TEXTATTRIBU >> 8) & 0xFF;
    command[1] = TEXTATTRIBU & 0xFF;
    
    command[2] = (value >> 8) & 0xFF;
    command[3] = value & 0xFF;
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Text attributes completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: move_cursor(int line, int column) {   // move cursor
    char command[6] = "";
    
    command[0] = (MOVECURSOR >> 8) & 0xFF;
    command[1] = MOVECURSOR & 0xFF;
    
    command[2] = (line >> 8) & 0xFF;
    command[3] = line & 0xFF;
    
    command[4] = (column >> 8) & 0xFF;
    command[5] = column & 0xFF;
    
    writeCOMMAND(command, 6, 1);
    current_row = line;
    current_col = column;
    
#if DEBUGMODE
    pc.printf("Move cursor completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: put_char(char c) {                       // draw a text char
    char command[4] = "";

    command[0] = (PUTCHAR >> 8) & 0xFF;
    command[1] = PUTCHAR & 0xFF;

    command[2] = 0x00;
    command[3] = c;

    writeCOMMAND(command, 4, 1);
    
#if DEBUGMODE
    pc.printf("Put character completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: put_string(char *s) {                    // draw a text string

    char command[1000]= "";
    int size = strlen(s);
    int i = 0;

    command[0] = (PUTSTRING >> 8) & 0xFF;
    command[1] = PUTSTRING & 0xFF;

    for (i=0; i<size; i++) command[2+i] = s[i];

    command[2+size] = 0;

    writeCOMMAND(command, 3 + size, 1);
    
#if DEBUGMODE
    pc.printf("Put string completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_fgd_color(int color) {    //  Set text foreground color 
    char command[4] = "";
    
    command[0] = (TEXTFGCOLOR >> 8) & 0xFF;
    command[1] = TEXTFGCOLOR & 0xFF;
    
    current_color = color;
    
    int red5   = (color >> (16 + 3)) & 0x1F;              // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;              // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;              // get blue on 5 bits

    command[2] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;  // first part of 16 bits color
    command[3] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Set text foreground color completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: text_bgd_color(int color) {  //  Set text background color 
    char command[4] = "";
    
    command[0] = (TEXTBGCOLOR >> 8) & 0xFF;
    command[1] = TEXTBGCOLOR & 0xFF;
    
    int red5   = (color >> (16 + 3)) & 0x1F;              // get red on 5 bits
    int green6 = (color >> (8 + 2))  & 0x3F;              // get green on 6 bits
    int blue5  = (color >> (0 + 3))  & 0x1F;              // get blue on 5 bits

    command[2] = ((red5 << 3)   + (green6 >> 3)) & 0xFF;  // first part of 16 bits color
    command[3] = ((green6 << 5) + (blue5 >>  0)) & 0xFF;  // second part of 16 bits color
    
    writeCOMMAND(command, 4, 3);
    
#if DEBUGMODE
    pc.printf("Set text foreground color completed.\n");
#endif
}

//****************************************************************************************************
void uVGAIII :: putc(char c)      // place char at current cursor position
                                   //used by virtual printf function _putc
{
#if DEBUGMODE
    pc.printf("\nCursor position: %d, %d",current_row,current_col);
#endif
    if(c<0x20) {
        if(c=='\n') {
            current_col = 0;
            current_row++;
            move_cursor(current_row, current_col);
        }
        if(c=='\r') {
            current_col = 0;
            move_cursor(current_row, current_col);
        }
        if(c=='\f') {
            cls();          //clear screen on form feed
        }
    } else {
        put_char(c);
        current_col++;
        move_cursor(current_row, current_col);
    }
    if (current_col >= max_col) {
        current_col = 0;
        current_row++;
        move_cursor(current_row, current_col);
    }
    if (current_row >= max_row) {
        current_row = 0;
        move_cursor(current_row, current_col);
    }
}

//****************************************************************************************************
void uVGAIII :: puts(char *s) {   // place string at current cursor position

    put_string(s);

    current_col += strlen(s);

    if (current_col >= max_col) {
        current_row += current_col / max_col;
        current_col %= max_col;
    }
    if (current_row >= max_row) {
        current_row %= max_row;
    }
}