/*Use #define MPU6050_ES before you include this file if you have an engineering sample (older EVBs will have them), to find out if you have one, check your accelerometer output. 
If it is half of what you expected, and you still are on the correct planet, you got an engineering sample
*/


#ifndef MPU6050_H
#define MPU6050_H

/**
 * Includes
 */
#include "mbed.h"


/**
 * Defines
 */
#ifndef MPU6050_ADDRESS
    #define MPU6050_ADDRESS             0x68 // address pin low (GND), default for InvenSense evaluation board
#endif

#ifdef MPU6050_ES
        #define DOUBLE_ACCELERO
#endif  

/**
 * Registers
 */
 
#define MPU6050_XGOFFS_TC        0x00 // Bit 7 PWR_MODE, bits 6:1 XG_OFFS_TC, bit 0 OTP_BNK_VLD                 
#define MPU6050_YGOFFS_TC        0x01                                                                          
#define MPU6050_ZGOFFS_TC        0x02
#define MPU6050_X_FINE_GAIN      0x03 // [7:0] fine gain
#define MPU6050_Y_FINE_GAIN      0x04
#define MPU6050_Z_FINE_GAIN      0x05
#define MPU6050_XA_OFFSET_H      0x06 // User-defined trim values for accelerometer
#define MPU6050_XA_OFFSET_L_TC   0x07
#define MPU6050_YA_OFFSET_H      0x08
#define MPU6050_YA_OFFSET_L_TC   0x09
#define MPU6050_ZA_OFFSET_H      0x0A
#define MPU6050_ZA_OFFSET_L_TC   0x0B
#define MPU6050_SELF_TEST_X      0x0D
#define MPU6050_SELF_TEST_Y      0x0E    
#define MPU6050_SELF_TEST_Z      0x0F
#define MPU6050_SELF_TEST_A      0x10
#define MPU6050_XG_OFFS_USRH     0x13  // User-defined trim values for gyroscope; supported in MPU-6050?
#define MPU6050_XG_OFFS_USRL     0x14
#define MPU6050_YG_OFFS_USRH     0x15
#define MPU6050_YG_OFFS_USRL     0x16
#define MPU6050_ZG_OFFS_USRH     0x17
#define MPU6050_ZG_OFFS_USRL     0x18
#define MPU6050_SMPLRT_DIV       0x19
#define MPU6050_CONFIG           0x1A

// Filter bandwidth of accelerometer setting
#define MPU6050_EXT_FSYNC_TEMP   0x08
#define MPU6050_DLPF_CFG_260     0x00
#define MPU6050_DLPF_CFG_184     0x01
#define MPU6050_DLPF_CFG_94      0x02
#define MPU6050_DLPF_CFG_44      0x03
#define MPU6050_DLPF_CFG_21      0x04
#define MPU6050_DLPF_CFG_10      0x05
#define MPU6050_DLPF_CFG_5       0x06

#define MPU6050_GYRO_CONFIG      0x1B
#define MPU6050_ACCEL_CONFIG     0x1C
#define MPU6050_FF_THR           0x1D  // Free-fall
#define MPU6050_FF_DUR           0x1E  // Free-fall
#define MPU6050_MOT_THR          0x1F  // Motion detection threshold bits [7:0]
#define MPU6050_MOT_DUR          0x20  // Duration counter threshold for motion interrupt generation, 1 kHz rate, LSB = 1 ms
#define MPU6050_ZMOT_THR         0x21  // Zero-motion detection threshold bits [7:0]
#define MPU6050_ZRMOT_DUR        0x22  // Duration counter threshold for zero motion interrupt generation, 16 Hz rate, LSB = 64 ms
#define MPU6050_FIFO_EN          0x23
#define MPU6050_I2C_MST_CTRL     0x24   
#define MPU6050_I2C_SLV0_ADDR    0x25
#define MPU6050_I2C_SLV0_REG     0x26
#define MPU6050_I2C_SLV0_CTRL    0x27
#define MPU6050_I2C_SLV1_ADDR    0x28
#define MPU6050_I2C_SLV1_REG     0x29
#define MPU6050_I2C_SLV1_CTRL    0x2A
#define MPU6050_I2C_SLV2_ADDR    0x2B
#define MPU6050_I2C_SLV2_REG     0x2C
#define MPU6050_I2C_SLV2_CTRL    0x2D
#define MPU6050_I2C_SLV3_ADDR    0x2E
#define MPU6050_I2C_SLV3_REG     0x2F
#define MPU6050_I2C_SLV3_CTRL    0x30
#define MPU6050_I2C_SLV4_ADDR    0x31
#define MPU6050_I2C_SLV4_REG     0x32
#define MPU6050_I2C_SLV4_DO      0x33
#define MPU6050_I2C_SLV4_CTRL    0x34
#define MPU6050_I2C_SLV4_DI      0x35
#define MPU6050_I2C_MST_STATUS   0x36
#define MPU6050_INT_PIN_CFG      0x37
#define MPU6050_INT_ENABLE       0x38
#define MPU6050_DMP_INT_STATUS   0x39  // Check DMP interrupt
#define MPU6050_INT_STATUS       0x3A
#define MPU6050_ACCEL_XOUT_H     0x3B
#define MPU6050_ACCEL_XOUT_L     0x3C
#define MPU6050_ACCEL_YOUT_H     0x3D
#define MPU6050_ACCEL_YOUT_L     0x3E
#define MPU6050_ACCEL_ZOUT_H     0x3F
#define MPU6050_ACCEL_ZOUT_L     0x40
#define MPU6050_TEMP_OUT_H       0x41
#define MPU6050_TEMP_OUT_L       0x42
#define MPU6050_GYRO_XOUT_H      0x43
#define MPU6050_GYRO_XOUT_L      0x44
#define MPU6050_GYRO_YOUT_H      0x45
#define MPU6050_GYRO_YOUT_L      0x46
#define MPU6050_GYRO_ZOUT_H      0x47
#define MPU6050_GYRO_ZOUT_L      0x48
#define MPU6050_EXT_SENS_DATA_00 0x49
#define MPU6050_EXT_SENS_DATA_01 0x4A
#define MPU6050_EXT_SENS_DATA_02 0x4B
#define MPU6050_EXT_SENS_DATA_03 0x4C
#define MPU6050_EXT_SENS_DATA_04 0x4D
#define MPU6050_EXT_SENS_DATA_05 0x4E
#define MPU6050_EXT_SENS_DATA_06 0x4F
#define MPU6050_EXT_SENS_DATA_07 0x50
#define MPU6050_EXT_SENS_DATA_08 0x51
#define MPU6050_EXT_SENS_DATA_09 0x52
#define MPU6050_EXT_SENS_DATA_10 0x53
#define MPU6050_EXT_SENS_DATA_11 0x54
#define MPU6050_EXT_SENS_DATA_12 0x55
#define MPU6050_EXT_SENS_DATA_13 0x56
#define MPU6050_EXT_SENS_DATA_14 0x57
#define MPU6050_EXT_SENS_DATA_15 0x58
#define MPU6050_EXT_SENS_DATA_16 0x59
#define MPU6050_EXT_SENS_DATA_17 0x5A
#define MPU6050_EXT_SENS_DATA_18 0x5B
#define MPU6050_EXT_SENS_DATA_19 0x5C
#define MPU6050_EXT_SENS_DATA_20 0x5D
#define MPU6050_EXT_SENS_DATA_21 0x5E
#define MPU6050_EXT_SENS_DATA_22 0x5F
#define MPU6050_EXT_SENS_DATA_23 0x60
#define MPU6050_MOT_DETECT_STATUS 0x61
#define MPU6050_I2C_SLV0_DO      0x63
#define MPU6050_I2C_SLV1_DO      0x64
#define MPU6050_I2C_SLV2_DO      0x65
#define MPU6050_I2C_SLV3_DO      0x66
#define MPU6050_I2C_MST_DELAY_CTRL 0x67
#define MPU6050_SIGNAL_PATH_RESET  0x68
#define MPU6050_MOT_DETECT_CTRL   0x69
#define MPU6050_USER_CTRL        0x6A  // Bit 7 enable DMP, bit 3 reset DMP
#define MPU6050_PWR_MGMT_1       0x6B // Device defaults to the SLEEP mode
#define MPU6050_PWR_MGMT_2       0x6C
#define MPU6050_DMP_BANK         0x6D  // Activates a specific bank in the DMP
#define MPU6050_DMP_RW_PNT       0x6E  // Set read/write pointer to a specific start address in specified DMP bank
#define MPU6050_DMP_REG          0x6F  // Register in DMP from which to read or to which to write
#define MPU6050_DMP_REG_1        0x70
#define MPU6050_DMP_REG_2        0x71
#define MPU6050_FIFO_COUNTH      0x72
#define MPU6050_FIFO_COUNTL      0x73
#define MPU6050_FIFO_R_W         0x74
#define MPU6050_WHO_AM_I 0x75 // Should return 0x68                 
 
 /**
  * Definitions
  */
#define MPU6050_SLP_BIT             6
#define MPU6050_BYPASS_BIT         1

#define MPU6050_BW_256              0
#define MPU6050_BW_188              1
#define MPU6050_BW_98               2
#define MPU6050_BW_42               3
#define MPU6050_BW_20               4
#define MPU6050_BW_10               5
#define MPU6050_BW_5                6

#define MPU6050_ACCELERO_RANGE_2G   0
#define MPU6050_ACCELERO_RANGE_4G   1
#define MPU6050_ACCELERO_RANGE_8G   2
#define MPU6050_ACCELERO_RANGE_16G  3

#define MPU6050_GYRO_RANGE_250      0
#define MPU6050_GYRO_RANGE_500      1
#define MPU6050_GYRO_RANGE_1000     2
#define MPU6050_GYRO_RANGE_2000     3


/** MPU6050 IMU library.
  *
  * Example:
  * @code
  * Later, maybe
  * @endcode
  */
class MPU6050 {
    public:
     /**
     * Constructor.
     *
     * Sleep mode of MPU6050 is immediatly disabled
     *
     * @param sda - mbed pin to use for the SDA I2C line.
     * @param scl - mbed pin to use for the SCL I2C line.
     */
     MPU6050(PinName sda, PinName scl);
     

     /**
     * Tests the I2C connection by reading the WHO_AM_I register. 
     *
     * @return True for a working connection, false for an error
     */     
     char testConnection( void );
     
     /**
     * Sets the bandwidth of the digital low-pass filter 
     *
     * Macros: MPU6050_BW_256 - MPU6050_BW_188 - MPU6050_BW_98 - MPU6050_BW_42 - MPU6050_BW_20 - MPU6050_BW_10 - MPU6050_BW_5
     * Last number is the gyro's BW in Hz (accelero BW is virtually identical)
     *
     * @param BW - The three bits that set the bandwidth (use the predefined macros)
     */     
     void setBW( char BW );
     
     /**
     * Sets the auxiliary I2C bus in bypass mode to read the sensors behind the MPU6050 (useful for eval board, otherwise just connect them to primary I2C bus) 
     *
     * @param state - Enables/disables the I2C bypass mode
     */     
     void setI2CBypass ( bool state );
     
     /**
     * Sets the Accelero full-scale range
     *
     * Macros: MPU6050_ACCELERO_RANGE_2G - MPU6050_ACCELERO_RANGE_4G - MPU6050_ACCELERO_RANGE_8G - MPU6050_ACCELERO_RANGE_16G
     *
     * @param range - The two bits that set the full-scale range (use the predefined macros)
     */
     void setAcceleroRange(char range);
     
     
     void setAcceleroFilter(char bandwidth);
     
     
     /**
     * Reads the accelero x-axis.
     *
     * @return 16-bit signed integer x-axis accelero data
     */   
     int getAcceleroRawX( void );
     
     /**
     * Reads the accelero y-axis.
     *
     * @return 16-bit signed integer y-axis accelero data
     */   
     int getAcceleroRawY( void );
     
     /**
     * Reads the accelero z-axis.
     *
     * @return 16-bit signed integer z-axis accelero data
     */   
     int getAcceleroRawZ( void );
     
     /**
     * Reads all accelero data.
     *
     * @param data - pointer to signed integer array with length three: data[0] = X, data[1] = Y, data[2] = Z
     */   
     void getAcceleroRaw( int *data );
     
     /**
     * Reads all accelero data, gives the acceleration in m/s2
     *
     * Function uses the last setup value of the full scale range, if you manually set in another range, this won't work.
     *
     * @param data - pointer to float array with length three: data[0] = X, data[1] = Y, data[2] = Z
     */   
     void getAccelero( float *data );
     
     /**
     * Sets the Gyro full-scale range
     *
     * Macros: MPU6050_GYRO_RANGE_250 - MPU6050_GYRO_RANGE_500 - MPU6050_GYRO_RANGE_1000 - MPU6050_GYRO_RANGE_2000
     *
     * @param range - The two bits that set the full-scale range (use the predefined macros)
     */
     void setGyroRange(char range);

     /**
     * Reads the gyro x-axis.
     *
     * @return 16-bit signed integer x-axis gyro data
     */   
     int getGyroRawX( void );
     
     /**
     * Reads the gyro y-axis.
     *
     * @return 16-bit signed integer y-axis gyro data
     */   
     int getGyroRawY( void );
     
     /**
     * Reads the gyro z-axis.
     *
     * @return 16-bit signed integer z-axis gyro data
     */   
     int getGyroRawZ( void );
     
     /**
     * Reads all gyro data.
     *
     * @param data - pointer to signed integer array with length three: data[0] = X, data[1] = Y, data[2] = Z
     */   
     void getGyroRaw( int *data );  
     
     /**
     * Reads all gyro data, gives the gyro in rad/s
     *
     * Function uses the last setup value of the full scale range, if you manually set in another range, this won't work.
     *
     * @param data - pointer to float array with length three: data[0] = X, data[1] = Y, data[2] = Z
     */   
     void getGyro( float *data);     
     
     /**
     * Reads temperature data.
     *
     * @return 16 bit signed integer with the raw temperature register value
     */  
     int getTempRaw( void );
     
     /**
     * Returns current temperature
     *
     * @returns float with the current temperature
     */  
     float getTemp( void );

     /**
     * Sets the sleep mode of the MPU6050 
     *
     * @param state - true for sleeping, false for wake up
     */     
     void setSleepMode( bool state );
     
     
     /**
     * Writes data to the device, could be private, but public is handy so you can transmit directly to the MPU. 
     *
     * @param adress - register address to write to
     * @param data - data to write
     */
     void write( char address, char data);
     
     /**
     * Read data from the device, could be private, but public is handy so you can transmit directly to the MPU. 
     *
     * @param adress - register address to write to
     * @return - data from the register specified by RA
     */
     char read( char adress);
     
     /**
     * Read multtiple regigsters from the device, more efficient than using multiple normal reads. 
     *
     * @param adress - register address to write to
     * @param length - number of bytes to read
     * @param data - pointer where the data needs to be written to 
     */
     void read( char adress, char *data, int length);
     

     
        
     private:

     I2C connection;
     char currentAcceleroRange;
     char currentGyroRange;
     

};



#endif
