
#ifndef SENSOR_DRIVER_H
#define SENSOR_DRIVER_H

#include <string>
#include <vector>
#include "mbed.h"

/* Serial Communication Pins */
#define SDA1        PB_9
#define SCL1        PB_6
#define SDA2        PF_0
#define SCL2        PF_1
#define SDA3        PF_15
#define SCL3        PF_14

/** SensorDriver Abstract class.
 *  @brief  Contains common methods and members of the various sensors
 *
 *  Example:
 *  @code{.cpp}
 *  #include "mbed.h"
 *  #include "sensor_driver.h"
 *
 *  int main() 
 *  {
 *      
 *  }
 *  @endcode
 */
 
extern Serial pc; /// for testing

typedef vector< pair <string, string> > datalist_t;

class SensorDriver 
{
    public:
        /// Public exposed methods
        virtual void InitSensor() = 0;      /// Initialise Sensor
        virtual int PollSensor() = 0;       /// Poll Sensor 

        std::string sensor_serial;    /// Sensor ID string
        datalist_t sensor_data;       /// Sensor Data Vector
        
        enum SensorStatus
        {
            SENSOR_DCN,
            SENSOR_CN,
            SENSOR_DATAERR,
            SENSOR_DATAOOR,
            SENSOR_DATAOK,
        };

    protected:
        /// For use within the specific sensor class
        std::string ConvertSerialNumber(uint8_t sn_arr[]);    /// Convert uint8_t array of Serial ASCII to string
        std::string ConvertDataToString(float data);        /// Convert float data to string
        
        int ValidateData(float data, float data_min, float data_max);       /// Validate data with defined limits
        void SendSensorEmTrace(int err, std::string trace_string = "");     /// Send Warning Trace to Event Manager

};
#endif  // SENSOR_DRIVER_H
