/**
 * @defgroup conversions Data Structure Conversions
 * @{
 */

#include <sstream>
#include <iomanip>
#include <vector>
#include "conversions.h"

/**
 *  @brief  Converts a C++ string to C-string; Use free() to release mem after copying the content.
 *  @author Lau Lee Hong
 *  @param  str C++ string
 *  @return Pointer to a null-terminated array of character
 */
char* StringToChar(const std::string& str) 
{
    char* buffer = (char*)std::malloc(std::strlen(str.c_str()) + 1);
    if(buffer != NULL)
    {
        std::strcpy(buffer, str.c_str());
    }
    
    return buffer; 
}

/**
 *  @brief  Converts an int-type to string-type hex.
 *  @author Lau Lee Hong
 *  @param  i Integer to be converted to hex string
 *  @return Hexadecimal of string format
 */
std::string IntToHex(uint32_t i)
{
  std::stringstream stream;
  stream << std::hex << i;
  return stream.str();
}

/**
 *  @brief  Converts an int-type to string.
 *  @author Lau Lee Hong
 *  @param  i (int)integer to be converted (std::string)integer
 *  @return Integer of string format
 */
std::string IntToString(int v)
{
    std::ostringstream oss;
    oss << v;
    return oss.str();
}

/**
 *  @brief  Converts an double-type to a C-string with variable number of decimal places.
 *  @author Lau Lee Hong
 *  @param  str             buffer to store result
 *  @param  v               value
 *  @param  decimal_digits  number of decimal places
 *  @return Null-terminated charcter array of double value with user-defined number of decimal places
 */
char* DoubleToChar(char* str, double v, int decimal_digits)
{
    int i = 1;
    int int_part, fract_part;
    int len;
    char *ptr;

    /* Prepare decimal digits multiplicator */
    for (; decimal_digits != 0; i *= 10, decimal_digits--);

    /* Calculate integer & fractional parts */
    int_part = (int)v;
    fract_part = (int)((v - (double)(int)v) * i);

    /* Fill in integer part */
    std::sprintf(str, "%i.", int_part);

    /* Prepare fill in of fractional part */
    len = std::strlen(str);
    ptr = &str[len];

    /* Fill in leading fractional zeros */
    for (i /= 10; i > 1; i /= 10, ptr++)
    {
        if (fract_part >= i)
        {
            break;
        }
        *ptr = '0';
    }

    /* Fill in (rest of) fractional part */
    std::sprintf(ptr, "%i", fract_part);

    return str;
}

/**
 *  @brief  Converts a C++ string to an integer
 *  @author Lee Tze Han
 *  @param  str C++ string
 *  @return Integer from string representation
 */
int StringToInt(const std::string& str)
{
    std::istringstream iss(str);
    int v;
    iss >> v;
    return v;   
}

/**
 *  @brief  Converts a time_t object (from NTPClient) to a C++ string
 *  @author Lee Tze Han
 *  @param  time time_t representing current time
 *  @return String representation of the underlying typedef'd uint32_t
 */
//std::string TimeToString(const std::time_t time)
//{
//    std::ostringstream oss;
//    oss << time;
//    return oss.str();   
//}
//
///**
// *  @brief  Converts a C++ string to time_t object
// *  @author Lee Tze Han
// *  @param  str C++ string
// *  @return time_t from string representation
// */
//std::time_t StringToTime(const std::string& str)
//{
//    std::istringstream iss(str);
//    std::time_t time;
//    iss >> time;
//    return time;    
//}

/** @}*/