#include "MAX30105.h"
#include "millis.h"

Serial dbug(USBTX, USBRX);

MAX30105::MAX30105()
{
    _i2c = new I2C(I2C_SDA, I2C_SCL);
    _i2c->frequency(_I2C_SPEED_FAST);
}
//******************************************************************************
MAX30105::MAX30105(I2C *i2c)
{
    _i2c = i2c;
    _i2c->frequency(_I2C_SPEED_FAST);
}
//******************************************************************************
int MAX30105::writeRegister8(uint8_t reg,  uint8_t value)
{
    char cmdData[2] = { (char)reg, (char)value };

    if (_i2c->write(_I2C_WRITE_ADDR, cmdData, sizeof(cmdData)) != 0) {
        return MAX30105_ERROR;
    }

    return MAX30105_NO_ERROR;
}
//******************************************************************************
int MAX30105::writeReg(uint8_t reg )
{
    char cmdData[1] = { (char)reg };

    if (_i2c->write(_I2C_WRITE_ADDR, cmdData, sizeof(cmdData)) != 0) {
        return MAX30105_ERROR;
    }

    return MAX30105_NO_ERROR;
}
//******************************************************************************
uint8_t MAX30105::readRegister8(uint8_t uch_addr)
{
    char ch_i2c_data;
    ch_i2c_data=uch_addr;
    if(_i2c->write(_I2C_WRITE_ADDR, &ch_i2c_data, 1, true)!=0) {

        return 0;
    }
    if(_i2c->read(_I2C_READ_ADDR, &ch_i2c_data, 1, false)==0) {
        return ch_i2c_data;
    } else
        return false;
}

//******************************************************************************
//Given a register, read it, mask it, and then set the thing
void MAX30105::bitMask(uint8_t reg, uint8_t mask, uint8_t thing)
{
    // Grab current register context
    uint8_t originalContents = readRegister8(reg);

    // Zero-out the portions of the register we're interested in
    originalContents = originalContents & mask;

    // Change contents
    writeRegister8(reg, originalContents | thing);
}

//Set sample average (Table 3, Page 18)
void MAX30105::setFIFOAverage(uint8_t numberOfSamples)
{
    bitMask(MAX30105_FIFOCONFIG, MAX30105_SAMPLEAVG_MASK, numberOfSamples);
}

//Setup the sensor
//The MAX30105 has many settings. By default we select:
// Sample Average = 4
// Mode = MultiLED
// ADC Range = 16384 (62.5pA per LSB)
// Sample rate = 50
//Use the default setup if you are just getting started with the MAX30105 sensor
void MAX30105::setup(uint8_t powerLevel, uint8_t sampleAverage, uint8_t ledMode, int sampleRate, int pulseWidth, int adcRange)
{
    softReset(); //Reset all configuration, threshold, and data registers to POR values

    //FIFO Configuration
    //-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
    //The chip will average multiple samples of same type together if you wish
    if (sampleAverage == 1) setFIFOAverage(MAX30105_SAMPLEAVG_1); //No averaging per FIFO record
    else if (sampleAverage == 2) setFIFOAverage(MAX30105_SAMPLEAVG_2);
    else if (sampleAverage == 4) setFIFOAverage(MAX30105_SAMPLEAVG_4);
    else if (sampleAverage == 8) setFIFOAverage(MAX30105_SAMPLEAVG_8);
    else if (sampleAverage == 16) setFIFOAverage(MAX30105_SAMPLEAVG_16);
    else if (sampleAverage == 32) setFIFOAverage(MAX30105_SAMPLEAVG_32);
    else setFIFOAverage(MAX30105_SAMPLEAVG_4);

    //setFIFOAlmostFull(2); //Set to 30 samples to trigger an 'Almost Full' interrupt
    enableFIFORollover(); //Allow FIFO to wrap/roll over
    //-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

    //Mode Configuration
    //-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
    if (ledMode == 3) setLEDMode(MAX30105_MODE_MULTILED); //Watch all three LED channels
    else if (ledMode == 2) setLEDMode(MAX30105_MODE_REDIRONLY); //Red and IR
    else setLEDMode(MAX30105_MODE_REDONLY); //Red only
    activeLEDs = ledMode; //Used to control how many uint8_ts to read from FIFO buffer
    //-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

    //Particle Sensing Configuration
    //-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
    if(adcRange < 4096) setADCRange(MAX30105_ADCRANGE_2048); //7.81pA per LSB
    else if(adcRange < 8192) setADCRange(MAX30105_ADCRANGE_4096); //15.63pA per LSB
    else if(adcRange < 16384) setADCRange(MAX30105_ADCRANGE_8192); //31.25pA per LSB
    else if(adcRange == 16384) setADCRange(MAX30105_ADCRANGE_16384); //62.5pA per LSB
    else setADCRange(MAX30105_ADCRANGE_2048);

    if (sampleRate < 100) setSampleRate(MAX30105_SAMPLERATE_50); //Take 50 samples per second
    else if (sampleRate < 200) setSampleRate(MAX30105_SAMPLERATE_100);
    else if (sampleRate < 400) setSampleRate(MAX30105_SAMPLERATE_200);
    else if (sampleRate < 800) setSampleRate(MAX30105_SAMPLERATE_400);
    else if (sampleRate < 1000) setSampleRate(MAX30105_SAMPLERATE_800);
    else if (sampleRate < 1600) setSampleRate(MAX30105_SAMPLERATE_1000);
    else if (sampleRate < 3200) setSampleRate(MAX30105_SAMPLERATE_1600);
    else if (sampleRate == 3200) setSampleRate(MAX30105_SAMPLERATE_3200);
    else setSampleRate(MAX30105_SAMPLERATE_50);

    //The longer the pulse width the longer range of detection you'll have
    //At 69us and 0.4mA it's about 2 inches
    //At 411us and 0.4mA it's about 6 inches
    if (pulseWidth < 118) setPulseWidth(MAX30105_PULSEWIDTH_69); //Page 26, Gets us 15 bit resolution
    else if (pulseWidth < 215) setPulseWidth(MAX30105_PULSEWIDTH_118); //16 bit resolution
    else if (pulseWidth < 411) setPulseWidth(MAX30105_PULSEWIDTH_215); //17 bit resolution
    else if (pulseWidth == 411) setPulseWidth(MAX30105_PULSEWIDTH_411); //18 bit resolution
    else setPulseWidth(MAX30105_PULSEWIDTH_69);
    //-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

    //LED Pulse Amplitude Configuration
    //-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
    //Default is 0x1F which gets us 6.4mA
    //powerLevel = 0x02, 0.4mA - Presence detection of ~4 inch
    //powerLevel = 0x1F, 6.4mA - Presence detection of ~8 inch
    //powerLevel = 0x7F, 25.4mA - Presence detection of ~8 inch
    //powerLevel = 0xFF, 50.0mA - Presence detection of ~12 inch
    
    powerLevel = 0x1F;
    setPulseAmplitudeRed(powerLevel);
    setPulseAmplitudeIR(powerLevel);
    setPulseAmplitudeGreen(powerLevel);
    setPulseAmplitudeProximity(powerLevel);
    //-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

    //Multi-LED Mode Configuration, Enable the reading of the three LEDs
    //-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
    enableSlot(1, SLOT_RED_LED);
    if (ledMode > 1) enableSlot(2, SLOT_IR_LED);
    if (ledMode > 2) enableSlot(3, SLOT_GREEN_LED);
    //enableSlot(1, SLOT_RED_PILOT);
    //enableSlot(2, SLOT_IR_PILOT);
    //enableSlot(3, SLOT_GREEN_PILOT);
    //-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

    clearFIFO(); //Reset the FIFO before we begin checking the sensor
}

//Enable roll over if FIFO over flows
void MAX30105::enableFIFORollover(void)
{
    bitMask(MAX30105_FIFOCONFIG, MAX30105_ROLLOVER_MASK, MAX30105_ROLLOVER_ENABLE);
}
void MAX30105::setLEDMode(uint8_t mode)
{
    // Set which LEDs are used for sampling -- Red only, RED+IR only, or custom.
    // See datasheet, page 19
    bitMask(MAX30105_MODECONFIG, MAX30105_MODE_MASK, mode);
}
void MAX30105::setADCRange(uint8_t adcRange)
{
    // adcRange: one of MAX30105_ADCRANGE_2048, _4096, _8192, _16384
    bitMask(MAX30105_PARTICLECONFIG, MAX30105_ADCRANGE_MASK, adcRange);
}

void MAX30105::setSampleRate(uint8_t sampleRate)
{
    // sampleRate: one of MAX30105_SAMPLERATE_50, _100, _200, _400, _800, _1000, _1600, _3200
    bitMask(MAX30105_PARTICLECONFIG, MAX30105_SAMPLERATE_MASK, sampleRate);
}

void MAX30105::setPulseWidth(uint8_t pulseWidth)
{
    // pulseWidth: one of MAX30105_PULSEWIDTH_69, _188, _215, _411
    bitMask(MAX30105_PARTICLECONFIG, MAX30105_PULSEWIDTH_MASK, pulseWidth);
}

// NOTE: Amplitude values: 0x00 = 0mA, 0x7F = 25.4mA, 0xFF = 50mA (typical)
// See datasheet, page 21
void MAX30105::setPulseAmplitudeRed(uint8_t amplitude)
{
    writeRegister8(MAX30105_LED1_PULSEAMP, amplitude);
}

void MAX30105::setPulseAmplitudeIR(uint8_t amplitude)
{
    writeRegister8(MAX30105_LED2_PULSEAMP, amplitude);
}

void MAX30105::setPulseAmplitudeGreen(uint8_t amplitude)
{
    writeRegister8(MAX30105_LED3_PULSEAMP, amplitude);
}

void MAX30105::setPulseAmplitudeProximity(uint8_t amplitude)
{
    writeRegister8(MAX30105_LED_PROX_AMP, amplitude);
}

void MAX30105::setProximityThreshold(uint8_t threshMSB)
{
    // Set the IR ADC count that will trigger the beginning of particle-sensing mode.
    // The threshMSB signifies only the 8 most significant-bits of the ADC count.
    // See datasheet, page 24.
    writeRegister8(MAX30105_PROXINTTHRESH, threshMSB);
}
void MAX30105::softReset(void)
{
    bitMask(MAX30105_MODECONFIG, MAX30105_RESET_MASK, MAX30105_RESET);

    // Poll for bit to clear, reset is then complete
    // Timeout after 100ms
    unsigned long startTime = millis();
    while (millis() - startTime < 100) {
        uint8_t response = readRegister8( MAX30105_MODECONFIG);
        if ((response & MAX30105_RESET) == 0) break; //We're done!
        wait_ms(1); //Let's not over burden the I2C bus
    }
}

void MAX30105::shutDown(void)
{
    // Put IC into low power mode (datasheet pg. 19)
    // During shutdown the IC will continue to respond to I2C commands but will
    // not update with or take new readings (such as temperature)
    bitMask(MAX30105_MODECONFIG, MAX30105_SHUTDOWN_MASK, MAX30105_SHUTDOWN);
}

void MAX30105::wakeUp(void)
{
    // Pull IC out of low power mode (datasheet pg. 19)
    bitMask(MAX30105_MODECONFIG, MAX30105_SHUTDOWN_MASK, MAX30105_WAKEUP);
}

//******************************************************************************

//Given a slot number assign a thing to it
//Devices are SLOT_RED_LED or SLOT_RED_PILOT (proximity)
//Assigning a SLOT_RED_LED will pulse LED
//Assigning a SLOT_RED_PILOT will ??
void MAX30105::enableSlot(uint8_t slotNumber, uint8_t device)
{

    //uint8_t originalContents;

    switch (slotNumber) {
        case (1):
            bitMask(MAX30105_MULTILEDCONFIG1, MAX30105_SLOT1_MASK, device);
            break;
        case (2):
            bitMask(MAX30105_MULTILEDCONFIG1, MAX30105_SLOT2_MASK, device << 4);
            break;
        case (3):
            bitMask(MAX30105_MULTILEDCONFIG2, MAX30105_SLOT3_MASK, device);
            break;
        case (4):
            bitMask(MAX30105_MULTILEDCONFIG2, MAX30105_SLOT4_MASK, device << 4);
            break;
        default:
            //Shouldn't be here!
            break;
    }
}

//Resets all points to start in a known state
//Page 15 recommends clearing FIFO before beginning a read
void MAX30105::clearFIFO(void)
{
    writeRegister8(MAX30105_FIFOWRITEPTR, 0);
    writeRegister8(MAX30105_FIFOOVERFLOW, 0);
    writeRegister8(MAX30105_FIFOREADPTR, 0);
}
//Report the most recent IR value
uint32_t MAX30105::getIR(void)
{
    //Check the sensor for new data for 250ms
    if(safeCheck(250))
        return (sense.IR[sense.head]);
    else
        return(0); //Sensor failed to find new data
}
//Check for new data but give up after a certain amount of time
//Returns true if new data was found
//Returns false if new data was not found
bool MAX30105::safeCheck(uint8_t maxTimeToCheck)
{
    uint32_t markTime = millis();

    while(1) {
        if(millis() - markTime > maxTimeToCheck) return(false);

        if(check() == true) //We found new data!
            return(true);

        wait_ms(1);
    }
}

uint16_t MAX30105::check(void)
{
  uint8_t readPointer = getReadPointer();
  uint8_t writePointer = getWritePointer();
  int numberOfSamples = 0;
  //Do we have new data?
  if (readPointer != writePointer)
  {
    //Calculate the number of readings we need to get from sensor
    numberOfSamples = writePointer - readPointer;
    if (numberOfSamples < 0) numberOfSamples += 32; //Wrap condition

    numberOfSamples = 1;
    //We now have the number of readings, now calc bytes to read
    //For this example we are just doing Red and IR (3 bytes each)
    int bytesLeftToRead = numberOfSamples * activeLEDs * 3;

    //Get ready to read a burst of data from the FIFO register
    
    writeReg(MAX30105_FIFODATA);
    

    //We may need to read as many as 288 bytes so we read in blocks no larger than I2C_BUFFER_LENGTH
    //I2C_BUFFER_LENGTH changes based on the platform. 64 bytes for SAMD21, 32 bytes for Uno.
    //Wire.requestFrom() is limited to BUFFER_LENGTH which is 32 on the Uno
    while (bytesLeftToRead > 0)
    {
      int toGet = bytesLeftToRead;
      if (toGet > I2C_BUFFER_LENGTH)
      {
        //If toGet is 32 this is bad because we read 6 bytes (Red+IR * 3 = 6) at a time
        //32 % 6 = 2 left over. We don't want to request 32 bytes, we want to request 30.
        //32 % 9 (Red+IR+GREEN) = 5 left over. We want to request 27.

        toGet = I2C_BUFFER_LENGTH - (I2C_BUFFER_LENGTH % (activeLEDs * 3)); //Trim toGet to be a multiple of the samples we need to read
      }

      bytesLeftToRead -= toGet;

      uint8_t buffer[32];
      //Request toGet number of bytes from sensor
      _i2c->read(_I2C_READ_ADDR,(char *)buffer, toGet);

      
      while (toGet > 0)
      {
        sense.head++; //Advance the head of the storage struct
        sense.head %= STORAGE_SIZE; //Wrap condition

        uint8_t temp[sizeof(uint32_t)]; //Array of 4 bytes that we will convert into long
        uint32_t tempLong;
        uint8_t i = 0;

        //Burst read three bytes - RED
        temp[3] = 0;
        temp[2] = buffer[i++];
        temp[1] = buffer[i++];
        temp[0] = buffer[i++];

        //Convert array to long
        memcpy(&tempLong, temp, sizeof(tempLong));
        
        tempLong &= 0x3FFFF; //Zero out all but 18 bits

        sense.red[sense.head] = tempLong; //Store this reading into the sense array

        if (activeLEDs > 1)
        {
          //Burst read three more bytes - IR
          temp[3] = 0;
          temp[2] = buffer[i++];
          temp[1] = buffer[i++];
          temp[0] = buffer[i++];

          //Convert array to long
          memcpy(&tempLong, temp, sizeof(tempLong));
         
          tempLong &= 0x3FFFF; //Zero out all but 18 bits
        
          sense.IR[sense.head] = tempLong;
        }

        if (activeLEDs > 2)
        {
          //Burst read three more bytes - Green
          temp[3] = 0;
          temp[2] = buffer[i++];
          temp[1] = buffer[i++];
          temp[0] = buffer[i++];

          //Convert array to long
          memcpy(&tempLong, temp, sizeof(tempLong));
          
          tempLong &= 0x3FFFF; //Zero out all but 18 bits

          sense.green[sense.head] = tempLong;
        }
        toGet -= activeLEDs * 3;
      }

    } //End while (bytesLeftToRead > 0)

  } //End readPtr != writePtr

  return (numberOfSamples); //Let the world know how much new data we found
 
}
uint8_t MAX30105::getWritePointer(void){
    return (readRegister8(MAX30105_FIFOWRITEPTR));
}
//Read the FIFO Read Pointer
uint8_t MAX30105::getReadPointer(void) {
  return (readRegister8(MAX30105_FIFOREADPTR));
}

//Report the most recent Green value
uint32_t MAX30105::getGreen(void)
{
  //Check the sensor for new data for 250ms
  if(safeCheck(250))
    return (sense.green[sense.head]);
  else
    return(0); //Sensor failed to find new data
}

//Report the most recent Green value
uint32_t MAX30105::getRed(void)
{
  //Check the sensor for new data for 250ms
  if(safeCheck(250))
    return (sense.red[sense.head]);
  else
    return(0); //Sensor failed to find new data
}
