#include "mbed.h"
#include "rtos.h"
#include "ds1307.h"
#include "EthernetInterface.h"
#include "SDFileSystem.h"
#include <stdio.h>
#include <cstring>
#include <string>
#include <iostream>
#include <fstream>
#include <sstream>
#include <vector>

#define HTTPD_SERVER_PORT   80
#define HTTPD_MAX_REQ_LENGTH   16383
#define HTTPD_MAX_HDR_LENGTH   255
#define HTTPD_MAX_FNAME_LENGTH   127
#define HTTPD_MAX_DNAME_LENGTH   127

Serial uart(USBTX, USBRX);         //setting UART ports for serial console on PC
DS1307 my1307(I2C_SDA, I2C_SCL);   //setting I2C ports for RTC device DS1307
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd"); // setting ports for SD card slot

/*    Setting Relay outputs    */
DigitalOut d8(PTC12);
DigitalOut d9(PTC4);
DigitalOut d10(PTD0);

/*    Ethernet setup    */
EthernetInterface eth;
TCPSocketServer server;
TCPSocketConnection client;

/*    Time data structure    */
struct tm_ds1307 {
    int sec;
    int min;
    int hour;
    int day;
    int date;
    int month;
    int year;
};

/*    Interval data structure    */
struct interval {
    int start_hour;
    int start_min;
    int end_hour;
    int end_min;
};

/*    HTTP server contstants setup    */
char buffer[HTTPD_MAX_REQ_LENGTH+1];
char httpHeader[HTTPD_MAX_HDR_LENGTH+1];
char fileName[HTTPD_MAX_FNAME_LENGTH+1];
char dirName[HTTPD_MAX_DNAME_LENGTH+1];
char *uristr;
char *eou;
char *qrystr;

FILE *fp;
int rdCnt;

void get_file(char* uri)
{
    uart.printf("get_file %s\n", uri);
    char *lstchr = strrchr(uri, NULL) -1;
    if ('/' == *lstchr) {
        uart.printf("Open directory /sd%s\n", uri);
        *lstchr = 0;
        sprintf(fileName, "/sd%s", uri);
        DIR *d = opendir(fileName);
        if (d != NULL) {
            sprintf(httpHeader,"HTTP/1.1 200 OK\r\nContent-Type: text/html\r\nConnection: Close\r\n\r\n");
            client.send(httpHeader,strlen(httpHeader));
            sprintf(httpHeader,"<html><head><title>Directory Listing</title></head><body><h1>%s Directory Listing</h1><ul>", uri);
            client.send(httpHeader,strlen(httpHeader));
            struct dirent *p;
            while((p = readdir(d)) != NULL) {
                sprintf(dirName, "%s/%s", fileName, p->d_name);
                uart.printf("%s\n", dirName);
                DIR *subDir = opendir(dirName);
                if (subDir != NULL) {
                    sprintf(httpHeader,"<li><a href=\"./%s/\">%s/</a></li>", p->d_name, p->d_name);
                } else {
                    sprintf(httpHeader,"<li><a href=\"./%s\">%s</a></li>", p->d_name, p->d_name);
                }
                client.send(httpHeader,strlen(httpHeader));
            }
        }
        closedir(d);
        uart.printf("Directory closed\n");
        sprintf(httpHeader,"</ul></body></html>");
        client.send(httpHeader,strlen(httpHeader));
    } else {
        sprintf(fileName, "/sd%s", uri);
        fp = fopen(fileName, "r");
        if (fp == NULL) {
            uart.printf("File not found\n");
            sprintf(httpHeader,"HTTP/1.1 404 Not Found \r\nContent-Type: text\r\nConnection: Close\r\n\r\n");
            client.send(httpHeader,strlen(httpHeader));
            client.send(uri,strlen(uri));
        } else {
            uart.printf("Sending: header.\n");
            sprintf(httpHeader,"HTTP/1.1 200 OK\r\nContent-Type: text/html\r\nConnection: Close\r\n\r\n");
            client.send(httpHeader,strlen(httpHeader));
            uart.printf(" file");
            while ((rdCnt = fread(buffer, sizeof( char ), HTTPD_MAX_REQ_LENGTH + 1, fp)) == HTTPD_MAX_REQ_LENGTH + 1) {
                client.send(buffer, rdCnt);
                uart.printf(".");
            }
            client.send(buffer, rdCnt);
            fclose(fp);
            uart.printf("Done sending header.\n");
        }
    }
}

enum controllers {
    LIGHT,
    PUMP,
    OZONE,
};

enum actions {
    OFF,
    ON,
    AUTO,
    STATUS,
};

enum states {
    STATE_OFF,
    STATE_ON,
};

enum statuses {
    STATUS_OFF,
    STATUS_ON,
    STATUS_AUTO,
};

void write_mode(enum controllers ctrl, std::string mode)
{
    std::ofstream outfile;
    std::string filename = "/sd/";
    switch (ctrl) {
        case LIGHT:
            filename += "light";
            break;
        case PUMP:
            filename += "pump";
            break;
        case OZONE:
            filename += "ozone";
            break;
        default:
            return;
    }
    filename += "_mode";
    
    outfile.open(filename.c_str(), ios::out);
    outfile << mode;
    outfile.close();
}

void write_intervals(enum controllers ctrl, std::string intervals)
{
    std::ofstream outfile;
    std::string filename = "/sd/";
    switch (ctrl) {
        case PUMP:
            filename += "pump";
            break;
        case OZONE:
            filename += "ozone";
            break;
        default:
            return;
    }
    filename += "_intervals";
    
    outfile.open(filename.c_str(), ios::out);
    outfile << intervals;
    outfile.close();
}

std::string read_mode(enum controllers ctrl)
{
    std::ifstream infile;
    std::string filename = "/sd/";
    std::string output;
    switch (ctrl) {
        case LIGHT:
            filename += "light";
            break;
        case PUMP:
            filename += "pump";
            break;
        case OZONE:
            filename += "ozone";
            break;
    }
    filename += "_mode";
    
    infile.open(filename.c_str(), ios::in);
    infile >> output;
    infile.close();
    
    return output;
}

std::vector<struct interval> read_intervals(enum controllers ctrl)
{
    std::vector<struct interval> output = std::vector<struct interval>();
    std::string filename = "/sd/";
    switch (ctrl) {
        case PUMP:
            filename += "pump";
            break;
        case OZONE:
            filename += "ozone";
            break;
        default:
            return std::vector<struct interval>();
    }
    filename += "_intervals";
    
    std::ifstream infile(filename.c_str());

    if (infile.is_open()) {
        std::string line;
        while (std::getline(infile, line)) {
            std::istringstream iss(line);
            struct interval tmp_i;

            if (!(iss >> tmp_i.start_hour >> tmp_i.start_min >> tmp_i.end_hour >> tmp_i.end_min)) {
                uart.printf("Error: wrong format of intervals file %s\n", filename.c_str());
                infile.close();
                return std::vector<struct interval>();
            }

            output.push_back(tmp_i);
        }
        infile.close();
    } else {
        uart.printf("Error: could not open file %s for reading!\n", filename.c_str());
    }
    return output;
}

enum states do2state(DigitalOut digo)
{
    switch (digo) {
        case 0:
            return STATE_OFF;
        case 1:
            return STATE_ON;
    }
    /* Should not happen. */
    return STATE_OFF;
}

/*    Reading actual state of each output port    */
enum states read_state(enum controllers ctrl)
{
    switch (ctrl) {
        case LIGHT:
            return do2state(d8);
        case PUMP:
            return do2state(d9);
        case OZONE:
            return do2state(d10);
    }
    /* Should not happen. */
    return STATE_OFF;
}

/*    Write required state to output port    */
void write_state(enum controllers ctrl, enum states state)
{
    switch (ctrl) {
        case LIGHT:
            d8 = state;
            break;
        case PUMP:
            d9 = state;
            break;
        case OZONE:
            d10 = state;
            break;
    }
}

/*    Put time structure to JSON-like formating    */
std::string jsonify_time(struct tm_ds1307 time)
{
    std::stringstream ss;
    ss << "{\"sec\": " << time.sec << ", \"min\": " << time.min << ", \"hour\": " << time.hour <<
        ", \"day\": " << time.day << ", \"date\": " << time.date << ", \"month\": " << time.month << ", \"year\": " << time.year << "}";
    return ss.str();
}

/*    Put status response to JSON-like formating    */
std::string jsonify_status_response(enum statuses status, enum states state, std::string time)
{
    std::stringstream ss;
    ss << "{\"status\": " << status << ", \"state\": " << state << ", \"time\": " << time << "}";
    return ss.str();
}

/*    Put interval to JSON-like formating    */
std::string jsonify_interval(int sh, int sm, int eh, int em)
{
    std::stringstream ss;
    ss << "{\"start_hour\": " << sh << ", \"start_minute\": " << sm << ", \"end_hour\": " << eh << ", \"end_minute\": " << em << "}";
    return ss.str();
}

/*    Put all intervals to JSON-like formating    */
std::string jsonify_intervals(std::vector<struct interval> intervals)
{
    std::stringstream ss;
    ss << "[";
    for (std::vector<struct interval>::iterator it = intervals.begin(); it != intervals.end(); it++) {
        if (it != intervals.begin()) {
            ss << ", ";
        }
        ss << jsonify_interval(it->start_hour, it->start_min, it->end_hour, it->end_min);
    }
    ss << "]";
    return ss.str();
}

/*    Convert day number to corresponding day name    */
std::string day_to_str(int day)
{
    switch (day) {
        case 1: return "Monday";
        case 2: return "Tuesday";
        case 3: return "Wednesday";
        case 4: return "Thursday";
        case 5: return "Friday";
        case 6: return "Saturday";
        case 7: return "Sunday";
        default: return "Not day";
    }
}

/*    Switch between ON, OFF and AUTO state, send time back to web client    */
void trigger(enum controllers ctrl, enum actions act)
{
    enum statuses status = STATUS_OFF;
    enum states state;
    std::string response;
    std::string mode;
    struct tm_ds1307 tmv;
    int ret;
    
    if (act == ON || act == OFF) {
        /* Turn on or off and write manual mode. */
        write_mode(ctrl, std::string("manual"));
        state = (act == ON) ? STATE_ON : STATE_OFF;
        write_state(ctrl, state);
    } else if (act == AUTO) {
        /* Write auto mode. */
        write_mode(ctrl, std::string("auto"));
    }
    
    mode = read_mode(ctrl);
    state = read_state(ctrl);
    if (mode.compare(std::string("manual")) == 0) {
        switch (state) {
            case STATE_OFF:
                status = STATUS_OFF;
                break;
            case STATE_ON:
                status = STATUS_ON;
                break;
        }
    } else if (mode.compare(std::string("auto")) == 0) {
        status = STATUS_AUTO;
    }

    ret = my1307.gettime(&tmv.sec, &tmv.min, &tmv.hour, &tmv.day, &tmv.date, &tmv.month, &tmv.year);
    if (ret != 0) {
        uart.printf("Error: could not get time!\n");
    } else {
        uart.printf("Current time got successfully: %d/%d/%d (%s) %d:%d:%d\n", tmv.year + 2000, tmv.month, tmv.date, day_to_str(tmv.day).c_str(), tmv.hour, tmv.min, tmv.sec);
    }
    
    response = jsonify_status_response(status, state, jsonify_time(tmv));
    client.send((char *)response.c_str(), strlen(response.c_str()));  
}

/*    Set clock on DS1307 RTC device, print set and current time to serial console    */
void set_clock_cmd(std::string payload)
{
    int ret;
    struct tm_ds1307 tmv;
    std::stringstream ss(payload);
    
    ss >> tmv.year;
    ss >> tmv.month;
    ss >> tmv.date;
    ss >> tmv.day;
    ss >> tmv.hour;
    ss >> tmv.min;
    ss >> tmv.sec;    
    
    ret = my1307.settime(tmv.sec, tmv.min, tmv.hour, tmv.day, tmv.date, tmv.month, tmv.year);
    if (ret != 0) {
        uart.printf("Error: could not set time!\n");
    } else {
        uart.printf("Time set successfully: %d/%d/%d (%s) %d:%d:%d\n", tmv.year + 2000, tmv.month, tmv.date, day_to_str(tmv.day).c_str(), tmv.hour, tmv.min, tmv.sec);
    }
    
    ret = my1307.gettime(&tmv.sec, &tmv.min, &tmv.hour, &tmv.day, &tmv.date, &tmv.month, &tmv.year);
    if (ret != 0) {
        uart.printf("Error: could not get time!\n");
    } else {
        uart.printf("Current time got successfully: %d/%d/%d (%s) %d:%d:%d\n", tmv.year + 2000, tmv.month, tmv.date, day_to_str(tmv.day).c_str(), tmv.hour, tmv.min, tmv.sec);
    }
    
    std::stringstream ss_response;
    ss_response << "{\"time\": " << jsonify_time(tmv) << "}";
    std::string response = ss_response.str();
    client.send((char *)response.c_str(), strlen(response.c_str()));
}

/*    Handling of intervals controlling pump and ozone generator switching ON/OFF    */
void handle_intervals(enum controllers ctrl, std::string payload, bool set)
{
    int ret;
    struct tm_ds1307 tmv;
    
    if (set) {
        write_intervals(ctrl, payload);
    }
    
    ret = my1307.gettime(&tmv.sec, &tmv.min, &tmv.hour, &tmv.day, &tmv.date, &tmv.month, &tmv.year);
    if (ret != 0) {
        uart.printf("Error: could not get time!\n");
    } else {
        uart.printf("Current time got successfully: %d/%d/%d (%s) %d:%d:%d\n", tmv.year + 2000, tmv.month, tmv.date, day_to_str(tmv.day).c_str(), tmv.hour, tmv.min, tmv.sec);
    }
    
    std::vector<struct interval> intervals = read_intervals(ctrl);
    
    std::stringstream ss_response;
    ss_response << "{\"intervals\": " << jsonify_intervals(intervals) << ", \"time\": " << jsonify_time(tmv) << "}";
    std::string response = ss_response.str();
    client.send((char *)response.c_str(), strlen(response.c_str()));
}

/*    Handling (running) HTTP POST commands from web client    */
void run_post_cmd(char *cmd, std::string payload)
{
    if (strcmp("/light_on", cmd) == 0) {
        trigger(LIGHT, ON);
    } else if (strcmp("/light_off", cmd) == 0) {
        trigger(LIGHT, OFF);
    } else if (strcmp("/light_status", cmd) == 0) {
        trigger(LIGHT, STATUS);
    } else if (strcmp("/pump_on", cmd) == 0) {
        trigger(PUMP, ON);
    } else if (strcmp("/pump_off", cmd) == 0) {
        trigger(PUMP, OFF);
    } else if (strcmp("/pump_auto", cmd) == 0) {
        trigger(PUMP, AUTO);
    } else if (strcmp("/pump_status", cmd) == 0) {
        trigger(PUMP, STATUS);
    } else if (strcmp("/ozone_on", cmd) == 0) {
        trigger(OZONE, ON);
    } else if (strcmp("/ozone_off", cmd) == 0) {
        trigger(OZONE, OFF);
    } else if (strcmp("/ozone_auto", cmd) == 0) {
        trigger(OZONE, AUTO);
    } else if (strcmp("/ozone_status", cmd) == 0) {
        trigger(OZONE, STATUS);
    } else if (strcmp("/set_clock", cmd) == 0) {
        set_clock_cmd(payload);
    } else if (strcmp("/pump_get_intervals", cmd) == 0) {
        handle_intervals(PUMP, payload, false);
    } else if (strcmp("/pump_set_intervals", cmd) == 0) {
        handle_intervals(PUMP, payload, true);
    } else if (strcmp("/ozone_get_intervals", cmd) == 0) {
        handle_intervals(OZONE, payload, false);
    } else if (strcmp("/ozone_set_intervals", cmd) == 0) {
        handle_intervals(OZONE, payload, true);
    }
}

/*    Sending header to web client    */
void handle_post_cmd(char *cmd, std::string payload)
{
    uart.printf("Sending: header\n");
    sprintf(httpHeader,"HTTP/1.1 200 OK\r\nContent-Type: application/json\r\nConnection: Close\r\n\r\n");
    client.send(httpHeader,strlen(httpHeader));
    
    run_post_cmd(cmd, payload);
    uart.printf("done\n");
}

/*    Time interval between running comparsions of actual time and set intervals (in miliseconds)    */
const uint32_t AUTO_CONTROL_INTERVAL = 20000;

/*    Condition, if time value is in interval    */
bool is_in_interval(int sh, int sm, int eh, int em, int hour, int min)
{
    if ((sh < hour && hour < eh) ||
            (sh <= hour && sm <= min && (hour < eh || (hour <= eh && min <= em))) ||
            (sh < hour && hour <= eh && min <= em)) {
        return true;
    } else {
        return false;
    }
}

/*    Manual and auto check and trigger, using condition "is_in_interval"    */
void check_and_trigger(enum controllers ctrl, int hour, int min)
{
    if (read_mode(ctrl).compare("auto") != 0) {
        /* Not auto mode, return now */
        return;
    }
    
    bool power_on = false;
    uart.printf("Checking auto for %d\n", ctrl);
    
    std::vector<struct interval> intervals = read_intervals(ctrl);
    for (std::vector<struct interval>::iterator it = intervals.begin(); it != intervals.end(); it++) {
        int sh = it->start_hour;
        int sm = it->start_min;
        int eh = it->end_hour;
        int em = it->end_min;
        
        if ((sh < eh) || (sh == eh && sm <= em)) {
            /*
             * Timeline:
             * -----------------------------------
             *      ^                     ^
             *     start                 end
             */
            if (is_in_interval(sh, sm, eh, em, hour, min)) {
                power_on = true;
                break;
            }
        } else {
            /*
             * Timeline:
             * -----------------------------------
             *      ^                     ^
             *     end                   start
             */
            if (!is_in_interval(eh, em, sh, sm, hour, min)) {
                power_on = true;
                break;
            }
        }
    }
    /* Switching power for pump and ozone, depending on state detected from interval or from manual state */ 
    if (power_on) {
        write_state(ctrl, STATE_ON);
    } else {
        write_state(ctrl, STATE_OFF);
    }
}

/*    Thread running automatic control mode    */
void auto_control_thread(void const *arg)
{
    while (true) {
        uart.printf("Checking and executing auto intervals.\n");
        
        struct tm_ds1307 tmv;
        int ret;
        
        ret = my1307.gettime(&tmv.sec, &tmv.min, &tmv.hour, &tmv.day, &tmv.date, &tmv.month, &tmv.year);
        if (ret != 0) {
            uart.printf("Error: could not get time!\n");
        } else {
            uart.printf("Current time got successfully: %d/%d/%d (%s) %d:%d:%d\n", tmv.year + 2000, tmv.month, tmv.date, day_to_str(tmv.day).c_str(), tmv.hour, tmv.min, tmv.sec);
        }    

        check_and_trigger(PUMP, tmv.hour, tmv.min);
        check_and_trigger(OZONE, tmv.hour, tmv.min);
        
        break;
    }
}

int main (void)
{
    int ret;
    
/*    Structure for time data    */    
    struct tm_ds1307 tmv = {
            .sec = 0,    // seconds 00-59
            .min = 0,    // minutes 00-59
            .hour = 18,   // hours 00-23
            .day = 6,    // position of day in the week, Monday=1
            .date = 13,   // date 1-31
            .month = 1,  // month 1-12
            .year = 2018    // year in 20xx format
        };  
    
/*    Serial Interface setup    */
    uart.baud(115200);  //serial interface speed
    uart.printf("Initializing\n");

/*    Check File System    */
    uart.printf("Checking File System\n");
    DIR *d = opendir("/sd/");
    if (d != NULL) {
        uart.printf("SD Card Present\n");
    } else {
        uart.printf("SD Card Root Directory Not Found\n");
    }
    
/*    Start clock and get actual time from RTC device   */
    uart.printf("Starting clock\n");
    ret = my1307.start_clock();
    if (ret != 0) {
        uart.printf("Clock could not be started!\n");
    } else {
        uart.printf("Clock started successfully!\n");
    }
    
    ret = my1307.gettime(&tmv.sec, &tmv.min, &tmv.hour, &tmv.day, &tmv.date, &tmv.month, &tmv.year);
    if (ret != 0) {
        uart.printf("Error: could not get time!\n");
    } else {
        uart.printf("Current time got successfully: %d/%d/%d (%s) %d:%d:%d\n", tmv.year + 2000, tmv.month, tmv.date, day_to_str(tmv.day).c_str(), tmv.hour, tmv.min, tmv.sec);
    }

/*    EthernetInterface eth initialization and connection to network    */
    uart.printf("Initializing Ethernet\n");
    eth.init(); //Use DHCP
    uart.printf("Connecting\n");
    eth.connect();
    uart.printf("IP Address is %s\n", eth.getIPAddress());

/*    TCPSocketServer server config & setup    */
    server.bind(HTTPD_SERVER_PORT);
    server.listen(20);
    server.set_blocking(false, AUTO_CONTROL_INTERVAL);  //set blocking interval to own value (20sec)
    uart.printf("Server Listening\n");
    
/*    While loop for web server and automatic control thread    */ 
    while (true) {
        uart.printf("\nWait for new connection...\r\n");
        server.accept(client);
        client.set_blocking(false, 1500); // Timeout after (1.5)s

        uart.printf("Connection from: %s\r\n", client.get_address());
        while (true) {
            int n = client.receive(buffer, sizeof(buffer));
            if (n <= 0) break;
            uart.printf("Recieved Data: %d\r\n\r\n%.*s\r\n",n,n,buffer);
            if (n >= HTTPD_MAX_REQ_LENGTH + 1) {
                sprintf(httpHeader,"HTTP/1.1 413 Request Entity Too Large \r\nContent-Type: text\r\nConnection: Close\r\n\r\n");
                client.send(httpHeader,strlen(httpHeader));
                client.send(buffer,n);
                break;
            } else {
                buffer[n]=0;
            }
            if (!strncmp(buffer, "GET ", 4)) {
                uristr = buffer + 4;
                eou = strstr(uristr, " ");
                if (eou == NULL) {
                    sprintf(httpHeader,"HTTP/1.1 400 Bad Request \r\nContent-Type: text\r\nConnection: Close\r\n\r\n");
                    client.send(httpHeader,strlen(httpHeader));
                    client.send(buffer,n);
                } else {
                    *eou = 0;
                    get_file(uristr);
                }
            } else if (!strncmp(buffer, "POST ", 5)) {   /* Handling of own AJAX-like POST commands */
                uristr = buffer + 5;
                eou = strstr(uristr, " ");
                char *header_end = strstr(uristr, "\r\n\r\n");
                if (eou == NULL || header_end == NULL) {
                    sprintf(httpHeader,"HTTP/1.1 400 Bad Request \r\nContent-Type: text\r\nConnection: Close\r\n\r\n");
                    client.send(httpHeader,strlen(httpHeader));
                    client.send(buffer,n);
                } else {
                    *eou = 0;
                    char *payload_ptr = header_end + strlen("\r\n\r\n");
                    int payload_size = n - (payload_ptr - buffer);
                    /* +1 for terminating character (null byte - '\0' */
                    char tmp_payload[payload_size + 1];
                    memset(tmp_payload, 0, payload_size + 1);
                    memcpy(tmp_payload, payload_ptr, payload_size);
                    std::string payload_str = std::string(tmp_payload);
                    uart.printf("Payload: %s\n", tmp_payload); // print actual string sent in HTTP header
                    
                    handle_post_cmd(uristr, payload_str);
                }
            }
        }

        client.close();
        

        auto_control_thread(NULL); // calling of automatic control thread (each 20sec) 
    }
}
