#include "mbed.h"
#include "config.h"
#include "debug.h"

// Multitech Cellular includes
#include "Cellular.h"
#include "Endpoint.h"
#include "IPStack.h"
#include "MTSSerialFlowControl.h"

// NanoService includes
#include "sn_nsdl.h"
#include "sn_coap_header.h"
#include "sn_coap_protocol.h"
#include "sn_nsdl_lib.h"
#include "sn_grs.h"
#include <stdint.h>

using namespace mts;

Cellular* cellular;
Endpoint nsp;
static const char* NSP_ADDRESS = "176.58.89.233"; // Motivation demo server
static const int NSP_PORT = 5683;
char endpoint_name[] = {"mbed-cellular-mts1"};
uint8_t ep_type[] = {"mbed_kl64z_multitech"};
uint8_t lifetime_ptr[] = {"86400"};
static const char* FIRMWARE_VER = "13"; // Committed revision number

typedef uint8_t (*sn_grs_dyn_res_callback_t)(sn_coap_hdr_s *, sn_nsdl_addr_s *, sn_proto_info_s *);

// ****************************************************************************
// Cellular initialization

static void cellular_init()
{
    //Setup serial interface to radio
    MTSSerialFlowControl* serial = new MTSSerialFlowControl(PTD3, PTD2, PTA12, PTC8);
    serial->baud(115200);

    //Setup Cellular class
    cellular = Cellular::getInstance();
    cellular->init(serial, PTA4, PTC9); //DCD and DTR pins for KL46Z

    //Run status and configuration commands
    DEBUG("\n\r////Start Status and Configuration Commands////");
    DEBUG("Command Test: %s", getCodeNames(cellular->test()).c_str()); //Make sure you can talk to the radio
    DEBUG("Signal Strength: %d", cellular->getSignalStrength()); //Check the signal strength should be above 8
    
    //Makes sure you are reistered with cell
    DEBUG("Registration State: %s", Cellular::getRegistrationNames(cellular->getRegistration()).c_str()); 
    
    //Shows example of how to send other commands, look at AT command guide for more info
    DEBUG("Send Basic Command (AT): %s", getCodeNames(cellular->sendBasicCommand("AT", 1000)).c_str());
    DEBUG("Send Command (AT+CSQ): %s", cellular->sendCommand("AT+CSQ", 1000).c_str());

    //Start Test
    DEBUG("\n\r////Start Network Connectivity Test////");
    DEBUG("Set APN: %s", getCodeNames(cellular->setApn(CELLULAR_APN)).c_str()); //Use APN from service provider!!!

    //Setup a data connection
    DEBUG("Attempting to Connect, this may take some time...");
    while (!cellular->connect()) {
        WARNING("Failed to connect... Trying again.");
        wait(1);
    }
    DEBUG("Connected to the Network!");

    //Try pinging default server "8.8.8.8" (Google's DNS)
    DEBUG("Ping Valid: %s", cellular->ping() ? "true" : "false");
    wait(3);

}

// ****************************************************************************
// NSP initialization

static void nsp_connect()
{
    nsp.set_address(NSP_ADDRESS, NSP_PORT);
    
    DEBUG("EP Name: %s", endpoint_name);
    DEBUG("NSP Location: coap://%s:%d\n", NSP_ADDRESS, NSP_PORT);
    
    // Bind the port
    cellular->bind(EP_PORT);

    // Open a TCP connection
    while (!cellular->open(NSP_ADDRESS, NSP_PORT, mts::IPStack::TCP)) 
    {
        WARNING("TCP connection failed.");
        wait(3);
    } 
    DEBUG("TCP connection to NSP successful.");  
}

extern "C" void *nsdl_alloc(uint16_t size)
{
    return malloc(size);
}

extern "C" void nsdl_free(void* ptr_to_free)
{
    free(ptr_to_free);
}

static uint8_t tx_cb(sn_nsdl_capab_e protocol, uint8_t *data_ptr, uint16_t data_len, sn_nsdl_addr_s *address_ptr)
{
    int buffer_len = data_len+2;
    DEBUG("TX callback! Sending %d bytes", buffer_len);
    char buffer[buffer_len];
    buffer[0] = data_len >> 8;
    buffer[1] = data_len & 0xFF;
    memcpy(buffer+2, data_ptr, data_len);
    
    if(cellular->write((char*)buffer, (int)buffer_len, 1000) != buffer_len)
        WARNING("Sending failed");

    return 1;
}

static uint8_t rx_cb(sn_coap_hdr_s *coap_packet_ptr, sn_nsdl_addr_s *address_ptr)
{
    DEBUG("RX callback!");
    return 0;
}

void nsdl_create_static_resource(sn_nsdl_resource_info_s *resource_structure, uint16_t pt_len, uint8_t *pt, uint16_t rpp_len, uint8_t *rpp_ptr, uint8_t *rsc, uint16_t rsc_len)
{
    resource_structure->access = SN_GRS_GET_ALLOWED;
    resource_structure->mode = SN_GRS_STATIC;
    resource_structure->pathlen = pt_len;
    resource_structure->path = pt;
    resource_structure->resource_parameters_ptr->resource_type_len = rpp_len;
    resource_structure->resource_parameters_ptr->resource_type_ptr = rpp_ptr;
    resource_structure->resource = rsc;
    resource_structure->resourcelen = rsc_len;
    sn_nsdl_create_resource(resource_structure);
}

void nsdl_create_dynamic_resource(sn_nsdl_resource_info_s *resource_structure, uint16_t pt_len, uint8_t *pt, uint16_t rpp_len, uint8_t *rpp_ptr, uint8_t is_observable, sn_grs_dyn_res_callback_t callback_ptr, int access_right)
{
    resource_structure->access = (sn_grs_resource_acl_e)access_right;
    resource_structure->resource = 0;
    resource_structure->resourcelen = 0;
    resource_structure->sn_grs_dyn_res_callback = callback_ptr;
    resource_structure->mode = SN_GRS_DYNAMIC;
    resource_structure->pathlen = pt_len;
    resource_structure->path = pt;
    resource_structure->resource_parameters_ptr->resource_type_len = rpp_len;
    resource_structure->resource_parameters_ptr->resource_type_ptr = rpp_ptr;
    resource_structure->resource_parameters_ptr->observable = is_observable;
    sn_nsdl_create_resource(resource_structure);
}

sn_nsdl_ep_parameters_s* nsdl_init_register_endpoint(sn_nsdl_ep_parameters_s *endpoint_structure, uint8_t* name, uint8_t* typename_ptr, uint8_t *lifetime_ptr)
{
    if (NULL == endpoint_structure)
    {   
        endpoint_structure = (sn_nsdl_ep_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_ep_parameters_s));
    }
    if (endpoint_structure)
    {
        memset(endpoint_structure, 0, sizeof(sn_nsdl_ep_parameters_s));
        endpoint_structure->endpoint_name_ptr = name;
        endpoint_structure->endpoint_name_len = strlen((char*)name);
        endpoint_structure->type_ptr = typename_ptr;
        endpoint_structure->type_len =  strlen((char*)typename_ptr);
        endpoint_structure->lifetime_ptr = lifetime_ptr;
        endpoint_structure->lifetime_len =  strlen((char*)lifetime_ptr);
    }
    return endpoint_structure;
}

void nsdl_clean_register_endpoint(sn_nsdl_ep_parameters_s **endpoint_structure)
{
    if (*endpoint_structure)
    {
        nsdl_free(*endpoint_structure);
        *endpoint_structure = NULL;
    }
}

void nsdl_init()
{
    uint8_t nsp_addr[4];
    sn_nsdl_mem_s memory_cbs;
    memory_cbs.sn_nsdl_alloc = &nsdl_alloc;
    memory_cbs.sn_nsdl_free = &nsdl_free;
    if(sn_nsdl_init(&tx_cb, &rx_cb, &memory_cbs) == -1) {
        ERROR("libNsdl init failed");
    } else {
        DEBUG("libNsdl init done");
    }
    /* Set nsp address for library */
    set_NSP_address(nsp_addr, 5683, SN_NSDL_ADDRESS_TYPE_IPV4);
}

static int create_resources()
{
    sn_nsdl_resource_info_s *resource_ptr = NULL;
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;
    
    DEBUG("Creating resources");

    /* Create resources */
    resource_ptr = (sn_nsdl_resource_info_s*)nsdl_alloc(sizeof(sn_nsdl_resource_info_s));
    if(!resource_ptr)
        return 0;
    memset(resource_ptr, 0, sizeof(sn_nsdl_resource_info_s));

    resource_ptr->resource_parameters_ptr = (sn_nsdl_resource_parameters_s*)nsdl_alloc(sizeof(sn_nsdl_resource_parameters_s));
    if(!resource_ptr->resource_parameters_ptr)
    {
        nsdl_free(resource_ptr);
        return 0;
    }
    memset(resource_ptr->resource_parameters_ptr, 0, sizeof(sn_nsdl_resource_parameters_s));

    // Static resources
    nsdl_create_static_resource(resource_ptr, sizeof("1/0/1")-1, (uint8_t*) "1/0/1", 0, 0,  (uint8_t*) lifetime_ptr, sizeof(lifetime_ptr)-1);
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/0")-1, (uint8_t*) "3/0/0", 0, 0,  (uint8_t*) "Freescale/Multitech", sizeof("Freescale/Multitech")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/1")-1, (uint8_t*) "3/0/1", 0, 0,  (uint8_t*) "KL64Z with Multitech Cellular", sizeof("KL64Z with Multitech Cellular")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/16")-1, (uint8_t*) "3/0/16", 0, 0,  (uint8_t*) "TCP", sizeof("TCP")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("3/0/3")-1, (uint8_t*) "3/0/3", 0, 0,  (uint8_t*) FIRMWARE_VER, strlen(FIRMWARE_VER));
    nsdl_create_static_resource(resource_ptr, sizeof("4/0/0")-1, (uint8_t*) "4/0/0", 0, 0,  (uint8_t*) "GPRS", sizeof("GPRS")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("4/0/1")-1, (uint8_t*) "4/0/1", 0, 0,  (uint8_t*) "GPRS", sizeof("GPRS")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("6/0/0")-1, (uint8_t*) "6/0/0", 0, 0,  (uint8_t*) "37.959611", sizeof("37.959611")-1);
    nsdl_create_static_resource(resource_ptr, sizeof("6/0/1")-1, (uint8_t*) "6/0/1", 0, 0,  (uint8_t*) "23.721136", sizeof("23.721136")-1);
    // nsdl_create_static_resource(resource_ptr, sizeof("gps/loc")-1, (uint8_t*) "gps/loc", 0, 0,  (uint8_t*) "37.959611,23.721136", sizeof("37.959611,23.721136")-1);
    // Broken: This cause the modem driver to crash when the request is received.

    // Dynamic resources
    // create_light_resource(resource_ptr);
    // create_gps_resource(resource_ptr);

    /* Register with NSP */
    endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
    if(sn_nsdl_register_endpoint(endpoint_ptr) != 0) {
        WARNING("NSP registering failed\r\n");
    } else {
        DEBUG("NSP registering OK\r\n");
    }
    nsdl_clean_register_endpoint(&endpoint_ptr);

    nsdl_free(resource_ptr->resource_parameters_ptr);
    nsdl_free(resource_ptr);
    return 1;
}

void nsp_register()
{
    sn_nsdl_ep_parameters_s *endpoint_ptr = NULL;

        endpoint_ptr = nsdl_init_register_endpoint(endpoint_ptr, (uint8_t*)endpoint_name, ep_type, lifetime_ptr);
        if(sn_nsdl_register_endpoint(endpoint_ptr) != 0) {
            WARNING("NSP re-registration failed\r\n");
        } else {
            DEBUG("NSP re-registration OK\r\n");
        }
        nsdl_clean_register_endpoint(&endpoint_ptr);
}

void socket_event_loop()
{
    sn_nsdl_addr_s received_packet_address;
    uint8_t received_address[4];
    char buffer[2048];
    int n;

    memset(&received_packet_address, 0, sizeof(sn_nsdl_addr_s));
    received_packet_address.addr_ptr = received_address; 

    DEBUG("Starting socket read loop...");
    while(1)
    {
        n = cellular->read(buffer, sizeof(buffer), 1000);
        DEBUG("Received %d bytes", n);
        if (n < 0)
        {
            ERROR("Socket error");
        }
        else
        {   
            uint16_t len = 0;
            if (n > 2) {
                len = 256 * buffer[0] + buffer[1]; 
                DEBUG("CoAP length header = %d bytes", len);
                sn_nsdl_process_coap((uint8_t*)buffer+2, len, &received_packet_address);
            }
        } 
    }
}

int main() 
{
    printf("\r\n*****************************************************************************\r\n");
    DEBUG("NanoService Example for KL46Z + Multitech Cellular");

    // Inititalize the Cellular modem
    cellular_init();

    // Bind the socket and configure NSP settings
    nsp_connect();
 
    // Initalize NanoService library
    nsdl_init();

    // Create resources & register with NSP
    create_resources(); 
     
    // Start socket listening loop
    socket_event_loop();
       
}
