/*------------------------------------------------------------------------------
 MAX116XX 8bit 4-/8-/12-Channel 2-Write Serial ADC Driver for MBED          
 Copyright 2019, Georg Zachl <uni@zachlge.org>                              
                                                                            
 NOTES:                                                                     
 This driver intentionally starts in I2C FAST MODE. The faster mode is not
 implemented, as it is not required for our use case.
 The driver has only been tested for the MAX11605 12 channel device!
 
 License: Do what ever you want, but if you modify this library and
 add more features, make them available to the public.
------------------------------------------------------------------------------*/

#ifndef __MAX116XX_ADC_H__
#define __MAX116XX_ADC_H__

/**
 * Includes
 */
#include "mbed.h"
#include <stdint.h>


/**
 * Defines & Datatypes
 */

// Internally used defines.
#define MAX116XX_I2C_CLOCK_NORM 400000u
#define MAX116XX_I2C_CLOCK_MAX  1700000u


// Defines for the internal setup register and for the setup function.
#define MAX116XX_SETUP_REF_VDD        0x00
#define MAX116XX_SETUP_REF_EXT        0x20
#define MAX116XX_SETUP_REF_INT_AUTO   0x40
#define MAX116XX_SETUP_REF_INT_ON     0x50
#define MAX116XX_SETUP_REF_INT_ON_OUT 0x60

#define MAX116XX_SETUP_CLK_INT       0x00
#define MAX116XX_SETUP_CLK_EXT       0x08

#define MAX116XX_SETUP_BIP          0x04
#define MAX116XX_SETUP_UNI          0x00

#define MAX116XX_SETUP_CONF_RESET   0x00
#define MAX116XX_SETUP_CONF_KEEP    0x02

#define MAX116XX_SETUP_REF_MASK    0x70



//Defines for the internal condif register.

#define MAX116XX_CONF_SCAN_0_TO_CS (0<<5)
#define MAX116XX_CONF_SCAN_CS_EIGHT (1<<5)
#define MAX116XX_CONF_SCAN_HALF (2<<5)
#define MAX116XX_CONF_SCAN_CS (3<<5)
//0 = SCAN from AIN0 to the input selected by CS3-CS0
//1 = Convert the selected input 8 times
//2 = Scans up/to AIN6 (depends on CS3-CS0 is below or above AIN6), on 00 and 01 devices: same with AIN2
//3 = Convert the selected input

#define MAX116XX_CONF_CS(num)       ((num&0x0F)<<1)

#define MAX116XX_CONF_SGL           0x01
#define MAX116XX_CONF_DIF           0x00

//Define
#define MAX116XX_SETUP_REG            0x80
#define MAX116XX_CONF_REG             0x00

//Maximum channel number of all devices, used internally.
#define MAX116XX_MAX_CHANNELS         12 
 
typedef enum { MAX11600, MAX11601, MAX11602, MAX11603, MAX11604, MAX11605, MAX116XX_UNDEF} max116xx_type_t;

/**
* Struct used to describe device features.
*/
typedef struct {
    uint8_t channels;
    uint32_t internal_ref_in_mV;
    uint8_t addr;
} max116xx_features_t;
    
/**
* Table with all device features. index should be used fromthe max116xx_type_t enum.
*/ 
static const max116xx_features_t max116xx_features[]={
    {4u, 4096u, 0x64u}, //00
    {4u, 2048u, 0x64u}, //01
    {8u, 4096u, 0x6Du}, //02
    {8u, 2048u, 0x6Du}, //03
    {12u, 4096u, 0x65u}, //04
    {12u, 2048u, 0x65u}, //05
};

/**
 * MAX116XX Analog to digital converter chip with internal voltage reference.
 */
class max116xx_adc {

public:

    /**
     * Constructor.
     *
     * @param sda mbed pin to use for SDA line of I2C interface.
     * @param scl mbed pin to use for SCL line of I2C interface.
     */
    max116xx_adc(PinName sda, PinName scl);
    
    /**
     * setup
     *
     * @param adc_type which type of ADC is used in this case (see max116xx_type_t).
     * @param VDD_in_mV approximate value of the supply voltage for the ADC. (For usage as reference voltage)
     * @param VREF_in_mV value of the (possible) external reference voltage, ignored when not used.
     * @param setup logically or combined setup consisting of MAX116XX_SETUP_* defines.
     *        see the MAX116XX datasheet for further information.
    */
    bool setup(max116xx_type_t adc_type, uint32_t VDD_in_mV, uint32_t VREF_in_mV, uint8_t setup);
    
    /**
     * read_channel
     *
     * Reads/converts a selected channel. Returns false if the communication
     * failed or when the channel is not available on the selected device.
     *
     * @param channel numeric value of the channel to read.
    */
    bool read_channel(uint8_t channel);

    /**
     * read_all
     *
     * Reads/converts all channels at once.
     * WARNING: This might take some time, as the conversion in the ADC is triggered
     *          when the I2C transfer is started.
    */
    bool read_all();
    
    
    
    /**
     * voltage_get
     *
     * Get the last read voltage as float. This automatically applies the transformation
     * to the actual voltage, based on the given reference voltag and settings.
    */    
    float voltage_get(uint8_t channel);
    

    /**
     * input_latest
     *
     * This array holds the latest read raw values. This might be read for direct
     * usage of the raw values. Overwriting those values results in false outputs
     * from voltage_get until the values are refreshed with read_all or read_channel.
    */
    uint8_t input_latest[MAX116XX_MAX_CHANNELS];

private:



    I2C             i2c_;
    max116xx_type_t type;
    uint32_t vdd;
    uint32_t vref_ext;
    uint32_t vref_int;
    uint32_t vref_act;
    uint8_t channels_num;
    uint8_t addr;

};




#endif /* __MAX116XX_ADC_H__ */