#include "max116xx_adc.h"


max116xx_adc::max116xx_adc(PinName sda, PinName scl) : i2c_(sda, scl), type(MAX116XX_UNDEF), vdd(0) {
        
}

bool max116xx_adc::setup(max116xx_type_t adc_type, uint32_t VDD_in_mV, uint32_t VREF_in_mV, uint8_t setup) {
    //Check if the ADC type is valid and if VDD is not above 6V (which is the absolute maximum rating for the voltage)
    if(adc_type>=MAX116XX_UNDEF || VDD_in_mV>6000 || VREF_in_mV>VDD_in_mV) {
        return false;
    }
    type = adc_type;
    vdd = VDD_in_mV;
    vref_ext = VREF_in_mV;
    vref_int = max116xx_features[type].internal_ref_in_mV;
    channels_num = max116xx_features[type].channels;
    
    //Shift the Addr one bit higher, because the LSB is the read/write bit.
    addr = (max116xx_features[type].addr) << 1;
    
    switch(setup&MAX116XX_SETUP_REF_MASK) {
        case MAX116XX_SETUP_REF_VDD:
            vref_act = vdd;
            break;
        case MAX116XX_SETUP_REF_EXT:
            vref_act = vref_ext;
            break;
            
        case MAX116XX_SETUP_REF_INT_AUTO:
        case MAX116XX_SETUP_REF_INT_ON:
        case MAX116XX_SETUP_REF_INT_ON_OUT:
            vref_act = vref_int;
            break;
        default:
            type = MAX116XX_UNDEF;
            return false;
    }
    
    //Set the REG=1 to write to the setup register
    setup |= MAX116XX_SETUP_REG;
    //Delete the N_RESET bit to force a reset of the configuration register.
    setup &= ~MAX116XX_SETUP_CONF_KEEP;
    
    //i2c_.frequency(MAX116XX_I2C_CLOCK_NORM);
    
    if(i2c_.write(addr, (const char*)&setup, 1)) {
        type = MAX116XX_UNDEF;
        return false;
    }
    
    
    return true;
}

bool max116xx_adc::read_channel(uint8_t channel) {
    //First we need to setup the SCAN mode 11    
    if(channel>=channels_num) {
        return false;
    }
    char conf = MAX116XX_CONF_SCAN_CS|MAX116XX_CONF_CS(channel)|MAX116XX_CONF_SGL;
    
    if(i2c_.write(addr, (const char*)&conf, 1)) {
        return false;
    }
    
    if(i2c_.read(addr, (char*)(&input_latest)+channel, 1)) {
        return false;
    }
    
    
    return true;
}

bool max116xx_adc::read_all() {
    //First we need to setup the SCAN mode 00  
    char conf = MAX116XX_CONF_SCAN_0_TO_CS|MAX116XX_CONF_CS(channels_num-1)|MAX116XX_CONF_SGL;
    
    if(i2c_.write(addr, (const char*)&conf, 1)) {
        return false;
    }
    
    if(i2c_.read(addr, (char*)&input_latest, channels_num)) {
        return false;
    }
    
    return true;
}

float max116xx_adc::voltage_get(uint8_t channel) {
    if(channel>=channels_num) {
        return -1000.0f;
    }
    return (float)input_latest[channel]*vref_act/1000.0f/255.0f;
}