/*YOZAKURAのアームコード
  UDP通信で送るデータ S_Data={linearD[4],pitchD[4],yawD[4],ThermoD1[16],ThermoD2[16],CO2[4]} 計48bytes
  UDP通信で受け取るデータ　R_Data=｛mode[2bit],linear_ref[2bit],pitch_ref[2bit],yaw_ref[2bit]} 計1byte
*/

#include "mbed.h"
#include "rtos.h"

DigitalOut myled1(LED1); DigitalOut myled2(LED2); DigitalOut myled3(LED3); DigitalOut myled4(LED4);
Mutex stdio_mutex;

/*--Ethernet:begin------------------------------------------------------------------------------------------*/
#include "mbed.h"
#include "EthernetInterface.h" 

const char* ECHO_SERVER_ADDRESS = "192.168.1.90"; 
const int ECHO_SERVER_PORT = 60000;

const char* CLIENT_ADDRESS = "192.168.1.100"; 
const int CLIENT_PORT = 70000;

EthernetInterface eth; 
UDPSocket sock; 
Endpoint echo_server; 

void Ethernet_init() {
    eth.init("192.168.1.100", "255.255.255.0", ""); //  Use hard IP
    eth.connect(); 

    sock.bind(CLIENT_PORT);
    sock.set_blocking(false);
    
    echo_server.set_address(ECHO_SERVER_ADDRESS, ECHO_SERVER_PORT);
}
/*--Ethernet:end--------------------------------------------------------------------------------------------*/


/*--Dynamixel:begin-----------------------------------------------------------------------------------------*/
#include "AX12.h"
#include "MX28.h"

DigitalOut low(p19); DigitalOut RelaySwitch(p17);

AX12 linear (p13, p14, 0);      //直動Dynamixel
MX28 pitch (p13, p14, 1);       //ピッチDynamixel
MX28 yaw (p13, p14, 2);         //ヨーDynamixel

int linear_goal, pitch_goal, yaw_goal;

//最小値，最大値，角速度，初期値を指定[unit:degree]
//MX:MultiTurnモードでは-2520°～2520°(SetGoalの書き換え不要)
int linear_min = 100; int linear_MAX = 720; float linear_Speed=0.1; int linear_Init = linear_MAX;
int pitch_min = 172; int pitch_MAX = 334; float pitch_Speed=0.2; int pitch_Init = pitch_MAX;
int yaw_min = 360; int yaw_MAX = 360; float yaw_Speed=0.2; int yaw_Init = 0; //MultiTurnモード

void Dyna_init() {
    low = 0; RelaySwitch = 1;
    linear.SetCWLimit(linear_min); linear.SetCCWLimit(linear_MAX); 
    linear.SetCRSpeed(linear_Speed); linear_goal=linear_Init; linear.SetGoal(linear_Init);
    pitch.SetCWLimit(linear_min); pitch.SetCCWLimit(linear_MAX); 
    pitch.SetCRSpeed(pitch_Speed); pitch_goal=linear_Init; pitch.SetGoal(pitch_Init);
    yaw.SetCWLimit(linear_min); yaw.SetCCWLimit(linear_MAX); 
    yaw.SetCRSpeed(yaw_Speed); yaw_goal=yaw_Init; yaw.SetGoal(yaw_Init);
    wait(1);
}

void GetDyna(char* data) {
    float lP, lV, pP, pC, yP, yC;
    
    lP = linear.GetPosition(); lV = linear.GetVolts();
    pP = pitch.GetPosition(); pC = pitch.GetCurrent();
    yP = yaw.GetPosition(); yC = yaw.GetCurrent();
    
    linear_goal=lP; pitch_goal=pP; yaw_goal=yP; //現在角度を目標値に設定
    
    sprintf(data,"%f %f %f %f %f %f ",lP,lV,pP,pC,yP,yC);  
}

void SetGoalDyna(int l_mode, int p_mode, int y_mode) {
    switch(l_mode){
        case 0: break;
        case 1: linear.SetTorqueLimit(1); linear_goal++; break;
        case 2: linear.SetTorqueLimit(1); linear_goal--; break;
    }    
    linear.SetGoal(linear_goal);
    
    switch(p_mode){
        case 0: break;
        case 1: pitch_goal++; break;
        case 2: pitch_goal--; break;
    }    
    pitch.SetGoal(pitch_goal);
    
    switch(y_mode){
        case 0: break;
        case 1: yaw_goal++; break;
        case 2: yaw_goal--; break;
    }
    yaw.SetGoal(yaw_goal);
}

void Dyna_home_position() {
    float lp;
    linear.SetGoal(linear_Init);
    lp=linear.GetPosition();
    if(lp > linear_MAX - 30) {  //ある程度縮んだら
        pitch.SetGoal(pitch_Init);
        yaw.SetGoal(yaw_Init);
    }
}

void Dyna_reset() {
    RelaySwitch = 0;
    wait_ms(10);
    RelaySwitch = 1;
}

void Dyna_end() {
    RelaySwitch = 0;
}
/*--Dynamixel:end-----------------------------------------------------------------------------------------*/


/*--Thermal_Sensor:begin----------------------------------------------------------------------------------*/
/*MEMS非接触温度センサ：形D6T-44L-06 4×4素子タイプ*/
/*データシート：http://www.omron.co.jp/ecb/products/sensor/special/mems/pdf/AN-D6T-01JP_r2.pdf*/
#include "MEMS.h"

MEMS MEMS1(p9, p10); // sda, scl
MEMS MEMS2(p28, p27); // sda, scl

void GetThermo(char* data) {
    float ThD1[16], ThD2[16];
    char con[10];
    
    MEMS1.temp(ThD1);
    MEMS2.temp(ThD2);
    
    for(int i=0;i<16;i++){
        sprintf(con,"%3.1f ",ThD1[i]);
        strcat(data,con);
    }
    for(int i=0;i<16;i++){
        sprintf(con,"%3.1f ",ThD2[i]);
        strcat(data,con);
    }
}
/*--Thermal_Sensor:end------------------------------------------------------------------------------------*/


/*--CO2_Sensor:begin--------------------------------------------------------------------------------------*/
/*CO2センサモジュール：A051020-AQ6B-01*/
/*データシート：http://www.fisinc.co.jp/common/pdf/A051020-AQ6.pdf*/
/*参考 外気:396.0[ppm](2013年) 呼気:13,200[ppm] ※このセンサで測れるのは3000[ppm]まで*/
#include "mbed.h"

AnalogIn ain(p20);

float GetCO2() {
    float v; //生データ：電圧
    float sensor_v,CO2;
    
    v = ain.read()*3.3;
    sensor_v = v * 5.0/3.3; //電圧レベルを合わせる
    CO2 = sensor_v * 1000 + 400;  //データシートより

    return(CO2);
}
/*--CO2_Sensor:end----------------------------------------------------------------------------------------*/


/*---------------
    MAIN ROOP
----------------*/
int main() {
    char Dyna_data[100];
    char Thermo_data[100];
    float CO2_data;
    
    Ethernet_init();    
    stdio_mutex.lock(); 
    Dyna_init();
    stdio_mutex.unlock();
    
    char S_data[200], R_data[2];
    int Joy_mode, linear_mode, pitch_mode, yaw_mode;
    
    while(1) {
        
        sock.receiveFrom(echo_server, R_data, sizeof(R_data)); //オペステからデータを取得
        wait_ms(1);
        int J = atoi(R_data);
        Joy_mode = J & 0x03; //ジョイスティックのモード判定
        
        switch(Joy_mode){
            case 0: //通常モード
                stdio_mutex.lock();
            /*--Dynamixel:begin---------------------------------------------------------------*/       
                //Dynamixelへの命令を判定
                linear_mode = R_data[0] & 0x0c; linear_mode = linear_mode >> 2;
                pitch_mode = R_data[0] & 0x30; pitch_mode = linear_mode >> 4;
                yaw_mode = R_data[0] & 0xc0; yaw_mode = linear_mode >> 6;                

                //現在の角度・電圧・電流を取得
                myled1=1;
                GetDyna(Dyna_data);
                myled1=0;
                
                //目標角度を変更
                myled2=1;
                SetGoalDyna(linear_mode, pitch_mode, yaw_mode);
                myled2=0;
            /*--Dynamixel:end------------------------------------------------------------------*/       
        
            /*--Thermal_Sensor:begin-----------------------------------------------------------*/
                //値を取得
                GetThermo(Thermo_data);
            /*--Thermal_Sensor:end-------------------------------------------------------------*/
        
            /*--CO2_Sensor:begin---------------------------------------------------------------*/
                //値を取得
                CO2_data=GetCO2();
            /*--CO2_Sensor:end-----------------------------------------------------------------*/
            
            //値を送信
                myled3=1;
                sprintf(S_data, "%s%s%f",Dyna_data,Thermo_data,CO2_data);
                stdio_mutex.unlock();
                sock.sendTo(echo_server, S_data, sizeof(S_data));
                myled3=0;           
                break;
            
            case 1: //ホームポジション
                stdio_mutex.lock(); Dyna_home_position(); stdio_mutex.unlock(); break;
                
            case 2: //リセット
                stdio_mutex.lock(); Dyna_reset(); stdio_mutex.unlock(); break;
            
            case 3: //終了
                stdio_mutex.lock(); myled4=1; Dyna_end(); stdio_mutex.unlock(); sock.close(true); eth.disconnect(); return 0;
        }
    }
}