/* mbed AX-12+ Servo Library
 *
 * Copyright (c) 2010, cstyles (http://mbed.org)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef MBED_MX28_H
#define MBED_MX28_H

#include "mbed.h"
#include "SerialHalfDuplex.h"

#define MX28_WRITE_DEBUG 0
#define MX28_READ_DEBUG 0
#define MX28_TRIGGER_DEBUG 0
#define MX28_DEBUG 0
#define MX28_ERROR_DEBUG 0

#define MX28_REG_ID 0x3
#define MX28_REG_CW_LIMIT 0x06
#define MX28_REG_CCW_LIMIT 0x08
#define MX28_REG_TORQUE_ENABLE 0x18
#define MX28_REG_GOAL_POSITION 0x1E
#define MX28_REG_MOVING_SPEED 0x20
#define MX28_REG_VOLTS 0x2A
#define MX28_REG_TEMP 0x2B
#define MX28_REG_MOVING 0x2E
#define MX28_REG_POSITION 0x24
#define MX28_REG_TORQUE_LIMIT 0x22
#define MX28_REG_CURRENT 0x38

#define MX28_MODE_POSITION  0
#define MX28_MODE_ROTATION  1

#define MX28_CW 1
#define MX28_CCW 0

class MX28
{

public:

    /** Create an MX28 servo object connected to the specified serial port, with the specified ID
     *
     * @param pin tx pin
     * @param pin rx pin
     * @param int ID, the Bus ID of the servo 1-255
     */
    MX28(PinName tx, PinName rx, int ID, int baudrate);

    /** Set the mode of the servo
     * @param mode
     *    0 = Positional, default
     *    1 = Continuous rotation
     */
    int SetMode(int mode);

    /** Set goal angle in integer degrees, in positional mode
     *
     * @param degrees 0-360
     * @param flags, defaults to 0
     *    flags[0] = blocking, return when goal position reached
     *    flags[1] = register, activate with a broadcast trigger
     *
     */
    int SetGoal(int degrees, int flags = 0);

    /** Set the speed of the servo in continuous rotation mode
     *
     * @param speed, -1.0 to 1.0
     *   -1.0 = full speed counter clock wise
     *    1.0 = full speed clock wise
     */
    int SetCRSpeed(float speed);


    /** Set the clockwise limit of the servo
     *
     * @param degrees, 0-360
     */
    int SetCWLimit(int degrees);

    /** Set the counter-clockwise limit of the servo
     *
     * @param degrees, 0-360
     */
    int SetCCWLimit(int degrees);

    // Change the ID

    /** Change the ID of a servo
     *
     * @param CurentID 1-255
     * @param NewID 1-255
     *
     * If a servo ID is not know, the broadcast address of 0 can be used for CurrentID.
     * In this situation, only one servo should be connected to the bus
     */
    int SetID(int CurrentID, int NewID);


    /** Poll to see if the servo is moving
     *
     * @returns true is the servo is moving
     */
    int isMoving(void);

    /** Send the broadcast "trigger" command, to activate any outstanding registered commands
     */
    void trigger(void);

    /** Read the current angle of the servo
     *
     * @returns float in the range 0.0-360.0
     */
    float GetPosition();

    /** Read the temperature of the servo
     *
     * @returns float temperature
     */
    float GetTemp(void);

    /** Read the supply voltage of the servo
     *
     * @returns float voltage
     */
    float GetVolts(void);

    /** Read the supply current of the servo
     *
     * @returns float current
     */
    float GetCurrent(void);

    int TorqueEnable(int mode);

    int SetTorqueLimit(float torque_limit);

private :

    SerialHalfDuplex _mx28;
//    Serial _mx28;
    int _ID;

    int read(int ID, int start, int length, char* data);
    int write(int ID, int start, int length, char* data, int flag=0);

};

#endif
