#ifndef ADE7912
#define ADE7912

#include "stm32h7xx_hal.h"
#include "stdlib.h"

// ADE7912 registers addresses
#define ADE7912_IWV_REG_ADDRESS 						0x00
#define ADE7912_V1WV_REG_ADDRESS 						0x08
#define ADE7912_V2WV_REG_ADDRESS 						0x10
#define ADE7912_ADC_CRC_REG_ADDRESS 				0x20
#define ADE7912_CTRL_CRC_REG_ADDRESS 				0x28
#define ADE7912_CNT_SNAPSHOT_REG_ADDRESS 		0x38
#define ADE7912_CONFIG_REG_ADDRESS 					0x40
#define ADE7912_STATUS0_REG_ADDRESS 				0x48
#define ADE7912_LOCK_REG_ADDRESS 						0x50
#define ADE7912_SYNC_SNAP_REG_ADDRESS 			0x58
#define ADE7912_COUNTER0_REG_ADDRESS 				0x60
#define ADE7912_COUNTER1_REG_ADDRESS 				0x68
#define ADE7912_EMI_CTRL_REG_ADDRESS 				0x70
#define ADE7912_STATUS1_REG_ADDRESS 				0x78
#define ADE7912_TEMPOS_REG_ADDRESS 					0xC0

// ADE7912 bits of CONFIG register
#define ADE7912_CONFIG_BIT_CLKOUTENB				1 << 0
#define ADE7912_CONFIG_BIT_PWRDWNENB				1 << 1
#define ADE7912_CONFIG_BIT_TEMPENB					1 << 3
#define ADE7912_CONFIG_ADC_FREQ_1K					0x0 << 4
#define ADE7912_CONFIG_ADC_FREQ_2K					0x1 << 4
#define ADE7912_CONFIG_ADC_FREQ_4K					0x2 << 4
#define ADE7912_CONFIG_ADC_FREQ_8K					0x3 << 4
#define ADE7912_CONFIG_ADC_FREQ							ADE7912_CONFIG_ADC_FREQ_8K
#define ADE7912_CONFIG_BIT_SWRST						1 << 6
#define ADE7912_CONFIG_BIT_BW								1 << 6

// ADE7912 bits of STATUS0 register
#define ADE7912_STATUS0_BIT_RESET_ON				1 << 0
#define ADE7912_STATUS0_BIT_CRC_STAT				1 << 1
#define ADE7912_STATUS0_BIT_IC_PROT					1 << 2

// ADE7912 bits of STATUS1 register
#define ADE7912_STATUS1_BIT_VERSION					0x7 << 0
#define ADE7912_STATUS1_BIT_ADC_NA					1 << 3

// ADE7912 transmite modes
#define ADE7912_READ_MODE										0x04
#define ADE7912_BRUSH_READ_MODE							0x04
#define ADE7912_WRITE_MODE									0x00

#define ADE7912_DALAY_TIME									2
#define ADE7912_WAITING_TIME								5 * ADE7912_DALAY_TIME

#define ADE7912_TEMPGAIN_WITH_3K3_BW				8.72101e-5f
#define ADE7912_TEMPGAIN_WITH_2K_BW					8.21015e-5f
#define ADE7912_CONST_TEMPOS								306.47f

#define ADE7912_IWV_TRANSLATE_COEF          5.87344e-9f
#define ADE7912_VWV_TRANSLATE_COEF          9.39369e-8f

struct ADE7912_BrushRead_Data
{
	int32_t IWV; 
	int32_t V1WV;
	int32_t V2WV;
	uint16_t ADC_CRC;
	uint8_t STATUS0;
	uint16_t CNT_SNAPSHOT;
};

struct ADE7912_Inst 
{
	SPI_HandleTypeDef *spi;
	uint8_t phasesEnable[4];
	GPIO_TypeDef *CS_ports[4];
	uint16_t CS_pins[4];
	struct ADE7912_BrushRead_Data *phasesData[4];
	uint8_t version[4];
	float tempos[4];
	float tempGain[4];
	GPIO_TypeDef *DReadyPort;
	uint16_t DReadyPin;
	IRQn_Type EXTIinterrupt;
};



enum ADE7912_Phases
{
	PHASE_A = 0,
	PHASE_B,
	PHASE_C,
	COM
};

enum ADE7912_DataUpdateFreq
{
	F_8KHZ,
	F_4KHZ,
	F_2KHZ,
	F_1KHZ
};

enum ADE7912_Bandwidths
{
	BW_3K3HZ,
	BW_2KHZ
};

enum ADE7912_CLKOUT_Functionality
{
	CLKOUT,
	DREADY
};

struct ADE7912_Phase_Settings
{
	GPIO_TypeDef *CS_port;
	uint16_t CS_pin;
	enum ADE7912_DataUpdateFreq freq;
	enum ADE7912_Bandwidths bandwidth;
	enum ADE7912_CLKOUT_Functionality clkoutFunc;
};



void ADE7912_UpdateData(struct ADE7912_Inst *ade);

struct ADE7912_Inst* New_ADE7912(SPI_HandleTypeDef *spi);

void ADE7912_PhaseInit(struct ADE7912_Inst *ade, struct ADE7912_Phase_Settings *settings, enum ADE7912_Phases phase);
void ADE7912_EnablePhase(struct ADE7912_Inst *ade, enum ADE7912_Phases phase);
void ADE7912_DisablePhase(struct ADE7912_Inst *ade, enum ADE7912_Phases phase);
void ADE7912_ResetPhase(struct ADE7912_Inst *ade, enum ADE7912_Phases phase);

void ADE7912_SetDataUpdateFreq(struct ADE7912_Inst *ade, enum ADE7912_Phases phase, enum ADE7912_DataUpdateFreq freq);
void ADE7912_SetPwrConverterEnabled(struct ADE7912_Inst *ade, enum ADE7912_Phases phase, bool enabled);
void ADE7912_SetTempEnabled(struct ADE7912_Inst *ade, enum ADE7912_Phases phase, bool enabled);
void ADE7912_SetBandwidth(struct ADE7912_Inst *ade, enum ADE7912_Phases phase, enum ADE7912_Bandwidths bandwidth);
void ADE7912_SetCLKOUTFunctionality(struct ADE7912_Inst *ade, enum ADE7912_Phases phase, enum ADE7912_CLKOUT_Functionality functionality);

void ADE7912_LockConfigurationRegisters(struct ADE7912_Inst *ade, enum ADE7912_Phases phase);
void ADE7912_UnlockConfigurationRegisters(struct ADE7912_Inst *ade, enum ADE7912_Phases phase);

uint8_t ADE7912_GetADCVersion(struct ADE7912_Inst *ade, enum ADE7912_Phases phase);
float ADE7912_GetVoltage(struct ADE7912_Inst *ade, enum ADE7912_Phases phase);
float ADE7912_GetCurrent(struct ADE7912_Inst *ade, enum ADE7912_Phases phase);
float ADE7912_GetTemp(struct ADE7912_Inst *ade, enum ADE7912_Phases phase);

#endif
