
/**

    @file main.cpp
    @brief Thrust Motor Controller 2018 edition ver. 0.2.0
    \mainpage Thrust Motor Controller 2018 edition
    TCPSocket経由でスラスターを制御するプログラムです。
    DHCPサーバとしては機能しません。
    固定IPを割り振る必要があります。
    \htmlonly
    <font color="red">mbedの電源を投入する前に,あらかじめクライアントのコンピュータを接続しておく必要があります。</font>
    \endhtmlonly
    @sa ethernet_thread
    @author Wada Suisei   Copyright (C) 2018 - All rights reserved          
    @author modified by yukimakura

    @date LastModified:2018 Oct 23 
*/

/*** Including Libraries ***/
#include "mbed.h"
#include "EthernetInterface.h"
#include "rtos.h"


/**
 * @brief PWMの周期（秒）
 * 
 */
#define PWM_FREQ 0.01 

/**
 * @brief 接続のタイムアウト（ミリ秒）
 * 
 */
#define TIMEOUT_MS 1000 


/*** Network Information ***/
///myIpAddrこのモータドライバのIPアドレスを入力
#define myIpAddr "10.42.0.98" 
/// myPortこのプログラムで使用するポートを入力  
#define myPort 50000
/// NetMask このモータドライバ用のネットマスクを入力 
#define NetMask "255.255.255.0"
/// Gateway このモータドライバ用のゲートウェイを入力 
#define Gateway "10.42.0.1"


/*** Instantiation ***/
Timer Timeout_timer;

//! PWM信号の差動出力ピンを指定
PwmOut motor_P(p26);
//! PWM信号の差動出力ピンを指定
PwmOut motor_N(p25);

Thread *(eth_pri);

double pipe_pwmduty;

Mutex g_mtx;

//Network
DigitalOut EthRecv_LED(LED1);
DigitalOut EthSend_LED(LED2);
DigitalOut Ethconnect_LED(LED4);
EthernetInterface eth;





/**
 * 
 * @brief コネクション時に重大なエラーが発生した時に呼び出されます
 * 
 * pwmをduty比それぞれ50%にします
 * 
 * 
 */
void timeout(void){
    g_mtx.lock();
    pipe_pwmduty = 0.0;
    g_mtx.unlock();
    motor_P = 0.5;
    motor_N = 0.5;
}



/**
 * 
 * @brief TCP/IPソケットでコミュニケーションかつデータ受信を行います
 * 
 *  RTOSによって管理されているスレッドで,
 * TCPSocketの確立前にデータリンク層あたりでコネクションされていないと
 * mbedOSに−１を返します。
 */
void ethernet_thread(void){
    char in_buf[64];
    
    EthernetInterface eth;
    //eth.init();
    eth.init(myIpAddr,NetMask,Gateway);
    eth.connect();
    printf("\nServer IP Address is %s\n", eth.getIPAddress());
    
    TCPSocketServer server;
    server.bind(myPort);
    server.listen();
    
    while (true) {
        printf("\nWait for new connection...\n");
        TCPSocketConnection *client = new TCPSocketConnection;
        server.accept(*client);
        //client.set_blocking(false, TIMEOUT_MS); // Timeout after (TIMEOUT_MS)ms
        
        printf("Connection from: %s\n", client->get_address());
        
        while (client->is_connected()) {
            
            Ethconnect_LED = 1;
            int len = client->receive(in_buf, sizeof(in_buf));
            EthRecv_LED = !EthRecv_LED;
            // print received message to terminal
            in_buf[len] = '\0';
            if(len > 0){
               // Timeout_timer.reset();
                in_buf[len] = '\0';
                //Get data
                EthRecv_LED = !EthRecv_LED;
                g_mtx.lock();
                memcpy(&pipe_pwmduty, &in_buf[0], 8);
                //printf("Received message from Client :'%f'\n",pipe_pwmduty);
                g_mtx.unlock();
                Timeout_timer.reset();
            }
            if(Timeout_timer.read_ms() > TIMEOUT_MS){
                break;
            }
        }
        printf("\nclient closed\n");
        Ethconnect_LED = 0;
        timeout();
        client->close();
        delete client;
    }
}

/**
 * @brief エントリーポイント
 * 
 * @return OSに返します。
 */
int main(){
    Timeout_timer.start();
    /* Declaration of Local Variables */
    float pwmduty;

    /* Initializing Variables */
    EthRecv_LED = 0;
    EthSend_LED = 0;
        
    /* Initializing Interruption */


    
    /* Initializing PWM Out */
    motor_P.period (PWM_FREQ);
    motor_N.period (PWM_FREQ);
    motor_P = 0.5;
    motor_N = 0.5;
    
    /* Start Thread */
    Thread thread_get(ethernet_thread);
    eth_pri = &thread_get;
    eth_pri -> set_priority(osPriorityHigh);
    
    /* Main Routine */
    while(true){
        //disconnect timeout
        if(Timeout_timer.read_ms() > TIMEOUT_MS){
            g_mtx.lock();
            pipe_pwmduty = 0.0;  
            g_mtx.unlock();
            timeout();
            EthRecv_LED = 0;
            EthSend_LED = 0;
            Ethconnect_LED = 0;
            //printf("%d\n",Timeout_timer.read_ms());
            Timeout_timer.reset();
            //return -1;
        }
        
        /* Calculate PWM Duty */
        g_mtx.lock();
        pwmduty = pipe_pwmduty / 2.0;
        g_mtx.unlock();

        /* Output PWM */
        motor_P = (0.5 - pwmduty);
        motor_N = (0.5 + pwmduty);

//        motor_P = (0.5 + pwmduty);
//        motor_N = (0.5 - pwmduty);

        
    }//while(1) END

}//main() END


