#include "setup.hpp"

// mcsvmのセットアップ : サンプル/係数のセット
static void mcsvm_setup(void)
{
    FILE* svm_setup_fp;
    char buf_str[20];
    int ret, line;
    float buf_data[DIM_SIGNAL];

    float* svm_tmp_sample     = new float[MCSVM_NUM_SAMPLES * DIM_SIGNAL];
    int* svm_tmp_sample_label = new int[MCSVM_NUM_SAMPLES];
    float* svm_tmp_mc_alpha   = new float[MCSVM_NUM_SAMPLES * NUM_WEATHERS * (NUM_WEATHERS - 1) / 2];

    // You can use /dat/SVM_SAMP.txt
    svm_setup_fp = fopen( "/local/SVM_SAMP.CSV" , "r" );
    if( svm_setup_fp == NULL ) {
        fprintf( stderr, "Error in svm setup : sample file cannot open. \r \n" );
        exit(1);
    }

    line = 0;
    while( ( ret = fscanf( svm_setup_fp, " %[^\n,],%f,%f,%f", buf_str, &(buf_data[0]), &(buf_data[1]), &(buf_data[2])) ) != EOF ) {

        if ( !strcmp(buf_str,"shiny") ) {
            svm_tmp_sample_label[line] = SHINY;
        } else if ( !strcmp(buf_str,"cloudy") ) {
            svm_tmp_sample_label[line] = CLOUDY;
        } else if ( !strcmp(buf_str,"rainy") ) {
            svm_tmp_sample_label[line] = RAINY;
        } else if ( !strcmp(buf_str,"snowy") ) {
            svm_tmp_sample_label[line] = SNOWY;
        } else {
            continue;
        }

        memcpy(&(svm_tmp_sample[line * DIM_SIGNAL]), buf_data, sizeof(float) * DIM_SIGNAL);
        // printf("svm sample loading.... ret : %d line : %d %s %f %f %f \r\n", ret, line, buf_str, svm_tmp_sample[line*3], svm_tmp_sample[line*3+1], svm_tmp_sample[line*3+2]);
        line++;
    }

    mcsvm = new MCSVM(NUM_WEATHERS, DIM_SIGNAL, MCSVM_NUM_SAMPLES, svm_tmp_sample, svm_tmp_sample_label);

    // Thank you freopen.
    // Here, we should not use fclose -> fopen
    
    // You can use /dat/SVM_ALPH.txt
    svm_setup_fp = freopen("/local/SVM_ALPH.CSV", "r", svm_setup_fp );
    fflush( svm_setup_fp ); // required.

    if ( svm_setup_fp == NULL ) {
        fprintf( stderr, "Error in open learned alpha data. \r\n");
        exit(1);
    }

    // 一列のデータではfscanfフォーマットがだるいので, fgetsを使用
    line = 0;
    while( fgets( buf_str, 20, svm_setup_fp ) != NULL ) {
        svm_tmp_mc_alpha[line] = atof(buf_str);
        // printf("%d %f \r\n", line, tmp_mc_alpha[line]);
        line++;
    }

    fclose( svm_setup_fp );

    mcsvm->set_alpha(svm_tmp_mc_alpha, MCSVM_NUM_SAMPLES, NUM_WEATHERS);

    delete [] svm_tmp_sample;
    delete [] svm_tmp_sample_label;
    delete [] svm_tmp_mc_alpha;

}

// SRNNのセットアップ. 初期データのセット.
static void srnn_setup(void)
{
    FILE* srnn_setup_fp;
    int ret;
    float buf_data[DIM_SIGNAL];
    float sample_maxmin[DIM_SIGNAL * 2];
    char buf_str[20];
    
    // グローバルなサンプルキューを**ここで**アロケート
    srnn_sample_queue = new float[DIM_SIGNAL * LEN_DATA_SEQUENCE];

    // 信号の正規化のために, 信号の最大値と最小値を決めてやる必要がある.
    sample_maxmin[0] = 50;
    sample_maxmin[1] = -20;    // 気温の最大/最小値（想定値）
    sample_maxmin[2] = 1030;
    sample_maxmin[3] = 960;    // 気圧
    sample_maxmin[4] = 100;
    sample_maxmin[5] = 0;      // 湿度

    srnn_setup_fp = fopen( SEQUENCE_DATA_NAME, "r");
    if( srnn_setup_fp == NULL ) {
        fprintf( stderr, "Error in SRNN setup. sample file cannot open. \r\n");
        exit(1);
    }

    int line = 0;
    while( ( ret = fscanf( srnn_setup_fp, " %[^\n,],%f,%f,%f", buf_str, &(buf_data[0]), &(buf_data[1]), &(buf_data[2])) ) != EOF ) {
        if ( line == LEN_DATA_SEQUENCE ) break;
        memcpy(&(srnn_sample_queue[line * DIM_SIGNAL]), buf_data, sizeof(float) * DIM_SIGNAL);
        // printf("sample %d : %f %f %f \r\n", line, MATRIX_AT(srnn_sample_queue,DIM_SIGNAL,line,0),MATRIX_AT(srnn_sample_queue,DIM_SIGNAL,line,1), MATRIX_AT(srnn_sample_queue,DIM_SIGNAL,line,2));
        line++;
    }

    fclose( srnn_setup_fp );

    /* アドバイス:RNNにおいては,ダイナミクス(中間層のニューロン数)は多いほど良い */
    srnn = new SRNN(DIM_SIGNAL, 20, LEN_DATA_SEQUENCE, PREDICT_LENGTH, srnn_sample_queue, sample_maxmin);

}

// センサーのセットアップ.
static void sensor_setup(void)
{
    sensor_module = new SensorModule(5);
}

// ネットワークのセットアップ
static void network_setup(void)
{
    // セットアップ, 最初の時間取得
    const char prefix_net_str[] = "[Network Status]";

    //setup ethernet interface
    printf("%s Ethernet initializing....", prefix_net_str);
    if ( eth_if.init() < 0 ) {// Use DHCP
        fprintf( stderr, "%s Ethernet init failed. \r\n", prefix_net_str);
        exit(1);
    }

    if ( eth_if.connect() < 0 ) {
        // (offlineが確定する -> offline modeへ).
        fprintf( stderr, "%s Ethernet connect failed. Go To offline \r\n", prefix_net_str);
        return;
    }

    // init time
    printf("%s Trying to update time...\r\n", prefix_net_str);

    // Please specify near ntp server. ex) Japan -> ntp.nict.jp:123
    if (ntp_client.setTime("ntp.nict.jp") == 0) {
        printf("%s Set time successfully! \r\n", prefix_net_str);
    } else {
        fprintf( stderr, "%s Error in setup time \r\n", prefix_net_str);
    }

    // setup http server
    // http_server = new HTTPServer(80, "/local/");

    printf("%s IP Address : %s \r\n", prefix_net_str, eth_if.getIPAddress());
    printf("%s Network setup finished! \r\n", prefix_net_str);
}

// グラフィックハンドラの初期化
static void graphic_handler_setup(void)
{
    graphic_handler = new GraphicHandler(DIM_SIGNAL,  PREDICT_INTERVAL_TIME, PREDICT_LENGTH, time(NULL));
}

// データの初期化（アロケート）
static void data_setup(void)
{
    new_seqence_data        = new float[DIM_SIGNAL];                    // 現在の(一番新しい)系列データ
    new_predict_data        = new float[DIM_SIGNAL * PREDICT_LENGTH];   // 現在の予測結果
    new_predict_weather     = new int[PREDICT_LENGTH];                  // 現在の予測天気
    new_predict_probability = new float[PREDICT_LENGTH];                // 現在の予測天気の確率
}

//　時刻の設定
static void jst_setup(int year, int month, int day, int hour, int min)
{
    // setup time structure
        struct tm t;
        t.tm_sec  = 0;          // 0-59
        t.tm_min  = min;        // 0-59
        t.tm_hour = hour;       // 0-23
        t.tm_mday = day;        // 1-31
        t.tm_mon  = (month-1);    // 0-11
        t.tm_year = (year-1900);  // year since 1900

        // convert to timestamp and set
        time_t seconds = mktime(&t);
        set_time(seconds);
}

// セットアップ.
void setup(void)
{
    printf("SETUP START ");
    printf("-------------------------- \r\n");
    mcsvm_setup();
    printf("SVM ...OK \r\n");
    srnn_setup();
    printf("SRNN ...OK \r\n");
    //sensor_setup();
    printf("SENSOR ...OK \r\n");
    // network_setup();
    jst_setup(2015, 2, 20, 4, 0);
    printf("NETWORK ...NO(offline) \r\n");
    graphic_handler_setup();
    printf("GRAPHIC ...OK \r\n");
    data_setup();
    printf("SHARED DATA ...OK \r\n");
    printf("SETUP SUCCESS ");
    printf("-------------------------- \r\n");
}
