#include "SensorModule.hpp"

// コンストラクタ
SensorModule::SensorModule(int n_sample)
{
    // サンプリング回数, インスタンス初期化
    this->n_sample = n_sample;
    this->temperture_sensor = new AnalogIn(p15);
    this->humidity_sensor   = new DHT11(p11);
    this->pressure_i2c      = new I2C(p9, p10);
    this->pressure_sensor   = new MPL115A2(pressure_i2c);
    pressure_sensor->begin();
    
    // デフォルト値をセットしとく
    new_temperture = 10;
    new_pressure = 1000;
    new_humidity = 40;

}

// デストラクタ
SensorModule::~SensorModule(void)
{
    delete temperture_sensor;
    delete humidity_sensor;
    delete pressure_i2c;
    delete pressure_sensor;
}

// 全センサーからn_sample回読み出し, 平均値で更新する.
void SensorModule::read_all_sensor(void)
{
    int fail_count;
    register float sum_temp = 0, sum_humid = 0, sum_pres = 0;
    // float sum_temp2 = 0; 気圧センサ, 実は温度も測れる(<-LEDで明らかに熱そうなので廃止.)

    pressure_sensor->ReadSensor();

    fail_count = 0;
    while(1) {
        if (fail_count >= 5) {
            fprintf( stderr, "Error in Temperaure / Pressure Sensor \r\n");
            exit(1);
        }
        // AnalogInを用いる場合は,n_samle回センサーから読み出す.
        sum_temp = sum_pres = 0;
        for (int count = 0; count < n_sample; count++ ) {
            // LM61CIZの温度読み取り
            sum_temp  += (temperture_sensor->read()-(0.6/3.3))/(0.01/3.3);
            // MPL115A2の気圧読み取り
            pressure_sensor->ReadSensor();
            sum_pres  += pressure_sensor->GetPressure();
            // sum_temp2 += pressure_sensor->GetTemperature();
            wait(0.05);
        }
        if ( fabsf(new_temperture - (sum_temp / n_sample)) < 40 
                && fabsf(new_pressure - (sum_pres / n_sample)) < 40 ) {
            break;
        } else {
            // 異常値ならばやり直し(たま〜に異常値をとる...)
            fail_count++;
        }
    }

    // 更新
    new_temperture = sum_temp / n_sample;
    new_pressure   = sum_pres / n_sample;

    // printf("LM61CIZ : %f vs. MPL115A2 : %f \r\n", sum_temp / n_sample, sum_temp2 / n_sample);

    // DHT11の湿度読み取り : 一発読み込み成功すれば良し.
    fail_count = 0;
    while(1) {
        if (fail_count >= 5) {
            fprintf( stderr, "Error in Humidity Sensor \r\n");
            exit(1);
        }

        if (humidity_sensor->readData() == DHT11::DHT_OK) {
            new_humidity = float(humidity_sensor->readHumidity());
            break;
        } else {
            fail_count++;
        }
        wait(0.1);
    }

}
