/*************************************************************************************
Author: Yuhang Zhu(b46609@freescale.com)
Date: July 20, 2013

Function name: main

Function description:
This main function implements a mode transfer.
When MCU resets, it's in MODE_IDLE. Then depends on the keyboard input(1 to 7), 
mode switches to transition modes such as 
MODE_SPI_RAW, which is for inputting 16 bit SPI stream directly
MODE_CHAxPHA, which is for phase input of channel x(1 or 2), or
MODE_CHAxAMP, which is for attenuation input of channel x(1 or 2), or
MODE_DISPLAY, which is for displaying the current parameters, or
MODE_HELP, which is for displaying the help information, or
MODE_SEND, which is for sending the bit stream through the SPI bus by MOSI and reading the data back from MISO.
After each transition mode completes, MCU returns to MODE_IDLE to wait for another user input.

***************************************************************************************/

#include "mbed.h"
#include "menu.h"
#include "parameters.h"
#include "handlers.h"
//#define DEBUG

Serial pc(USBTX, USBRX);   // serial io
SPI spi(PTD2, PTD3, PTD1); // mosi, miso, sclk
DigitalOut cs(PTD0);       //SPI chip select

char recv;


int state;
char cha1_pha, cha2_pha;
char cha1_amp, cha2_amp;
int ret;

unsigned short spi_stream;
unsigned char spi_first_byte_s;
unsigned char spi_second_byte_s;  // The first and second byte to send

unsigned char spi_first_byte_r;   // The first and second byte to receive
unsigned char spi_second_byte_r;

char line_buf[4];   //line_buf[3] is used to indicate if it's empty 0 empty 1 not, other bytes for char data
int line_pt;        //line buffer pointer points to next buffer location it's going to write
int line_length;    //store the current line length
int i;

int main() {

    pc.baud(19200);   //Configure baud rate. Note: 115200 seems too high for the current board
    cs = 1;           //SPI config
    spi.format(8,0);  //16 bit frame is not working, so need to send twice 8 bit frame
                      //For more info about format, see http://mbed.org/forum/bugs-suggestions/topic/4419/?page=1#comment-22056
    spi.frequency(1000000);  
    
    state = MODE_IDLE;
    line_buf[3] = 0;
    line_buf[2] = 255;
    line_buf[1] = 255;
    line_buf[0] = 255;
    cha1_pha = 0;
    cha2_pha = 0;
    cha1_amp = 0;
    cha2_amp = 0;
    
    while(1)
    {
        switch(state)
        {
            case MODE_IDLE:
                pc.printf("%s", main_menu);
                recv = pc.getc();
                mode_idle_handler(&state, recv);  //Change state based on recv input
                 
                break;
            case MODE_SPI_RAW:                    //Input the SPI raw bits mode
                pc.printf("%s", cmd5);
                line_buf[3] = 255;
                line_buf[2] = 255;
                line_buf[1] = 255;
                line_buf[0] = 255;
                line_pt = 0;
                line_length = 0;
                while(1)
                {
                    recv = pc.getc();
                    if(recv == 13)    //enter is pressed
                    {
                        #ifdef DEBUG
                        pc.printf("decimal output : %d %d %d %d", line_buf[3], line_buf[2], line_buf[1], line_buf[0]);
                        pc.printf("char output: %c %c %c %c", line_buf[3], line_buf[2], line_buf[1], line_buf[0]);
                        pc.printf("length = %d\n", line_length);
                        #endif
                        break;
                    }
                    else if(recv == 8 || recv == 127)
                    {
                        if(line_length > 0)  //Delete one char on screen
                        {
                            pc.putc(8);
                            pc.putc(32);
                            pc.putc(8);
                        }
                        if(line_length <= 4 && line_length>=1)   
                        {
                            line_pt --;
                            line_buf[line_pt] = 255;
                        }
                        if(line_length > 0)
                            line_length --;
                    }
                    else if( (recv >= '0' && recv <= '9')||(recv >= 'a' && recv <= 'f')||(recv >= 'A' && recv <= 'F') )  //Input one char
                    {
                        pc.putc(recv);
                        line_length ++;
                        
                        if(line_length <= 4)
                        {
                            line_buf[line_pt] = recv;
                            line_pt ++;
                        }
                    
                    }
                
                }
            // Input over, checking
            ret = parse_spi_raw(line_buf, line_length, &cha1_pha, &cha2_pha, &cha1_amp, &cha2_amp);
            
            if(ret == PARSE_LONG)   //Input string too long
            { 
                pc.printf("%s", msg_strlong);
                pc.getc();
            }
            else if(ret == PARSE_EMPTY)  //Empty string
            {
                pc.printf("%s", msg_empty);
                pc.printf("%s", msg_ret);
                pc.getc();
            }
            else if(ret == PARSE_SHORT) //Input string too short
            {
                 pc.printf("%s", msg_strshort);
                 pc.getc();
            } 
            else   //Input is good
            {    
                 pc.printf("%s", msg_ok);
                 #ifdef DEBUG
                 pc.printf("%d %d %d %d \n", cha2_amp, cha2_pha, cha1_amp, cha1_pha);
                 #endif
                 pc.printf("%s", msg_ret);
                 pc.getc();
            }  
            state = MODE_IDLE;
            break;
            
                
            case MODE_CHA1PHA:
            case MODE_CHA1AMP:
            case MODE_CHA2PHA:
            case MODE_CHA2AMP:
                if(state == MODE_CHA1PHA)
                    pc.printf("%s", cmd1);
                else if(state == MODE_CHA1AMP)
                    pc.printf("%s", cmd2);
                else if(state == MODE_CHA2PHA)
                    pc.printf("%s", cmd3);
                else
                    pc.printf("%s", cmd4);
                    
                line_buf[3] = 0;
                line_buf[2] = 255;
                line_buf[1] = 255;
                line_buf[0] = 255;
                line_pt = 0;
                line_length = 0;
                
                while(1)   //Character input into line buffer, terminated by enter key
                {
                    recv = pc.getc();
                    if(recv == 13)   //Enter is pressed, break
                    {
                        if(line_length == 0)
                            line_buf[3] = LINEBUF_EMPTY;
                        else if(line_length > 3)
                            line_buf[3] = LINEBUF_TOOLONG;
                        else
                            line_buf[3] = LINEBUF_OK;
                        #ifdef DEBUG    
                        pc.printf("decimal output : %d %d %d %d", line_buf[3], line_buf[2], line_buf[1], line_buf[0]);
                        pc.printf("char output: %c %c %c %c", line_buf[3], line_buf[2], line_buf[1], line_buf[0]);
                        #endif
                           
                        break;  
                    }
                    else if(recv == 8 || recv == 127)   //Backspace is pressed, delete one character both on screen and line buffer
                    {
                        if(line_length > 0)  //Delete one char on screen
                        {
                            pc.putc(8);
                            pc.putc(32);
                            pc.putc(8);
                        }
                        
                        if(line_length <= 3 && line_length>=1)   
                        {
                            line_pt --;
                            line_buf[line_pt] = 255;
                        }
                        if(line_length > 0)
                            line_length --;
                    }
                    else if(recv == '.' || recv == '-' || (recv >= '0' && recv <= '9') )   //The line buffer accepts -, ., 0-9 only
                    {
                        pc.putc(recv);
                        line_length ++;
                        
                        if(line_length <= 3)
                        {
                            line_buf[line_pt] = recv;
                            line_pt ++;
                        }
                    }
                }
                //Input over, now checking 
                //Check phase shift input
                if(state == MODE_CHA1PHA || state == MODE_CHA2PHA)
                {  
                   if(state == MODE_CHA1PHA)
                        ret = parse_phase(line_buf, &cha1_pha);
                   else
                        ret = parse_phase(line_buf, &cha2_pha);
                        
                   if(ret == PARSE_LONG)   //Input string too long
                   { 
                        pc.printf("%s", msg_strlong);
                        pc.getc();
                   }
                   else if(ret == PARSE_EMPTY)
                   {
                        pc.printf("%s", msg_empty);
                        pc.printf("%s", msg_ret);
                        pc.getc();
                   }
                   else if(ret == PARSE_ERR) //Input string has error
                   {
                        pc.printf("%s", msg_err);
                        pc.getc();
                   }
                   else if(ret == PARSE_ROUNDED) //Input is rounded, show the rounded value
                   {    
                        if(state == MODE_CHA1PHA)
                            pc.printf("%s %d\n", msg_rounded, cha1_pha*-7);
                        else if(state == MODE_CHA2PHA)
                            pc.printf("%s %d\n", msg_rounded, cha2_pha*-7);
                            
                        pc.printf("%s", msg_ret);
                        pc.getc();
                   }
                   else   //Input is good
                   {    
                        pc.printf("%s", msg_ok);
                        #ifdef DEBUG
                        pc.printf("%d %d\n", cha1_pha*7, cha2_pha*7);
                        #endif
                        pc.printf("%s", msg_ret);
                        pc.getc();
                   }
                } 
                
                //Check attenuation input
                if(state == MODE_CHA1AMP || state == MODE_CHA2AMP)
                {
                    if(state == MODE_CHA1AMP)
                        ret = parse_amp2(line_buf, &cha1_amp, line_length);
                    else
                        ret = parse_amp2(line_buf, &cha2_amp, line_length);
                    
                   if(ret == PARSE_LONG)   //Input string too long
                   { 
                        pc.printf("%s", msg_strlong);
                        pc.getc();
                   }
                   else if(ret == PARSE_EMPTY)
                   {
                        pc.printf("%s", msg_empty);
                        pc.printf("%s", msg_ret);
                        pc.getc();
                   }
                   else if(ret == PARSE_ERR) //Input string has error
                   {
                        pc.printf("%s", msg_err);
                        pc.getc();
                   }
                   else if(ret == PARSE_ROUNDED) //Input is rounded, show the rounded value
                   {    
                        if(state == MODE_CHA1AMP)
                            pc.printf("%s %.1f\n", msg_rounded, cha1_amp*5.0/10.0);
                        else if(state == MODE_CHA2AMP)
                            pc.printf("%s %.1f\n", msg_rounded, cha2_amp*5.0/10.0);
                            
                        pc.printf("%s", msg_ret);
                        pc.getc();
                   }
                   else   //Input is good
                   {    
                        pc.printf("%s", msg_ok);
                        #ifdef DEBUG
                        pc.printf("%d %d\n", cha1_amp*5, cha2_amp*5);
                        #endif
                        pc.printf("%s", msg_ret);
                        pc.getc();
                   }
                }  
                state = MODE_IDLE;  //back to idle state after input
                break;
            
            //In display mode, show current parameters
            case MODE_DISPLAY:
                pc.printf("\n\n ******** Current Parameters ********\n\n");
                pc.printf("Phase shift  for RF out1 is %d degrees\n", cha1_pha*-7);
                pc.printf("Attenuation  for RF out1 is %.1f dB\n", cha1_amp*5.0/10.0);
                pc.printf("Phase shift  for RF out2 is %d degrees\n", cha2_pha*-7);
                pc.printf("Attenuation  for RF out2 is %.1f dB\n", cha2_amp*5.0/10.0);
                pc.printf("SPI stream(b7-b0 a7-a0) in hex: %c%c%c%c\n", hex2char(cha2_amp), hex2char(cha2_pha), hex2char(cha1_amp), hex2char(cha1_pha));
                pc.printf("%s", msg_ret);
                pc.getc();
                state = MODE_IDLE;
                break;
            //Help mode, displays help info   
            case MODE_HELP:
                pc.printf("\n\n******** Help info ********\n");
                pc.printf("%s", msg_help);
                pc.printf("%s", msg_ret);
                pc.getc();
                state = MODE_IDLE;
                break;
                
            case MODE_SEND:  //In sending mode, display and send through SPI
            
                //Calculate the send info
                spi_stream = (cha2_amp << 12) | (cha2_pha << 8) | (cha1_amp << 4) | (cha1_pha);
                spi_first_byte_s = (cha2_amp << 4) | (cha2_pha);
                spi_second_byte_s = (cha1_amp << 4) | (cha1_pha);
                
                cs = 0;   //SPI start sending, send and record the response
                spi_first_byte_r = spi.write(spi_first_byte_s);
                spi_second_byte_r = spi.write(spi_second_byte_s);
                cs = 1;
                
                //Start printint...
                pc.printf("******** Now sending parameters ********\n\n");
                pc.printf("Phase shift  for RF out1 is %d degrees\n", cha1_pha*-7);
                pc.printf("Attenuation  for RF out1 is %.1f dB\n", cha1_amp*5.0/10.0);
                pc.printf("Phase shift  for RF out2 is %d degrees\n", cha2_pha*-7);
                pc.printf("Attenuation  for RF out2 is %.1f dB\n", cha2_amp*5.0/10.0);
                pc.printf("\nSPI bit stream sent:\n");
                pc.printf("%s", bit_index);
                
                //Output the SPI bit stream sent for user reference
                for(i=0; i<=15; i++)
                {
                    
                    if(spi_stream >= 0x8000)
                        pc.printf("%d  ", 1);
                    else
                        pc.printf("%d  ", 0);
                    
                    if(i == 7)
                        pc.printf("   ");
                        
                    spi_stream = spi_stream << 1;
                }
                //pc.printf("\n\nParameters sent OK\n\n\n\n");
                
                
                //Output the previous parameters:
                pc.printf("\n\n\n******** Now displaying previous parameters ********\n");
                pc.printf("\nPrevious phase shift  for RF out1 is %d degrees\n", (spi_second_byte_r&0x07)*-7);
                pc.printf("Previous attenuation for RF out1 is %.1f dB\n", (spi_second_byte_r>>4)*5.0/10.0);
                pc.printf("Previous phase shift  for RF out2 is %d degrees\n", (spi_first_byte_r&0x07)*-7);
                pc.printf("Previous attenuation for RF out2 is %.1f dB\n", (spi_first_byte_r>>4)*5.0/10.0);
                //SPI bit stream for previous parameters
                pc.printf("\nSPI bit stream received:\n");
                pc.printf("%s", bit_index);
                for(i=0; i<=7; i++)
                {
                    if(spi_first_byte_r >= 0x80)
                        pc.printf("%d  ", 1);
                    else
                        pc.printf("%d  ", 0);
                    
                    spi_first_byte_r = spi_first_byte_r << 1;
                }
                pc.printf("   ");
                for(i=0; i<=7; i++)
                {
                    if(spi_second_byte_r >= 0x80)
                        pc.printf("%d  ", 1);
                    else
                        pc.printf("%d  ", 0);
                    
                    spi_second_byte_r = spi_second_byte_r << 1;
                }
                
                pc.printf("\n\n\n");
                
                pc.printf("%s", msg_ret);
                pc.getc();
                state = MODE_IDLE;
                break; 
            default:
                state = MODE_IDLE;
        }
        
        
    }
    
    
    return 0;
}