/********************************************
Function: void mode_idle_handler(int* state, char sel) 
Function description:
This function converts character inputs to state information, which is used for later stages in main function

Function: int parse_phase(char* line_buf, char* adam_pha)
Function description:
This function checks data in the line buffer. The line_buf contains phase value in ascii.
If it's good then write the converted value into address adam_pha otherwise it returns error code

Function: int parse_amp(char* line_buf, char* adam_amp)
Function description:
This function checks data in the line buffer. The line_buf contains attenuation value in ascii.
If it's good then write the converted value into address adam_amp otherwise it returns error code.

Function: int parse_amp2(char* line_buf, char* adam_amp)
Function description: This function is more robust than the previous one. Some other formats, like
.x, .xxx, x.xxx is supported. The function is used in main.


Function: static char char2hex_4bits(char c)
Function description:
This function converts a character input into 4 bit hex. For example, '1'->1, '9'->9, 'A'->10, 'a'->10

Function: static char char2hex_3bits(char c)
Function description:
This function converts a character input into 3 bit hex. For example, '8'->0, '7'->7,'f'->7

Function: int parse_spi_raw(char* buf, int length, char* cha1_pha, char* cha2_pha, char* cha1_amp, char* cha2_amp)
Function description:
This function converts characters in buf and put the data in cha1_pha, cha2_pha, cha1_amp, cha2_amp. 
If length is incorrect, return the error code.

Function: char hex2char(char hex)
Function description:
This function converts a hex number to character. For example, 1->'1', 10->'A'
*********************************************/

#include "parameters.h"

void mode_idle_handler(int* state, char sel)
{
    if(sel>='1' && sel<='8')
        *state = sel - '0';    //select a mode
    else 
        *state = MODE_IDLE;
}


int parse_phase(char* line_buf, char* adam_pha)
{

    char temp = 0;
    
    if(line_buf[3] == LINEBUF_EMPTY)
        return PARSE_EMPTY;    //empty line
    if(line_buf[3] == LINEBUF_TOOLONG)
        return PARSE_LONG;    //buffer too long
        
    if(line_buf[0] == '0')  //input is 0
    {
        if(line_buf[1] == 255 && line_buf[2] == 255)
        {
            *adam_pha = 0; 
            return PARSE_OK;
        }
        else
            return PARSE_ERR;     
    }
    
    

        
    if(line_buf[1]>='0' && line_buf[1]<='4' && line_buf[2]>='0'&& line_buf[2]<='9' && line_buf[0] == '-')
    {
        temp = (line_buf[1] - '0')*10;
        temp += line_buf[2] - '0';
        *adam_pha = temp/7;

        if(temp % 7 == 0)
            return PARSE_OK;
        else
            return PARSE_ROUNDED;
    }  //double digit input
    else if(line_buf[2] == 255 && line_buf[1]>='0' && line_buf[1] <= '9' && line_buf[0] == '-')
    {
        temp = line_buf[1] - '0';
        *adam_pha = temp/7;

        if(temp % 7 == 0)
            return PARSE_OK;
        else
            return PARSE_ROUNDED;
    }   //single digit input 
    else
        return PARSE_ERR;
}  

int parse_amp(char* line_buf, char* adam_amp)
{

    char temp = 0;

    if(line_buf[3] == LINEBUF_EMPTY)
        return PARSE_EMPTY;
    if(line_buf[3] == LINEBUF_TOOLONG)
        return PARSE_LONG;
    //Single digit input. single digit is 0,1,2,3,4,5,6,7
    if(line_buf[0]>='0' && line_buf[0]<='7' && line_buf[1] == 255 && line_buf[2] == 255)
    {
        temp = (line_buf[0] - '0')*10;
        *adam_amp = temp/5;
        return PARSE_OK;
    }  
    //x.x format input eg. 0.0 0.5 1.5 1.0  7.0 7.5 etc
    else if(line_buf[0]>='0' && line_buf[0]<='7' && line_buf[2]>='0'&& line_buf[2]<='9' && line_buf[1] == '.')
    {
        temp = (line_buf[0] - '0')*10;
        temp += line_buf[2] - '0';
        *adam_amp = temp / 5;

        if(temp % 5 == 0)
            return PARSE_OK;
        else
            return PARSE_ROUNDED;
    }
    else
        return PARSE_ERR;
}


int parse_amp2(char* line_buf, char* adam_amp, int length)
{

    char temp = 0;

    if(line_buf[3] == LINEBUF_EMPTY)
        return PARSE_EMPTY;
    //if(line_buf[3] == LINEBUF_TOOLONG)
    //    return PARSE_LONG;
    //Single digit input. single digit is 0,1,2,3,4,5,6,7
    else if(line_buf[0]>='0' && line_buf[0]<='7' && line_buf[1] == 255 && line_buf[2] == 255)
    {
        temp = (line_buf[0] - '0')*10;
        *adam_amp = temp/5;
        return PARSE_OK;
    }  
    //x.x or x.xxx format input eg. 0.0 0.5 0.555 1.5 1.0  7.0 7.5 7.555 etc
    else if(line_buf[0]>='0' && line_buf[0]<='7' && line_buf[2]>='0'&& line_buf[2]<='9' && line_buf[1] == '.')
    {
        temp = (line_buf[0] - '0')*10;
        temp += line_buf[2] - '0';
        *adam_amp = temp / 5;

        if((temp % 5 == 0) && (length == 3))
            return PARSE_OK;
        else
            return PARSE_ROUNDED;
    }
    //.x or .xxxxx format input  it will rounded to 0.5 if it's like 0.5555 or to 0 if it's like .333
    else if(line_buf[0] == '.')    
    {
        if(line_buf[1] >= '5')
            *adam_amp = 1;
        else
            *adam_amp = 0;
            
        if((line_buf[1] == '5' || line_buf[1] == '0')&& length == 2)
            return PARSE_OK;
        else
            return PARSE_ROUNDED; 
    }
    return PARSE_ERR;
}


static char char2hex_4bits(char c)
{
    if(c >= '0' && c <= '9')
        return c-'0';
    else if(c >= 'a' && c <= 'f')
        return c-'a'+10;
    else if(c >= 'A' && c <= 'F')
        return c-'A'+10;
    else
        return 255;
}

static char char2hex_3bits(char c)
{
    if(c>='0' && c<='7')
        return c-'0';
    else if(c == '8')
        return 0;
    else if(c == '9')
        return 1;
    else if(c >= 'a' && c <= 'f')
        return c-'a'+2;
    else if(c >= 'A' && c <= 'F')
        return c-'A'+2;
    else return 255;
}


int parse_spi_raw(char* buf, int length, char* cha1_pha, char* cha2_pha, char* cha1_amp, char* cha2_amp)
{
    if(length == 0)
        return PARSE_EMPTY;
    if(length < 4)
        return PARSE_SHORT;
    else if(length > 4)
        return PARSE_LONG;
    else
    {
        *cha2_amp = char2hex_4bits(buf[0]);
        *cha2_pha = char2hex_3bits(buf[1]);
        *cha1_amp = char2hex_4bits(buf[2]);
        *cha1_pha = char2hex_3bits(buf[3]);   
        return PARSE_OK;     
    }
}


char hex2char(char hex)
{
    if(hex < 10)
        return hex + '0';
    else
        return hex-10+'A';
}