/*
 *  LCD module "AQM0802A-RN-GBW" sample program
 *
 *  AQM0802A-RN-GBW is an I2C text LCD panel (Controller: Sitronix ST7032i)
 *  I bought this from AKIZUKI DENSHI TSUSHO CO.,LTD.
 *  http://akizukidenshi.com/catalog/g/gP-06669/ (Online shop page (Japanese))
 *  http://akizukidenshi.com/download/ds/sitronix/st7032.pdf (datasheet of the chip)
 *
 *  Original Arduino version was developed by
 *  Copyright (c) 2013 Masahiro WAKAYAMA at SWITCH SCIENCE
 *
 *  Copyright (c) 2013 Yoshihiro TSUBOI
 *
 *  Released under the MIT License: http://mbed.org/license/mit
 *
 *  revision 1.0  19-June-2013   a. 1st release
 *
 */
 
 #include "mbed.h"

I2C i2c(p28, p27); // sda, scl
const int AQCM0802_addr = 0x7C;

unsigned char mode;
unsigned char contrast = 0; // 0-63
unsigned char contrastFlag = false;
int CGcounter;
int FADEcounter;

void lcd_cmd(char x) {
  char data[2];
  data[0] = 0x00; // CO = 0,RS = 0
  data[1] = x;
  i2c.write(AQCM0802_addr, data, 2);
}

void lcd_contdata(char x) {
  char data[2];
  data[0] = 0xC0; //0b11000000 CO = 1, RS = 1
  data[1] = x;
  i2c.write(AQCM0802_addr, data, 2);
}

void lcd_lastdata(char x) {
  char data[2];
  data[0] = 0x40; //0b11000000 CO = 0, RS = 1
  data[1] = x;
  i2c.write(AQCM0802_addr, data, 2);
}

void lcd_printStr(const char *s) {
  while(*s) {
    if(*(s + 1)) {
      lcd_contdata(*s);
    } else {
      lcd_lastdata(*s);
    }
    s++;
  }
}

void lcd_printHex(unsigned char num) {
  lcd_contdata(num);
}

void lcd_init() {
  wait(0.04);
  // LCD initialize
  lcd_cmd(0x38); // function set
  lcd_cmd(0x39); // function set
  lcd_cmd(0x04); // EntryModeSet
  lcd_cmd(0x14); // interval osc
  lcd_cmd(0x70 | (contrast & 0xF)); // contrast Low
  lcd_cmd(0x5C | ((contrast >> 4) & 0x3)); // contast High/icon/power
  lcd_cmd(0x6C); // follower control
  wait(0.2);
  lcd_cmd(0x38); // function set
  lcd_cmd(0x0C); // Display On
  lcd_cmd(0x01); // Clear Display
  wait(0.2); // need additional wait to Clear Display
}

void lcd_setCursor(unsigned char x,unsigned char y) {
  lcd_cmd(0x80 | (y * 0x40 + x));
}

unsigned char cg[13 * 8] = {
/*
  0b00001111,0b00010000,0b00010000,0b00001110,0b00000001,0b00000001,0b00011110,0b00000000,
  0b00010001,0b00010001,0b00010001,0b00010101,0b00010101,0b00010101,0b00001010,0b00000000,
  0b00001110,0b00000100,0b00000100,0b00000100,0b00000100,0b00000100,0b00001110,0b00000000,
  0b00011111,0b00000100,0b00000100,0b00000100,0b00000100,0b00000100,0b00000100,0b00000000,
  0b00001110,0b00010001,0b00010000,0b00010000,0b00010000,0b00010001,0b00001110,0b00000000,
  0b00010001,0b00010001,0b00010001,0b00011111,0b00010001,0b00010001,0b00010001,0b00000000,
*/
  0x0F,0x10,0x10,0x0E,0x01,0x01,0x1E,0x00, // S
  0x11,0x11,0x11,0x15,0x15,0x15,0x0A,0x00, // W
  0x0E,0x04,0x04,0x04,0x04,0x04,0x0E,0x00, // I
  0x1F,0x04,0x04,0x04,0x04,0x04,0x04,0x00, // T
  0x0E,0x11,0x10,0x10,0x10,0x11,0x0E,0x00, // C
  0x11,0x11,0x11,0x1F,0x11,0x11,0x11,0x00, // H
/*
  0b00001111,0b00010000,0b00010000,0b00001110,0b00000001,0b00000001,0b00011110,0b00000000,
  0b00001110,0b00010001,0b00010000,0b00010000,0b00010000,0b00010001,0b00001110,0b00000000,
  0b00001110,0b00000100,0b00000100,0b00000100,0b00000100,0b00000100,0b00001110,0b00000000,
  0b00011111,0b00010000,0b00010000,0b00011110,0b00010000,0b00010000,0b00011111,0b00000000,
  0b00010001,0b00010001,0b00011001,0b00010101,0b00010011,0b00010001,0b00010001,0b00000000,
  0b00001110,0b00010001,0b00010000,0b00010000,0b00010000,0b00010001,0b00001110,0b00000000,
  0b00011111,0b00010000,0b00010000,0b00011110,0b00010000,0b00010000,0b00011111,0b00000000,
*/
  0x0F,0x10,0x10,0x0E,0x01,0x01,0x1E,0x00, // S
  0x0E,0x11,0x10,0x10,0x10,0x11,0x0E,0x00, // C
  0x0E,0x04,0x04,0x04,0x04,0x04,0x0E,0x00, // I
  0x1F,0x10,0x10,0x1E,0x10,0x10,0x1F,0x00, // E
  0x11,0x11,0x19,0x15,0x13,0x11,0x11,0x00, // N
  0x0E,0x11,0x10,0x10,0x10,0x11,0x0E,0x00, // C
  0x1F,0x10,0x10,0x1E,0x10,0x10,0x1F,0x00  // E
};

void setCG(int src,int dst,int len) {
  lcd_cmd(0x38);
  lcd_cmd(0x40 + dst);
  if (src >= 0) {
    for (int i = 0;i < len;i++) lcd_printHex(cg[src + i]);
  } else {
    for (int i = 0;i < len;i++) lcd_printHex(0);
  }
}

void setContrast(unsigned char c) {
  lcd_cmd(0x39);
  lcd_cmd(0x70 | (c & 0x0f)); // contrast Low
  lcd_cmd(0x5C | ((c >> 4) & 0x03)); // contast High/icon/power
  lcd_cmd(0x38);
}

int main() {
    lcd_init();
    while(1) {
      switch(mode) {
        case 0:  // init
          lcd_setCursor(0, 0);
          for (int i = 0;i < 6;i++) lcd_printHex(i);
          lcd_setCursor(1, 1);
          lcd_printStr("       ");
          setCG(-1,0,6 * 8);
          contrast = 35;
          setContrast(contrast);
          CGcounter = 0;
          mode++;
          break;
        case 1:  // SWITCH
          if(CGcounter <= (6 * 8)) {
            setCG((CGcounter / 8) * 8,((CGcounter / 8) * 8) + 7 - (CGcounter % 8),CGcounter % 8);
            CGcounter++;
          } else {
            lcd_setCursor(0, 0);
            lcd_printStr("SWITCH");
            setCG(-1,0,3 * 8);
            lcd_setCursor(1, 1);
            for (int i = 0;i < 3;i++) lcd_printHex(i);
            CGcounter = 0;
            mode++;
          }
          break;
        case 2:  // SCI
          if(CGcounter <= (3 * 8)) {
            setCG(((CGcounter / 8) + 6) * 8,((CGcounter / 8) * 8) + 7 - (CGcounter % 8),CGcounter % 8);
            CGcounter++;
          } else {
            lcd_setCursor(1, 1);
            lcd_printStr("SCI");
            setCG(-1,0,4 * 8);
            lcd_setCursor(4, 1);
            for (int i = 0;i < 4;i++) lcd_printHex(i);
            CGcounter = 0;
            mode++;
          }
          break;
        case 3:  // ENCE
          if(CGcounter <= (4 * 8)) {
            setCG(((CGcounter / 8) + 9) * 8,((CGcounter / 8) * 8) + 7 - (CGcounter % 8),CGcounter % 8);
            CGcounter++;
          } else {
            lcd_setCursor(4, 1);
            lcd_printStr("ENCE");
            FADEcounter = 0;
            mode++;
          }
          break;
        case 4:
          if (contrastFlag == false) {
            if (++contrast >= 54) contrastFlag = true;
          } else {
            if (--contrast <= 17) {
              contrastFlag = false;
              if(++FADEcounter >= 2) {
                mode = 0;
              }
            }
          }
          setContrast(contrast);
          break;
      }
      wait(0.05);
    }
}
