 /* mbed library for 128*160 pixel display TFT based on ST7735 LCD Controller
 * ST7735 specific routines (initialization, window addressing, pixel output) 
 * Copyright (c) 2011 Jonne Valola
 *
 * WARNING !! WORK IN PROGRESS !!!
 *
 * Graphics routines and SPI routines derived work used with permission from:
 * mbed library for 240*320 pixel display TFT based on HX8347D LCD Controller
 * Copyright (c) 2011 Peter Drescher - DC2PD
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


#include "ST7735_TFT.h"
#include "mbed.h"

#define BPP         16                  // Bits per pixel                

ST7735_TFT::ST7735_TFT(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName rs, PinName reset, const char *name)
        : _spi(mosi, miso, sclk), _cs(cs), _rs(rs), _reset(reset),GraphicsDisplay(name) {
    tft_reset();
    orientation = 2;
    char_x = 0;
}


int ST7735_TFT::width() {
    if (orientation == 0 || orientation == 2) return 128;
    else return 160; 
}


int ST7735_TFT::height() {
    if (orientation == 0 || orientation == 2) return 160;
    else return 128; 
}


void ST7735_TFT::set_orientation(unsigned int o) {
    orientation = o;
    switch (orientation) {
        case 0:
            wr_reg(ST7735_MADCTL, 0x0008);
            break;
        case 1:
            wr_reg(ST7735_MADCTL, 0x0068);
            break;
        case 2:
            wr_reg(ST7735_MADCTL, 0x00C8);
            break;
        case 3:
            wr_reg(ST7735_MADCTL, 0x00A8);
            break;
    }
}



void ST7735_TFT::wr_cmd(int cmd) {
    _rs = 0; // rs low, cs low for transmitting command
    _cs = 0;
    _spi.write(cmd);
    _cs = 1;
}



void ST7735_TFT::wr_dat(int dat) {
    _rs = 1; // rs high, cs low for transmitting data
    _cs = 0;                         
    _spi.write(dat);                                                           
    _cs = 1;
}



void ST7735_TFT::wr_dat_start(void) {
    _rs = 1; //  rs high, cs low for transmitting data
    _cs = 0;
}



void ST7735_TFT::wr_dat_stop (void) {
    _cs = 1;
}



void ST7735_TFT::wr_dat_only (unsigned short dat) {  
    _spi.write(dat);                                            
}


unsigned short ST7735_TFT::rd_dat (void) {
    unsigned short val = 0;
    _cs = 0;
    _spi.write(0);                                /* Dummy read 1                 */
    val   = _spi.write(0);                        /* Read D8..D15                 */
    val <<= 8;
    val  |= _spi.write(0);                        /* Read D0..D7                  */
    _cs = 1;
    return (val);
}

void ST7735_TFT::wr_reg (unsigned char reg, unsigned short val) {

    wr_cmd(reg);
    wr_dat(val);
}


unsigned short ST7735_TFT::rd_reg (unsigned char reg) {

    wr_cmd(reg);
    return(rd_dat());
}

void ST7735_TFT::read_area(unsigned int x, unsigned int y, unsigned int w, unsigned int h,unsigned char *buffer) {
    // BEWARE !
    // DOES NOT WORK CORRECTLY YET !!!
    int val;
    window(x,y,w,h);
    wr_cmd(ST7735_RAMRD);  // write to RAM
    _cs = 0;
    _rs = 1;
    _spi.write(0);                                /* Dummy read 1                 */
    
    val   = _spi.write(0);                        /* Read D8..D15                 */
    val <<= 8;
    val  |= _spi.write(0);                        /* Read D0..D7                  */
    _cs = 1;
}

int ST7735_TFT::getpixel(unsigned int x, unsigned int y) {
    // BEWARE !
    // DOES NOT WORK CORRECTLY YET !!!
    int val;
    _spi.format(8,3);
    wr_cmd(ST7735_CASET);  // column addr set
    wr_dat(0x00);
    wr_dat(x+2);   // XSTART 
    wr_dat(0x00);
    wr_dat(x+2+2);   // XEND

    wr_cmd(ST7735_RASET);  // row addr set
    wr_dat(0x00);
    wr_dat(y+1);    // YSTART
    wr_dat(0x00);
    wr_dat(y+1+1);    // YEND
    
    _rs = 0; // rs low, cs low for transmitting command
    _cs = 0;
    _spi.write(0x2E);
    _rs = 1;
    _spi.write(0x00);                                /* Dummy read 1                 */
    
    val   = _spi.write(0x00);                        /* Read D8..D15                 */
    val <<= 8;
    val  |= _spi.write(0x00);                        /* Read D0..D7                  */

    _cs = 1;
    return val;
}

void ST7735_TFT::tft_reset() {
    static unsigned short driverCode;
    
    // init SPI
    _spi.format(8,3);                 // 8 bit spi mode 3
    _spi.frequency(16000000);         // 16Mhz SPI clock ... 15Mhz is maximum for display, but it seems to work
                                      // RY: Changed to 4
    // reset exactly like in Arduino version
    _cs = 0;
    _reset = 1;                       // reset
    wait_ms(500);
    _reset = 0;                       // reset
    wait_ms(500);
    _reset = 1;                       // reset
    wait_ms(500);
    
    /* Start Initial Sequence ----------------------------------------------------*/
    wr_cmd(ST7735_SWRESET);                         /* SW Reset                       */
    wait_ms(150);
    wr_cmd(ST7735_SLPOUT);                         /* Out of sleepmode               */
    wait_ms(500);
    
    wr_cmd(ST7735_FRMCTR1);                         /* Frame rate in normal mode            */
    wr_dat(0x01);                              
    wr_dat(0x2C);
    wr_dat(0x2D);
    
    wr_cmd(ST7735_FRMCTR2);                         /* Frame rate in idle mode            */
    wr_dat(0x01);                              
    wr_dat(0x2C);
    wr_dat(0x2D);

    wr_cmd(ST7735_FRMCTR3);                         /* Frame rate in partial mode            */
    wr_dat(0x01);                              
    wr_dat(0x2C);
    wr_dat(0x2D);
    wr_dat(0x01);   // inversion mode settings                              
    wr_dat(0x2C);
    wr_dat(0x2D);
    
    wr_cmd(ST7735_INVCTR);   // Inverted mode off
    wr_dat(0x07);   

    wr_cmd(ST7735_PWCTR1);   // POWER CONTROL 1   
    wr_dat(0xA2); 
    wr_dat(0x02);           // -4.6V
    wr_dat(0x84);           // AUTO mode 
    
    wr_cmd(ST7735_PWCTR2);   // POWER CONTROL 2   
    wr_dat(0xC5);            // VGH25 = 2.4C VGSEL =-10 VGH = 3*AVDD
    
    wr_cmd(ST7735_PWCTR3);  // POWER CONTROL 3   
    wr_dat(0x0A);           // Opamp current small
    wr_dat(0x00);           // Boost freq

    wr_cmd(ST7735_PWCTR4);   // POWER CONTROL 4   
    wr_dat(0x8A);            // BCLK/2, Opamp current small / medium low 
    wr_dat(0x2A);            // 
  
    wr_cmd(ST7735_PWCTR5);   // POWER CONTROL 5   
    wr_dat(0x8A);            // BCLK/2, Opamp current small / medium low 
    wr_dat(0xEE);            //
    
    wr_cmd(ST7735_VMCTR1);   // POWER CONTROL 6   
    wr_dat(0x0E);            // 

    wr_cmd(ST7735_INVOFF);   // INVOFF
    
    wr_cmd(ST7735_MADCTL);   // ORIENTATION   
    wr_dat(0xC8);   // 
    
    wr_cmd(ST7735_COLMOD);   // COLOR MODE   
    wr_dat(0x05);            //      
    
    wr_cmd(ST7735_CASET);   // COLUMN ADDR SET   
    wr_dat(0x00);   //
    wr_dat(0x00);   // xstart = 0
    wr_dat(0x00);   //
    wr_dat(0x7F);   // xend = 127
    
    wr_cmd(ST7735_RASET);   // ROW ADDR SET   
    wr_dat(0x00);   //
    wr_dat(0x00);   // ystart = 0
    wr_dat(0x00);   //
    wr_dat(0x9F);   // yend = 159            
    
    /* Gamma settings  -----------------------------------------------------------*/

  wr_cmd(0xE0); // GMCTRP1
  wr_dat(0x02);
  wr_dat(0x1c);
  wr_dat(0x07);
  wr_dat(0x12);
  wr_dat(0x37);
  wr_dat(0x32);
  wr_dat(0x29);
  wr_dat(0x2d);
  wr_dat(0x29);
  wr_dat(0x25);
  wr_dat(0x2B);
  wr_dat(0x39);
  wr_dat(0x00);
  wr_dat(0x01);
  wr_dat(0x03);
  wr_dat(0x10);
  wr_cmd(0xE1); // GMCTRN1
  wr_dat(0x03); 
  wr_dat(0x1d); 
  wr_dat(0x07); 
  wr_dat(0x06); 
  wr_dat(0x2E); 
  wr_dat(0x2C); 
  wr_dat(0x29); 
  wr_dat(0x2D); 
  wr_dat(0x2E); 
  wr_dat(0x2E); 
  wr_dat(0x37); 
  wr_dat(0x3F); 
  wr_dat(0x00); 
  wr_dat(0x00); 
  wr_dat(0x02); 
  wr_dat(0x10); 

  wr_cmd(ST7735_NORON);  // normal display on
  wait_ms(10);
  wr_cmd(ST7735_DISPON); // display ON
  wait_ms(100);  
/*  
  
  switch (orientation) {
        case 0:
            wr_reg(0xC8, 0x0008);
            break;
        case 1:
            wr_reg(0xC8, 0x0068);
            break;
        case 2:
            wr_reg(0xC8, 0x00C8);
            break;
        case 3:
            wr_reg(0xC8, 0x00A8);
            break;
    }
*/
    WindowMax ();
}


void ST7735_TFT::pixel(int x, int y, int color) {
  // setup for data
    
  if ((x >= width()) || (y >= height())) return;
  window(x,y,x,y);
  wr_dat_start();
  wr_dat(color>>8);
  wr_dat(color);
  wr_dat_stop();
}

void ST7735_TFT::window (unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1) {
  wr_cmd(ST7735_CASET);  // column addr set
  wr_dat(0x00); //RY Changed this from the GreenTab to RedTab
  wr_dat(x0);   // XSTART 
  wr_dat(0x00); //RY Changed this from the GreenTab to RedTab
  wr_dat(x1);   // XEND

  wr_cmd(ST7735_RASET);  // row addr set
  wr_dat(0x00); //RY Changed this from the GreenTab to RedTab
  wr_dat(y0);    // YSTART
  wr_dat(0x00); //RY Changed this from the GreenTab to RedTab
  wr_dat(y1);    // YEND

  wr_cmd(ST7735_RAMWR);  // write to RAM
}


void ST7735_TFT::WindowMax (void) {
    window (0, 0, width(),  height());
}

void ST7735_TFT::cls (void) {
    fillrect(0, 0, width(),  height(), _background);
}

void ST7735_TFT::circle(int x0, int y0, int r, int color) {

    int draw_x0, draw_y0;
    int draw_x1, draw_y1;
    int draw_x2, draw_y2;
    int draw_x3, draw_y3;
    int draw_x4, draw_y4;
    int draw_x5, draw_y5;
    int draw_x6, draw_y6;
    int draw_x7, draw_y7;
    int xx, yy;
    int di;
    WindowMax();
    if (r == 0) {       /* no radius */
        return;
    }

    draw_x0 = draw_x1 = x0;
    draw_y0 = draw_y1 = y0 + r;
    if (draw_y0 < height()) {
        pixel(draw_x0, draw_y0, color);     /* 90 degree */
    }

    draw_x2 = draw_x3 = x0;
    draw_y2 = draw_y3 = y0 - r;
    if (draw_y2 >= 0) {
        pixel(draw_x2, draw_y2, color);    /* 270 degree */
    }

    draw_x4 = draw_x6 = x0 + r;
    draw_y4 = draw_y6 = y0;
    if (draw_x4 < width()) {
        pixel(draw_x4, draw_y4, color);     /* 0 degree */
    }

    draw_x5 = draw_x7 = x0 - r;
    draw_y5 = draw_y7 = y0;
    if (draw_x5>=0) {
        pixel(draw_x5, draw_y5, color);     /* 180 degree */
    }

    if (r == 1) {
        return;
    }

    di = 3 - 2*r;
    xx = 0;
    yy = r;
    while (xx < yy) {

        if (di < 0) {
            di += 4*xx + 6;
        } else {
            di += 4*(xx - yy) + 10;
            yy--;
            draw_y0--;
            draw_y1--;
            draw_y2++;
            draw_y3++;
            draw_x4--;
            draw_x5++;
            draw_x6--;
            draw_x7++;
        }
        xx++;
        draw_x0++;
        draw_x1--;
        draw_x2++;
        draw_x3--;
        draw_y4++;
        draw_y5++;
        draw_y6--;
        draw_y7--;

        if ( (draw_x0 <= width()) && (draw_y0>=0) ) {
            pixel(draw_x0, draw_y0, color);
        }

        if ( (draw_x1 >= 0) && (draw_y1 >= 0) ) {
            pixel(draw_x1, draw_y1, color);
        }

        if ( (draw_x2 <= width()) && (draw_y2 <= height()) ) {
            pixel(draw_x2, draw_y2, color);
        }

        if ( (draw_x3 >=0 ) && (draw_y3 <= height()) ) {
            pixel(draw_x3, draw_y3, color);
        }

        if ( (draw_x4 <= width()) && (draw_y4 >= 0) ) {
            pixel(draw_x4, draw_y4, color);
        }

        if ( (draw_x5 >= 0) && (draw_y5 >= 0) ) {
            pixel(draw_x5, draw_y5, color);
        }
        if ( (draw_x6 <=width()) && (draw_y6 <= height()) ) {
            pixel(draw_x6, draw_y6, color);
        }
        if ( (draw_x7 >= 0) && (draw_y7 <= height()) ) {
            pixel(draw_x7, draw_y7, color);
        }
    }
    return;
}

void ST7735_TFT::fillcircle(int x, int y, int r, int color) {
    int i;
    for (i = 0; i <= r; i++)
        circle(x,y,i,color);
}

void ST7735_TFT::hline(int x, int y, int w, int color) {
  int hi = color >> 8, lo = color;

  // Rudimentary clipping
  if((x >= width()) || (y >= height())) return;
  if((x+w-1) >= width())  w = width()-x;
  window(x, y, x+w-1, y);

  while (w--) {
    wr_dat(hi);
    wr_dat(lo);
  }
    wr_dat_stop();
    return;
}

void ST7735_TFT::vline(int x, int y, int h, int color) {
  int hi = color >> 8, lo = color;

  // Rudimentary clipping
  if((x >= width()) || (y >= height())) return;
  if((y+h-1) >= height()) h = height()-y;
  window(x, y, x, y+h-1);

  while (h--) {
    wr_dat(hi);
    wr_dat(lo);
  }

  wr_dat_stop();
    return;
}

void ST7735_TFT::line(int x0, int y0, int x1, int y1, int color) {
    WindowMax();
    int   dx = 0, dy = 0;
    int   dx_sym = 0, dy_sym = 0;
    int   dx_x2 = 0, dy_x2 = 0;
    int   di = 0;

    dx = x1-x0;
    dy = y1-y0;

    if (dx == 0) {        /* vertical line */
        if (y1 > y0) vline(x0,y0,y1,color);
        else vline(x0,y1,y0,color);
        return;
    }

    if (dx > 0) {
        dx_sym = 1;
    } else {
        dx_sym = -1;
    }
    if (dy == 0) {        /* horizontal line */
        if (x1 > x0) hline(x0,x1,y0,color);
        else  hline(x1,x0,y0,color);
        return;
    }

    if (dy > 0) {
        dy_sym = 1;
    } else {
        dy_sym = -1;
    }

    dx = dx_sym*dx;
    dy = dy_sym*dy;

    dx_x2 = dx*2;
    dy_x2 = dy*2;

    if (dx >= dy) {
        di = dy_x2 - dx;
        while (x0 != x1) {

            pixel(x0, y0, color);
            x0 += dx_sym;
            if (di<0) {
                di += dy_x2;
            } else {
                di += dy_x2 - dx_x2;
                y0 += dy_sym;
            }
        }
        pixel(x0, y0, color);
    } else {
        di = dx_x2 - dy;
        while (y0 != y1) {
            pixel(x0, y0, color);
            y0 += dy_sym;
            if (di < 0) {
                di += dx_x2;
            } else {
                di += dx_x2 - dy_x2;
                x0 += dx_sym;
            }
        }
        pixel(x0, y0, color);
    }
    return;
}

void ST7735_TFT::rect(int x0, int y0, int x1, int y1, int color) {

    if (x1 > x0) hline(x0,x1,y0,color);
    else  hline(x1,x0,y0,color);

    if (y1 > y0) vline(x0,y0,y1,color);
    else vline(x0,y1,y0,color);

    if (x1 > x0) hline(x0,x1,y1,color);
    else  hline(x1,x0,y1,color);

    if (y1 > y0) vline(x1,y0,y1,color);
    else vline(x1,y1,y0,color);

    return;
}

void ST7735_TFT::fillrect(int x, int y, int w, int h, int color) {
  int hi = color >> 8, lo = color;

  // rudimentary clipping (drawChar w/big text requires this)
  if((x >= width()) || (y >= height())) return;
  if((x + w - 1) >= width())  w = width()  - x;
  if((y + h - 1) >= height()) h = height() - y;

  window(x, y, x+w-1, y+h-1);
  for(y=h; y>0; y--) {
    for(x=w; x>0; x--) {
      wr_dat(hi);
      wr_dat(lo);
    }
  }
  wr_dat_stop();
}

void ST7735_TFT::locate(int x, int y) {
    char_x = x;
    char_y = y;
}

int ST7735_TFT::columns() {
    return width() / font[1];
}

int ST7735_TFT::rows() {
    return height() / font[2];
}

int ST7735_TFT::_putc(int ch) {
  if((ch == 10) || (ch == 13) || (ch == 27)){
    char_y++; char_x=0;
    if(char_y>15){
      char_y = 0;
    }
    drawstring(0,char_y,"                     ");
    return 1;
  }
  character(char_x*6,char_y*10,ch, 1);
  char_x++;
  if(char_x>20){
    char_x = 20;
    character(char_x*6,char_y*10,'*', 1);
  }
  return 1;
}


void ST7735_TFT::character(int x, int y, int c, int size) {
  int line; // vertical column of pixels of character in font
  int i, j;
  if((x >= width())            || // Clip right
     (y >= height())           || // Clip bottom
     ((x + 6 * size - 1) < 0) || // Clip left
     ((y + 8 * size - 1) < 0))   // Clip top
    return;

  for (i=0; i<6; i++ ) {
    if (i == 5)
      line = 0x0;
    else
      line = Font[(c*5)+i];
    for (j = 0; j<8; j++) {
      if (line & 0x1) {
        if (size == 1) // default size
          pixel(x+i, y+j,_foreground);
        else {  // big size
          fillrect(x+(i*size), y+(j*size), size, size, _foreground);
        }
      } else if (_background != _foreground) {
        if (size == 1) // default size
          pixel(x+i, y+j, _background);
        else {  // big size
          fillrect(x+i*size, y+j*size, size, size, _background);
        }
      }
      line >>= 1;
    }
  }
}

void ST7735_TFT::drawstring(int x, int y, char *pt) {
  int count = 0;
  if(y>15) return;
  while(*pt){
    character(x*6, y*10, *pt, 1);
    pt++;
    x = x+1;
    if(x>20) return;  // number of characters printed
    count++;
  }
  return;  // number of characters printed
}

void ST7735_TFT::Bitmap( short x,  short y,  short w, short h, const unsigned short *image) {
  short skipC = 0;                   // non-zero if columns need to be skipped due to clipping
  short originalWidth = w;           // save this value; even if not all columns fit on the screen, the image is still this width in ROM
  int i = w*(h - 1);

  if((x >= width()) || ((y - h + 1) >= height()) || ((x + w) <= 0) || (y < 0)){
    return;                             // image is totally off the screen, do nothing
  }
  if((w > width()) || (h > height())){    // image is too wide for the screen, do nothing
    //***This isn't necessarily a fatal error, but it makes the
    //following logic much more complicated, since you can have
    //an image that exceeds multiple boundaries and needs to be
    //clipped on more than one side.
    return;
  }
  if((x + w - 1) >= width()){            // image exceeds right of screen
    skipC = (x + w) - width();           // skip cut off columns
    w = width() - x;
  }
  if((y - h + 1) < 0){                  // image exceeds top of screen
    i = i - (h - y - 1)*originalWidth;  // skip the last cut off rows
    h = y + 1;
  }
  if(x < 0){                            // image exceeds left of screen
    w = w + x;
    skipC = -1*x;                       // skip cut off columns
    i = i - x;                          // skip the first cut off columns
    x = 0;
  }
  if(y >= height()){                     // image exceeds bottom of screen
    h = h - (y - height() + 1);
    y = height() - 1;
  }

  window(x, y-h+1, x+w-1, y);

  for(y=0; y<h; y=y+1){
    for(x=0; x<w; x=x+1){
                                        // send the top 8 bits
      wr_dat((unsigned char)(image[i] >> 8));
                                        // send the bottom 8 bits
      wr_dat((unsigned char)(image[i]));
      i = i + 1;                        // go to the next pixel
    }
    i = i + skipC;
    i = i - 2*originalWidth;
  }

  wr_dat_stop();
}
