/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "nRF51822n.h"
#include "MPU6050.h"

nRF51822n  nrf;                 /* BLE radio driver */

Serial     pc(USBTX, USBRX);
MPU6050 mpu(P0_22, P0_20);

/* Battery Level Service */
uint8_t            batt      = 88; /* Battery level */

GattService        battService (GattService::UUID_BATTERY_SERVICE);
GattCharacteristic battLevel   (GattCharacteristic::UUID_BATTERY_LEVEL_CHAR,
                                1,
                                1,
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY |
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
                                
GattService        tempService (GattService::UUID_HEALTH_THERMOMETER_SERVICE);
GattCharacteristic  tempLevel  (GattCharacteristic::UUID_TEMPERATURE_MEASUREMENT_CHAR,
                                2,
                                3,
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);
GattCharacteristic tempLocation (
                                GattCharacteristic::UUID_BODY_SENSOR_LOCATION_CHAR,
                                1,
                                1,
                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);

/* Heart Rate Service */
/* Service:  https://developer.bluetooth.org/gatt/services/Pages/ServiceViewer.aspx?u=org.bluetooth.service.heart_rate.xml */
/* HRM Char: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.heart_rate_measurement.xml */
/* Location: https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.body_sensor_location.xml */
GattService        hrmService    (GattService::UUID_HEART_RATE_SERVICE);
GattCharacteristic hrmRate       (
    GattCharacteristic::UUID_HEART_RATE_MEASUREMENT_CHAR,
    2,
    3,
    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);
GattCharacteristic hrmLocation   (
    GattCharacteristic::UUID_BODY_SENSOR_LOCATION_CHAR,
    1,
    1,
    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);

/* Device Information service */
uint8_t deviceName[] = {'Y', 'o', 'u','Y', 'o', 'u'};
GattService        deviceInformationService (
    GattService::UUID_DEVICE_INFORMATION_SERVICE);
GattCharacteristic deviceManufacturer (
    GattCharacteristic::UUID_MANUFACTURER_NAME_STRING_CHAR,
    sizeof(deviceName),
    sizeof(deviceName),
    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
uint8_t version[]={'Y', 'o', 'u','Y', 'o', 'u',' ','1','.','0'};
GattCharacteristic firmwareVersion (
    GattCharacteristic::UUID_FIRMWARE_REVISION_STRING_CHAR,
    sizeof(version),
    sizeof(version),
    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);

/* Advertising data and parameters */
GapAdvertisingData advData;
GapAdvertisingData scanResponse;
GapAdvertisingParams advParams (GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
uint16_t           uuid16_list[] = {GattService::UUID_BATTERY_SERVICE,
                                    GattService::UUID_DEVICE_INFORMATION_SERVICE,
                                    GattService::UUID_HEART_RATE_SERVICE,
                                    GattService::UUID_HEALTH_THERMOMETER_SERVICE};

/**************************************************************************/
/*!
    @brief  This custom class can be used to override any GapEvents
            that you are interested in handling on an application level.
*/
/**************************************************************************/
class GapEventHandler : public GapEvents
{
    virtual void onTimeout(void)
    {
        pc.printf("Advertising Timeout!\r\n");
        // Restart the advertising process with a much slower interval,
        // only start advertising again after a button press, etc.
    }

    virtual void onConnected(void)
    {
        pc.printf("Connected!\r\n");
    }

    virtual void onDisconnected(void)
    {
        pc.printf("Disconnected!\r\n");
        pc.printf("Restarting the advertising process\r\n");
        nrf.getGap().startAdvertising(advParams);
    }
};

/**************************************************************************/
/*!
    @brief  This custom class can be used to override any GattServerEvents
            that you are interested in handling on an application level.
*/
/**************************************************************************/
class GattServerEventHandler : public GattServerEvents
{
    //virtual void onDataSent(uint16_t charHandle) {}
    //virtual void onDataWritten(uint16_t charHandle) {}

    virtual void onUpdatesEnabled(uint16_t charHandle)
    {
        if (charHandle == hrmRate.getHandle()) {
            pc.printf("Heart rate notify enabled\r\n");
        }
        if (charHandle == tempLevel.getHandle()) {
            pc.printf("Temperature notify enabled\r\n");
        }
    }

    virtual void onUpdatesDisabled(uint16_t charHandle)
    {
        if (charHandle == hrmRate.getHandle()) {
            pc.printf("Heart rate notify disabled\r\n");
        }
         if (charHandle == tempLevel.getHandle()) {
            pc.printf("Temperature notify disabled\r\n");
        }
    }

    //virtual void onConfirmationReceived(uint16_t charHandle) {}
};

/**************************************************************************/
/*!
    @brief  Program entry point
*/
/**************************************************************************/
int main(void)
{
    float accdata[3];
    uint16_t time=0;
    wait(1);
    pc.baud(38400);
    
    /* Setup the local GAP/GATT event handlers */
    nrf.getGap().setEventHandler(new GapEventHandler());
    nrf.getGattServer().setEventHandler(new GattServerEventHandler());

    /* Initialise the nRF51822 */
    pc.printf("Initialising the nRF51822...\r\n");
    nrf.init();

    /* Make sure we get a clean start */
    nrf.reset();

    /* Add BLE-Only flag and complete service list to the advertising data */
    advData.addFlags(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    advData.addData(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS,
                    (uint8_t *)uuid16_list, sizeof(uuid16_list));
    advData.addAppearance(GapAdvertisingData::HEART_RATE_SENSOR_HEART_RATE_BELT);
    nrf.getGap().setAdvertisingData(advData, scanResponse);

    /* Add the Battery Level service */
    battService.addCharacteristic(battLevel);
    nrf.getGattServer().addService(battService);

    /* Add the Device Information service */
    deviceInformationService.addCharacteristic(deviceManufacturer);
    deviceInformationService.addCharacteristic(firmwareVersion);
    nrf.getGattServer().addService(deviceInformationService);

    /* Add the Heart Rate service */
    hrmService.addCharacteristic(hrmRate);
    hrmService.addCharacteristic(hrmLocation);
    nrf.getGattServer().addService(hrmService);
    
    tempService.addCharacteristic(tempLevel);
    tempService.addCharacteristic(tempLocation);
    nrf.getGattServer().addService(tempService);

    /* Start advertising (make sure you've added all your data first) */
    nrf.getGap().startAdvertising(advParams);
    
    
    mpu.setFrequency(100000);
    wait_ms(10); 
    if (mpu.testConnection())
    {
        pc.printf("MPU connection succeeded!\r\n");
    }
    else
    {
        pc.printf("MPU connection failed!\r\n"); //Todo: If connection fails, retry a couple times. Try resetting MPU (this would need another wire?)
    }
    wait(2);
    mpu.setSleepMode(false);
    mpu.setBW(MPU6050_BW_20); 
    mpu.setGyroRange(MPU6050_GYRO_RANGE_500);
    mpu.setAcceleroRange(MPU6050_ACCELERO_RANGE_2G);
    

    /* Wait until we are connected to a central device before updating
     * anything */
    pc.printf("Waiting for a connection ...\r\n");
    while (!nrf.getGap().state.connected) {
    }
    wait(3);
    
    pc.printf("Connected!\r\n");

    /* Now that we're live, update the battery level characteristic, and */
    /* change the device manufacturer characteristic to 'mbed' */
    nrf.getGattServer().updateValue(battLevel.getHandle(), (uint8_t *)&batt,
                                    sizeof(batt));
    nrf.getGattServer().updateValue(deviceManufacturer.getHandle(),
                                    deviceName,
                                    sizeof(deviceName));
    nrf.getGattServer().updateValue(firmwareVersion.getHandle(),
                                    version,
                                    sizeof(version));
                                    
    /* Set the heart rate monitor location (one time only) */
    /* See --> https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.body_sensor_location.xml */
    uint8_t location = 0x03; /* Finger */
    uint8_t hrmCounter = 100;
    nrf.getGattServer().updateValue(hrmLocation.getHandle(),
                                    (uint8_t *)&location,
                                    sizeof(location));
    nrf.getGattServer().updateValue(tempLocation.getHandle(),
                                    (uint8_t *)&location,
                                    sizeof(location));     

    
    for (;; ) 
    {
        wait(1);

        /* Update battery level 
        batt++;
        if (batt > 100) {
            batt = 72;
        }
        nrf.getGattServer().updateValue(battLevel.getHandle(),
                                        (uint8_t *)&batt,
                                        sizeof(batt));  */

      /* Update the HRM measurement */
      /* First byte = 8-bit values, no extra info, Second byte = uint8_t HRM value */
      /* See --> https://developer.bluetooth.org/gatt/characteristics/Pages/CharacteristicViewer.aspx?u=org.bluetooth.characteristic.heart_rate_measurement.xml */
        hrmCounter++;
        if (hrmCounter == 175) {
            hrmCounter = 100;
        }
        uint8_t bpm[2] = {0x00, hrmCounter};
        nrf.getGattServer().updateValue(hrmRate.getHandle(), bpm, sizeof(bpm));
        
        float f =mpu.getTemp();
        int t = f;
        uint8_t temper[2];
        temper[0]=t>>8;
        temper[1]=t;
        nrf.getGattServer().updateValue(tempLevel.getHandle(), temper, sizeof(temper));
        
        pc.printf("HRM:");
        pc.printf("%d",hrmCounter);
        pc.printf("    Temperature:");
        pc.printf("%.2f",f);
        pc.printf("    ");
        mpu.getAccelero(accdata);
        time++;
        pc.printf("X:%.2f  Y:%.2f  Z:%.2f  Time:%ds\r\n",accdata[0], accdata[1], accdata[2],time);     
    }
}