/*

Copyright (c) 2012-2014 RedBearLab

Permission is hereby granted, free of charge, to any person obtaining a copy of this software
and associated documentation files (the "Software"), to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
/*
 *    The application works with the Science Journal App.
 *    Sensors Value from the BLE sensors tag to send
 *    to the Science Journal App.
 *    Characteristics Value received from App will print on Terminal.
 *    Sensors value will display the Science Jorunal App.
 */

#include <stdio.h>
#include "mbed.h"
#include "ble/BLE.h"
#include "goosci_utility.h"
#include "config_change.h"
#include "BMP180.h"

#define SCIENCE_UUID_SERVICE                    0x0001 /**< The UUID of the Nordic UART Service. */
#define SCIENCE_UUID_VALUE_CHARACTERISTIC       0x0003 /**< The UUID of the TX Characteristic. */
#define SCIENCE_UUID_CONFIG_CHARACTERISTIC      0x0010 /**< The UUID of the RX Characteristic. */

#define TXRX_BUF_LEN                     20
#define INTERNAL                         200

BLE  ble;
Timer mytimer;
Serial pc(P0_23, P0_25);

//LED setup
DigitalOut LED_R(P0_20);
DigitalOut LED_G(P0_19);

//Sensors setup
I2C i2c(P0_17, P0_18);
BMP180 bmp180(&i2c);

//Light sensor
//Analog setup
AnalogIn myanalog(P0_1);

//LIS3DH set up
SPI spi_master(P0_6,P0_5,P0_7); //mosi miso sclk 
DigitalOut cs(P0_4);


uint16_t x_a,y_a,z_a;

uint8_t LIS3DH_SPI_RD(uint8_t addr)
{
     uint8_t  temp;
     cs = 0;
     wait_us(10);
     spi_master.write(addr);      
     temp=spi_master.write(0xff);
     wait_us(10);
     cs = 1;
     return temp;
}
void LIS3DH_SPI_WR(uint8_t addr,uint8_t wrdata)
{  
     cs = 0;
     wait_us(10);
     spi_master.write(addr);
     spi_master.write(wrdata);
     wait_us(10);
     cs = 1;
}
void SPI_LIS3DH_Init()
{
    spi_master.format(8,3);
    spi_master.frequency(100000);
    wait_ms(5);
    LIS3DH_SPI_WR(0x24,0x80);
    wait_ms(5);
    LIS3DH_SPI_WR(0x20,0x17);
    LIS3DH_SPI_WR(0x23,0x80);
}
void Get_LIS3DH_Val(void)
{
    uint8_t Dx_L=1,Dy_L=1,Dz_L=1;
    uint8_t Dx_H=1,Dy_H=1,Dz_H=1;
    if(LIS3DH_SPI_RD(0x0f|0x80)==0x33)
    {
         printf("check device ok!\r\n");
         Dx_H=LIS3DH_SPI_RD(0x29|0x80);   
         Dx_L=LIS3DH_SPI_RD(0x28|0x80);
         Dy_H=LIS3DH_SPI_RD(0x2b|0x80);
         Dy_L=LIS3DH_SPI_RD(0x2A|0x80);
         Dz_H=LIS3DH_SPI_RD(0x2d|0x80);
         Dz_L=LIS3DH_SPI_RD(0x2C|0x80);  
     }
     else printf("check device err!\r\n");
     x_a=Dx_H<<8|Dx_L/16;
     y_a=Dy_H<<8|Dy_L/16;
     z_a=Dz_H<<8|Dz_L/16;
}


void Sensors_Init()
{
    SPI_LIS3DH_Init();
    
    bmp180.init();
}

//

// The Science Journal Service
static const uint8_t science_uuid_conn[]= {0xe8,0x74,0x2c,0x65,0xf0,0x01,0x38,0x95,0x7a,0x46,0xaa,0x0a,0x01,0x00,0x5a,0x55};

static const uint8_t science_base_uuid[] = {0x55,0x5a,0x00,0x01,0x0a,0xaa,0x46,0x7a,0x95,0x38,0x01,0xf0,0x65,0x2c,0x74,0xe8};

static const uint8_t science_value_uuid[]= {0x55,0x5a,0x00,0x03,0x0a,0xaa,0x46,0x7a,0x95,0x38,0x01,0xf0,0x65,0x2c,0x74,0xe8};

static const uint8_t science_config_uuid[]= {0x55,0x5a,0x00,0x10,0x0a,0xaa,0x46,0x7a,0x95,0x38,0x01,0xf0,0x65,0x2c,0x74,0xe8};




uint8_t valuePayload[TXRX_BUF_LEN] = {0,};
uint8_t configPayload[TXRX_BUF_LEN] = {0,};

static uint8_t config_buf[TXRX_BUF_LEN];
static uint8_t config_len=0;

GattCharacteristic  valueCharacteristic (science_value_uuid, valuePayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);

GattCharacteristic  configCharacteristic (science_config_uuid, configPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE);

GattCharacteristic *scienceChars[] = {&valueCharacteristic, &configCharacteristic};

GattService         scienceService(science_base_uuid, scienceChars, sizeof(scienceChars) / sizeof(GattCharacteristic *));

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    pc.printf("Disconnected \r\n");
    pc.printf("Restart advertising \r\n");
    ble.startAdvertising();
}

void WrittenHandler(const GattWriteCallbackParams *Handler)
{
    uint8_t buf[TXRX_BUF_LEN];
    uint16_t bytesRead;

    if (Handler->handle == configCharacteristic.getValueAttribute().getHandle()) 
    {
        ble.readCharacteristicValue(configCharacteristic.getValueAttribute().getHandle(), buf, &bytesRead);
        memset(valuePayload, 0, TXRX_BUF_LEN);
        memcpy(valuePayload, buf, TXRX_BUF_LEN);
        pc.printf("WriteHandler \r\n");
        int8_t handle_len = bytesRead;
        handle(valuePayload,handle_len);
        pc.printf("pin num : %d ",pin);

    }
}

void uartCB(void)
{
    while(pc.readable()) {
        config_buf[config_len++] = pc.getc();
        if(config_len>=20 || config_buf[config_len-1]=='\0' || config_buf[config_len-1]=='\n') 
        {
            ble.updateCharacteristicValue(valueCharacteristic.getValueAttribute().getHandle(), config_buf, config_len);
            pc.printf("RecHandler \r\n");
            pc.printf("Length: ");
            pc.putc(config_len);
            pc.printf("config_len:   %d",config_len);
            pc.printf("\r\n");
            config_len = 0;
            break;
        }
    }
}


int main(void)
{
    mytimer.start();
    ble.init();
    
    Sensors_Init();

    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(WrittenHandler);

    pc.baud(9600);
    pc.printf("BLE sensors tag init \r\n");

    pc.attach( uartCB , pc.RxIrq);

    //setup advertising
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);

    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME,
                                    (const uint8_t *) "BLE-mbed",sizeof("BLE-mbed") - 1);
    

    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)science_uuid_conn, sizeof(science_uuid_conn));
    // 100ms; in multiples of 0.625ms.
    ble.setAdvertisingInterval(160);

    ble.addService(scienceService);

    ble.startAdvertising();
    pc.printf("Advertising Start \r\n");

    while(1) 
    {
        LED_R = 0;
        LED_G = 0;
        
        if (pin == 0 && pin_type == ANALOG)   //Light sensor
        {
            //read analog value
            unsigned short AnalogValue=myanalog.read_u16();
            wait_ms(10);       
            pc.printf("Analog: %d\n",AnalogValue);    
            send_int_data(valueCharacteristic,mytimer.read_ms(),AnalogValue);  
        }
        else if(pin == 1 && pin_type == ANALOG)  //Temperature
        {
            bmp180.startTemperature();
            wait_ms(10);     // Wait for conversion to complete
            float temp;
            bmp180.getTemperature(&temp);
            int value = temp;
            pc.printf("Tem:%d \n",value);
            send_int_data(valueCharacteristic,mytimer.read_ms(),value); 
        }
        else if(pin == 2 && pin_type == ANALOG)    //Barometric pressure
        {
            bmp180.startPressure(BMP180::ULTRA_LOW_POWER);
            wait_ms(10);    // Wait for conversion to complete
            int pressure;
            bmp180.getPressure(&pressure);
            pc.printf("Pre:%d \n",pressure);
            send_int_data(valueCharacteristic,mytimer.read_ms(),pressure);   
        }
        else if(pin == 3 && pin_type == ANALOG)     //X axis acceleration  
        {
          Get_LIS3DH_Val();
          pc.printf("Dx=:%d\r\n",x_a);
          send_int_data(valueCharacteristic,mytimer.read_ms(),x_a);
        }
        else if(pin == 4 && pin_type == ANALOG)      //Y axis acceleration
        {
          Get_LIS3DH_Val();
          pc.printf("Dy=:%d\r\n",y_a);
          send_int_data(valueCharacteristic,mytimer.read_ms(),y_a);
        }
        else if(pin == 5 && pin_type == ANALOG)     //Z axis acceleration
        {
          Get_LIS3DH_Val();
          pc.printf("Dz=:%d\r\n",z_a);
          send_int_data(valueCharacteristic,mytimer.read_ms(),z_a);          
        }
        else if(pin == 6 && pin_type == ANALOG)
        {
            //add another sensor in here
        }
        else
        {
            int error_code = 404;
            send_int_data(valueCharacteristic,mytimer.read_ms(),error_code);
        }
        LED_R = 1;
        LED_G = 1;
       
        wait_ms(INTERNAL);
        ble.waitForEvent();
    }
}
















