#ifndef SOLAR_H
#define SOLAR_H

#include "mbed.h"
#include <string>
#include <stdlib.h>
#include <vector>

class solarArray {
  private:
    AnalogIn* panels;
    unsigned* pins;
    vector<float>* data; // a data vector for each panel
    size_t size; // up to 6 solar panels
    
  private:
    void allocatePanels();
    void samplingCycle(unsigned duration, float samplingPeriod);
    void sample();
    
  public:
    solarArray(); // default ctor
    solarArray(solarArray&); // copy ctor
    solarArray(size_t numPanels, unsigned* pinNums);
    ~solarArray();
    
    void gatherInfo_days(unsigned numDays, unsigned start, unsigned duration ,unsigned samplingPeriod);
    void gatherInfo_secs(unsigned start, unsigned duration, unsigned samplingPeriod);
    vector<float> getPanelData(unsigned panel);
};

solarArray::solarArray() : panels(NULL), pins(NULL),size(0)
{
}

void solarArray::allocatePanels() {
    for (int i = 0; i < size; i++) {
        if (pins[i] == 15)
            new (panels + i) AnalogIn(p15);
        else if (pins[i] == 16)
            new (panels + i) AnalogIn(p16);
        else if (pins[i] == 17)
            new (panels + i) AnalogIn(p17);
        else if (pins[i] == 18)
            new (panels + i) AnalogIn(p18);
        else if (pins[i] == 19)
            new (panels + i) AnalogIn(p19);
        else if (pins[i] == 20)
            new (panels + i) AnalogIn(p20);
    }
}

solarArray::solarArray(solarArray& rhs) : size(rhs.size)
{
    panels = (AnalogIn*) malloc(sizeof(AnalogIn)*size);
    pins = (unsigned*) malloc(sizeof(unsigned)*size);
    
    for (int i = 0; i < size; i++)
        new (pins + i) unsigned(rhs.pins[i]);
    
    allocatePanels();
}

solarArray::solarArray(size_t numPanels, unsigned* pinNums) : size(numPanels)
{
    panels = (AnalogIn*) malloc(sizeof(AnalogIn)*size);
    pins = (unsigned*) malloc(sizeof(unsigned)*size);
    
    for (int i = 0; i < size; i++)
        new (pins + i) unsigned(pinNums[i]);
        
    allocatePanels();
}

solarArray::~solarArray()
{
    for (int i = 0; i < size; i++)
        (panels + i)->~AnalogIn();
        
    if (panels != NULL && size > 0)
        free(panels);
        
    for (int i = 0; i < size; i++)
        (pins + i)->~unsigned();
        
    if (pins != NULL && size > 0)
        free(pins);
}

vector<float> solarArray::getPanelData(unsigned panel)
{
    return data[panel];
}

void solarArray::sample() {
    for (int i = 0; i < size; i++)
        data[i].push_back(panels[i]);
}

void solarArray::samplingCycle(unsigned duration, float samplingPeriod) {
    //Ticker t;
    
    data = (vector<float>*) malloc(sizeof(vector<float>)*size);
    
    for (int i = 0; i < size; i++)
        new (data + i) vector<float>;
    
    //t.attach(this::sample, samplingPeriod);
    for (float i = 0; i < duration; i += samplingPeriod) {
        for (int j = 0; j < size; j++)
            data[j].push_back(panels[j]*3.3);
        
        wait(samplingPeriod);   
    }
    //wait(duration);
}

void solarArray::gatherInfo_days(unsigned numDays, unsigned start, unsigned duration ,unsigned samplingPeriod)
{
    Ticker t;
    
    for (int i = 0; i < numDays; i++) {
        wait(start);
        
        samplingCycle(duration, samplingPeriod);
        
        wait(3600*24 - start - duration); // wait until the next day
    }
}

void solarArray::gatherInfo_secs(unsigned start, unsigned duration,unsigned samplingPeriod) {
    wait(start);
    samplingCycle(duration, samplingPeriod);
}

#endif