#include "mbed.h"
#include "rtos.h"
#include "tHIH6130.h"

// I2C
#define I2C_SDA p28
#define I2C_SCL p27

// CAN 
#define CAN_RD p30
#define CAN_TD p29

#define SIG_RX_CAN  0x01
#define ADR_HIH     0x27            // adresse I2C réelle (sur 7 bits)
#define FREQUENCY_CAN  20000 

// DATAS
#define TROP_CHAUD_TEMP 40  // Il fait "trop chaud" lorsqu'on dépasse 40°C
#define TROP_FROID_TEMP -10  // Il fait "trop froid" lorsqu'on est en dessous de 10°C
#define IL_PLEUT 80  // Forte probabilité de pleuvoir lorsqu'on est au dessus de 80% d'humidité

// TEMPORISATION
#define DELAY_ACQUISITION_MS 500

// IDENTIFIANTS TRAMES CAN
    // Module HIH6130
#define ID_TEMP             0x011
#define ID_HUMI             0x021
#define ID_TEMP_HUMI        0x031
#define ID_TROP_CHAUD       0x111
#define ID_TEMP_NORMAL      0x112
#define ID_TROP_FROID       0x113
#define ID_IL_PLEUT         0x121
#define ID_IL_PLEUT_PAS     0x122
    // Module LCD
#define ID_REQUETE_LCD      0x401


// LED
DigitalOut led1(LED1);

// CAPTEUR HIH6130
I2C i2c(I2C_SDA , I2C_SCL ); 
tHIH6130 Capteur(ADR_HIH,&i2c);
int currentTemp, currentHumi ;

// CAN
CAN CanPort(CAN_RD, CAN_TD);
CANMessage  MessageRx; 
CANMessage  MessageTx;

// Autres
Thread threadPrincipal;
Thread threadCANSend ;

/**
    Fonction d'envoi des données CAN.
**/
void envoiCAN(){
    while(true){
        Thread::signal_wait(SIG_RX_CAN); 
        
        // Config Tx general (à notre protocole applicatif)
        MessageTx.format = CANStandard;
        
        // Température datas
        MessageTx.type = CANData ;
        MessageTx.id = ID_TEMP_HUMI ;
        MessageTx.len= 2 ;
        MessageTx.data[0] = (char)currentTemp;
        MessageTx.data[1] = (char)currentHumi;
        CanPort.write(MessageTx);
        
        // Trop chaud/Trop froid/Température normale
            // Selection
        if(currentTemp >= TROP_CHAUD_TEMP)
            MessageTx.id = ID_TROP_CHAUD;
        else if(currentTemp <= TROP_FROID_TEMP)
            MessageTx.id = ID_TROP_FROID;
        else
            MessageTx.id = ID_TEMP_NORMAL;
            
            // Envoi 
        MessageTx.type = CANRemote ;
        MessageTx.len = 0 ;
        CanPort.write(MessageTx);
        
        
        // Il pleut/Il pleut pas  
            // Selection
        if(currentHumi >= IL_PLEUT)
            MessageTx.id = ID_IL_PLEUT ;
        else
            MessageTx.id = ID_IL_PLEUT_PAS ;
            
            // Envoi
        MessageTx.type = CANRemote ;
        MessageTx.len = 0 ;
        CanPort.write(MessageTx);
    }
}

/**
    Fonction de lecture de la communication CAN.
**/
void canReader(){
    if (CanPort.read(MessageRx))
    {
        if(MessageRx.id == ID_REQUETE_LCD){  // Si le message est du LCD alors...
            led1 = !led1;   // On change l'état de la LED eventuellement (effet de clignotement si rapide)
            threadPrincipal.signal_set(SIG_RX_CAN); // On réveille notre thread qui enverra les valeurs des capteurs
        }
    }
}

/**
    Fonction de lecture des données des capteurs.
**/
void lireCapteurs(){
    // Récupérer la valeur des capteurs
    Capteur.StartMesure();
    wait_ms(50);
    Capteur.UpdateData();
    currentTemp = (int) Capteur.getTemp();
    currentHumi = (int) Capteur.getHumi();
    
}

/**
    Fonction d'acquisition des données des capteurs.
**/
void acquisition(){
    while(1){
        lireCapteurs();
        wait_ms(DELAY_ACQUISITION_MS);
    }   
}

/**
    Fonction d'initialisation de la communication CAN et des threads
**/
void init(){
    // Initialisation du CAN
    CanPort.frequency(FREQUENCY_CAN);  
    
    // Attach 
    CanPort.attach(canReader,CAN::RxIrq);
    
    // Threads
    threadPrincipal.start(acquisition); 
    threadCANSend.start(envoiCAN);
}

int main() { 
        
    // initialisation
    init();
    
    printf("Start OK\n");
    
    while(1) 
    {
    }
}

