/* C-based low-memory footprint JSON parser for mbed
 * Based on Serge Zaitsev's JSMN https://bitbucket.org/zserge/jsmn/wiki/Home
 * JSMN is distributed under MIT license.
 *
 * Copyright (c) 2010 Serge A. Zaitsev
 * Copyright (c) 2014 YoongHM
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef MBED_JSMN_H
#define MBED_JSMN_H

/**
 * A c-based low-memory low-memory footprint JSON parser for mbed.
 *
 * The code is based on Serge Zaitsev's JSMN on
 * https://bitbucket.org/zserge/jsmn/wiki/Home
 *
 * @code
 *#include "mbed.h"
 *
 *#include "jsmn.h"
 *
 *#define       MAXTOKEN       64
 *
 *const char *jsmn_type_str[] = {
 *  "PRIMITIVE",
 *  "OBJECT",
 *  "ARRAY",
 *  "STRING"
 *};
 *
 *int main()
 *{
 *  const char *js;            // Pointer to json string
 *  int         r;             // Number of token parsed
 *  jsmn_parser p;             // jsmn parser
 *  jsmntok_t   t[MAXTOKEN];   // Parsed token
 *
 *  // JSON may contain multibyte characters or encoded unicode in
 *  // \uXXXX format.
 *  // mbed compiler may complain "invalid multibyte characters ...".
 *  js =
 *"{"
 *"  \"menu\":"
 *"  {"
 *"    \"id\": 1234,"
 *"    \"group\": \"File\","
 *"    \"popup\":"
 *"    {"
 *"      \"menuitem\":"
 *"      ["
 *"        {\"value\": true,    \"onclick\"  : \"বিশাল\"},"
 *"        {\"value\": false,   0x1328       : \"groß\"},"
 *"        {\"value\": null,    \"15\u00f8C\": \"3\u0111\"},"
 *"        {\"value\": \"測試\", -12.34       :  99}"
 *"      ]"
 *"    }"
 *"  }"
 *"}";
 *
 *  jsmn_init(&p);
 *  r = jsmn_parse(&p, js, strlen(js), t, MAXTOKEN);
 *
 *  printf("Parsed %d tokens\n", r);
 *  
 *  printf("            TYPE       START   END  SIZE PAR\n");
 *  printf("           ----------  -----  ----  ---- ---\n");
 *
 *  char        ch;
 *  jsmntok_t   at;            // A token for general use
 *  for (int i = 0; i < r; i++)
 *  {
 *    at = t[i];
 *    printf("Token %2d = %-10.10s (%4d - %4d, %3d, %2d) --> ",
 *           i, jsmn_type_str[at.type],
 *           at.start, at.end,
 *           at.size, at.parent);
 *
 *    switch (at.type)
 *    {
 *      case JSMN_STRING:
 *        printf("%-10.*s\n", at.end - at.start + 2, js + at.start - 1);
 *        break;
 *
 *      case JSMN_PRIMITIVE:
 *        ch = *(js + at.start);
 *
 *        if (isdigit(ch) || ch == '-') 
 *          printf("%-10.*s\n", at.end - at.start, js + at.start);
 *        else if (tolower(ch) == 'n')
 *          printf("null\n");
 *        else if (tolower(ch) == 't')
 *          printf("true\n");
 *        else if (tolower(ch) == 'f')
 *          printf("false\n");
 *        break;
 *
 *      default:
 *        printf("\n");
 *        break;
 *    }
 *  }
 * 
 *  while (1)
 *    ;
 *} 
 * @endcode
 */

/** jsmntype_t - JSON type recongized by JSMN.
 *  There are
 *  - Primitive: number, boolean (true/false) or null
 *  - Object
 *  - Array
 *  - String
 */
typedef enum {
  JSMN_PRIMITIVE = 0,
  JSMN_OBJECT    = 1,
  JSMN_ARRAY     = 2,
  JSMN_STRING    = 3,
  JSMN_INVALID   = 4
} jsmntype_t;


/** Error message return by JSMN API
 * 
 */
#define  JSMN_ERR_OKAY    0  /* No problem                            */
#define  JSMN_ERR_NOMEM  -1  /* Not enough tokens were provided       */
#define  JSMN_ERR_INVAL  -2  /* Invalid character inside JSON string  */
#define  JSMN_ERR_PART   -3  /* Incomplete JSON packet                */ 


/** jsmntok_t - JSON token structure.
 * 
 * It is a c structure that contained parsed JSON token.
 *
 * @param   type    type (primitive, object, array, string)
 * @param   start   start position in JSON data string, excl " if is a string
 * @param   end     end position in JSON data string, incl " if is a string
 * @param   size    number of token directly under it
 * @param   parent  link to its direct parent token number
 */
typedef struct {
  jsmntype_t type;
  int        start;          /* Token's start pos;excl " if is a str  */
  int        end;            /* Token's last  pos;incl " if is a str  */
  int        size;           /* Number of token directly under it     */
  int        parent;         /* Nth-token which is its direct parent  */
} jsmntok_t;


/** JSON parser - JSON parser structure.
 *
 * Contains an array of token blocks available. Also stores
 * the string being parsed now and current position in that string
 */
typedef struct {
  unsigned int pos;          /* offset in the JSON string */
  unsigned int toknext;      /* next token to allocate */
  int          toksuper;     /* superior token node, (parent or array)*/
} jsmn_parser;


/** Initialize jsmn_parser structure for a new JSON parsing
 *
 * @param *parser  jsmn_parser structure
 */
void 
jsmn_init
(jsmn_parser *parser
);

/** Parse a given JSON data string into array of jsmntok_t tokens.
 * 
 * @param *parser  jsmn_parser structure
 * @param *js      JSON data string
 * @param  len     string length of *js
 * @param *tokens  tokens to hold the parsed data 
 */
int
jsmn_parse
(jsmn_parser *parser
,const char  *js
,size_t       len
,jsmntok_t   *tokens
,unsigned int num_tokens
);


#endif // JSMN_H