/*
 * @author Rita Yang
 * @filename ADIS16467.h
 *
 * @section DESCRIPTION
 *
 * Software Driver for ADIS16467-3 Precision MEMS IMU Module
 *
 * Datasheet:
 *
 * https://www.analog.com/media/en/technical-documentation/data-sheets/adis16467.pdf
 */  

#ifndef ADIS16467_H
#define ADIS16467_H

#include "mbed.h"

namespace ADIS16467
{
	typedef struct BurstReadResult
	{
		int32_t gyroX;
		int32_t gyroY;
		int32_t gyroZ;
		int32_t accelX;
		int32_t accelY;
		int32_t accelZ;
		uint16_t statusDiag;
		uint16_t internalTemp;
		uint16_t dataCounter;
		uint16_t checkSum;
	} BurstReadResult;

	typedef struct FirmwareInfo
	{
		uint8_t firmRevMajor;
		uint8_t firmRevMinor;

		uint8_t firmwareDay;
		uint8_t firmwareMonth;
		uint16_t firmwareYear;
		uint16_t serialNum;
	} FirmwareInfo;

	class ADIS16467
	{
		public:
			ADIS16467(Serial *debug, PinName DIN, PinName DOUT, PinName SCLK, PinName CS, PinName RST);
			void initADIS();

			uint32_t readResetTimer();
			uint16_t readStatDiag();
			void setErrorFlags(uint16_t statusDiagVal);
			bool getClockError();
			bool getMemFailure();
			bool getSensorFailure();
			bool getStandbyMode();
			bool getSPIError();
			bool getFlashUpFail();
			bool getDatapathOverrun();
			bool checkExistence();
			void resetAllFlags();
			int32_t readGyroX();
			int32_t readGyroY();
			int32_t readGyroZ();
			int32_t readAccelX();
			int32_t readAccelY();
			int32_t readAccelZ();
			uint16_t readInternalTemp();
			uint16_t readTimeStamp();
			uint16_t readDataCounter();
			void burstRead(BurstReadResult &newResult);
			void setPollRate(uint16_t hz);
			void sensorSelfTest();
			bool updateDeltaAngles();
			int64_t getDeltaAngleXSum();
			int64_t getDeltaAngleYSum();
			int64_t getDeltaAngleZSum();
			void updateDeltaVels();
			int64_t getDelVelXSum();
			int64_t getDelVelYSum();
			int64_t getDelVelZSum();
			void setGyroBiases(float xBias, float yBias, float zBias);
			void getFirmwareInformation(FirmwareInfo &info);
			void resetDeltaAngle();
			void resetDeltaVel();

			//conversion factors
			static constexpr float GYRO_CONV = 0.1f / (1 << 16); //10 LSB/degree/sec
			static constexpr float ACCEL_CONV = 1.25f / (1 << 16); //1 LSB = 1.25 mg
			static constexpr float TEMP_CONV = 0.1f; //1 LSB = 0.1 degree Celcius
			static constexpr float TIME_CONV = 49.02f; //1 LSB = 49.02 us
			static constexpr float DELTA_ANGLE_CONV =  2160.0f / (1 << 31); //degrees per LSB 	
			static constexpr float DELTA_VEL_CONV = 400.0f / (1 << 15); //meters/sec per LSB

		private:
			Serial *debugPort;
			SPI spi;
			DigitalOut cs;
			DigitalOut rst;
			Timer resetTimer; //keep track of reset time
			bool clockError;
			bool memoryFailure;
			bool sensorFailure;
			bool standbyMode;
			bool comSPIError;
			bool flashMemUpdateFailure;
			bool datapathOverrun;
			void writeConfigurationRegister(uint8_t addressLow, uint8_t addressHigh, int16_t data);
			uint16_t readRegister(uint8_t address);
			int32_t read32BitValue(uint8_t highAddress, uint8_t lowAddress);
			void begin();
			void select();
			void deselect();

			int64_t deltaAngleX_sum;
			int64_t deltaAngleY_sum;
			int64_t deltaAngleZ_sum;
			int32_t readDeltaAngleX();
			int32_t readDeltaAngleY();
			int32_t readDeltaAngleZ();
			int32_t lastDC_DeltaAngle;

			int64_t deltaVelocityX_sum;
			int64_t deltaVelocityY_sum;
			int64_t deltaVelocityZ_sum;
			int32_t readDeltaVelX();
			int32_t readDeltaVelY();
			int32_t readDeltaVelZ();
			int32_t lastDC_DeltaVel;

			uint16_t curr_DecRate;
	};
}



#endif
