#ifndef GAME_ONE_H
#define GAME_ONE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Coin.h"
#include "Object.h"
#include "Game_one_cha.h"
#include "Score.h"

/** Game One Class
* @brief This class is the game engine for the first mini-game
* @version 2.0
* @author Yiu Fai Kwok
* @date May, 2019
*/

class Game_one
{

public:

    /** Constructor */
    Game_one();
    /** Deconstructor */
    ~Game_one();

    /**
     * @brief Initialize the class parameters
     * @param speed (int)
     * @param character cha (int)
     * @param ratio r (int)
     * @details Initialize the object speed (speed), the character sprite (cha) and the ratio (r) between coin and block
     */
    void init(int speed, int cha, int r);
    
    /**
     * @brief Render the lcd screen
     * @param character cha (int)
     * @details Render all objects in game, draw character sprite according to (cha) parameter
     */
    void render(N5110 &lcd, int cha);
    
    /**
     * @brief Read input from Gamepad
     * @details Read input from joystick on the Gamepad to determine direction and magnitude of character's motion
     */
    void read_input(Gamepad &pad);
    
    /**
     * @brief Update the lcd screen
     * @details Update the lcd screen according to the input and current state of the display
     */
    void update(Gamepad &pad, N5110 &lcd);
    
    /**
     * @brief Draw function for Game 1
     * @param character cha (int)
     * @details Draw function of Game 1 which draw all objects and characters according to the changing variables of positions, alternative sprites and constant (cha)
     */
    void draw(N5110 &lcd, int cha);
    
    /**
     * @brief Get count value
     * @return The current count (int)
     * @details The function returns the number of coins that have appeared in the game to terminate the game loop once count reaches 10
     */
    int get_count();
    
    /**
     * @brief Print Scores on screen
     * @return The current score (int)
     * @details The function print the number of coins collected and spawned on the lcd screen
     */
    int print_scores(N5110 &lcd);
    
    /**
     * @brief Intruction 
     * @details The function displays the instructions which describe game rules at the beginning of the game
     */
    void intro(Gamepad &pad, N5110 &lcd);
    
    /**
     * @brief Update parameter alt
     * @return current value of alt (int)
     * @details The function flips the value of alt to draw the alternative sprite of character to create a running animation
     */
    int update_alt();
    
private:

    /**
     * @brief Check coin collection
     * @details The private function checks if the coin has been collected, update the counter, the score and initiates new object in the game
     */
    void check_player_collect(Gamepad &pad);
    
    /**
     * @brief Check block collision
     * @details The private function checks if the block is collided, terminates the game loop and display the gameover screen
     */
    void check_player_collide(Gamepad &pad, N5110 &lcd);
    
    /**
     * @brief Check coin missed
     * @details The private function checks if the coin has been missed, update the counter, and initiates new object in the game
     */
    void check_miss_coin(Gamepad &pad);
    
    /**
     * @brief Check block missed
     * @details The private function checks if the block has been missed, then initiates new object in the game
     */
    void check_miss_block(Gamepad &pad);
    
    /**
     * @brief Gameover screen
     * @details The private function initiates the gameover screen and termiate term
     */
    void gameover(N5110 &lcd, Gamepad &pad);
    
    /**
     * @brief Moving background
     * @param position x (int)
     * @details The private function draws the moving background according to parameter x
     */
    void draw_background(int x, N5110 &lcd);
    
    /**
     * @brief Stationary background
     * @details The private function draws the stationary background for further objects
     */
    void background(N5110 &lcd);
    
    Score music;
    Game_one_cha _p1;

    int _speed;
    int _cha;
    int _rand;
    
    // x positions of the player
    int _p1x;
    int _x;
    
    Coin _coin;
    Object _block;
    
    Direction _d;
    float _mag;
    int _count;
    int _alt;
    int _type;

};

#endif