#include "DefineMPU.h"
#include "mbed.h"
#include "math.h"

#define Kp 3.0f * 5.0f // 2 - 5 these are the free parameters in the Mahony filter and fusion scheme, Kp for proportional feedback, Ki for integral
#define Ki 1.0f

class MPU9250 {
    public :     
        int numero;                
   
    private:   
        
        static float Filter;
                  
        static float magCalibration[3]; // Factory mag calibration and mag bias
        static float magbias[3];        // Factory mag calibration and mag bias
        
        static float gyroBias[3];  // Bias corrections for gyro and accelerometer
        static float accelBias[3]; // Bias corrections for gyro and accelerometer
        
        int16_t accelCount[3];  // Stores the 16-bit signed accelerometer sensor output
        int16_t gyroCount[3];   // Stores the 16-bit signed gyro sensor output
        int16_t magCount[3];    // Stores the 16-bit signed magnetometer sensor output
        
        int16_t tempCount;   // Stores the real internal chip temperature in degrees Celsius
        float temperature;
        float SelfTest[6];
        
        static float GyroMeasError;   // gyroscope measurement error in rads/s (start at 60 deg/s), then reduce after ~10 s to 3
        static float beta;            // compute beta
        static float GyroMeasDrift;   // gyroscope measurement drift in rad/s/s (start at 0.0 deg/s/s)
        static float zeta;  // compute zeta, the other free parameter in the Madgwick scheme usually set to a small or zero value
    
        float pitch, yaw, roll;
        
        static float q[4];        // vector to hold quaternion
        static float eInt[3];     // vector to hold integral error for Mahony method 
         
        float aRes, gRes, mRes;      // scale resolutions per LSB for the sensors
        float ax, ay, az, gx, gy, gz, mx, my, mz; // variables to hold latest sensor data values 
                 
    public:  
      
//===================================================================================================================
//====== Set of useful function to access acceleratio, gyroscope, and temperature data
//===================================================================================================================

    MPU9250() {}

    MPU9250(int n){ numero = n; }
    
    int GetPitch (void){ return (int)pitch; }
    int GetRoll (void){ return (int)roll; }      
    int GetYaw (void){ return (int)yaw; }     
       
    void writeByte(uint8_t address, uint8_t subAddress, uint8_t data){
       char data_write[2];
       data_write[0] = subAddress;
       data_write[1] = data;       
       i2c.write(address, data_write, 2, 0);
   }

    char readByte(uint8_t address, uint8_t subAddress){
        char data[1]; 
        char data_write[1];
        data_write[0] = subAddress;
        i2c.write(address, data_write, 1, 1); 
        i2c.read(address, data, 1, 0); 
        return data[0]; 
    }

    void readBytes(uint8_t address, uint8_t subAddress, uint8_t count, uint8_t * dest){     
        char data[14];
        char data_write[1];
        data_write[0] = subAddress;
        i2c.write(address, data_write, 1, 1); 
        i2c.read(address, data, count, 0); 
        for(int ii = 0; ii < count; ii++) {
            dest[ii] = data[ii];
        }
    } 
 
    void getMres() {
        switch (Mscale){
            case MFS_14BITS: mRes = 10.0*4219.0/8190.0; break;
            case MFS_16BITS: mRes = 10.0*4219.0/32760.0; break;
        }
    }

    void getGres() {
        switch (Gscale){
            case GFS_250DPS: gRes = 250.0/32768.0; break;
            case GFS_500DPS: gRes = 500.0/32768.0; break;
            case GFS_1000DPS: gRes = 1000.0/32768.0; break;
            case GFS_2000DPS: gRes = 2000.0/32768.0; break;
        }
    }

    void getAres() {
        switch (Ascale){
            case AFS_2G: aRes = 2.0/32768.0; break;
            case AFS_4G: aRes = 4.0/32768.0; break;
            case AFS_8G: aRes = 8.0/32768.0; break;
            case AFS_16G: aRes = 16.0/32768.0; break;
        }
    }

    void readAccelData(int16_t * destination){
        uint8_t rawData[6];  // x/y/z accel register data stored here
        readBytes(MPU9250_ADDRESS, ACCEL_XOUT_H, 6, &rawData[0]);  // Read the six raw data registers into data array
        destination[0] = (int16_t)(((int16_t)rawData[0] << 8) | rawData[1]) ;  // Turn the MSB and LSB into a signed 16-bit value
        destination[1] = (int16_t)(((int16_t)rawData[2] << 8) | rawData[3]) ;  
        destination[2] = (int16_t)(((int16_t)rawData[4] << 8) | rawData[5]) ; 
    }
    
    void readGyroData(int16_t * destination){
        uint8_t rawData[6];  // x/y/z gyro register data stored here
        readBytes(MPU9250_ADDRESS, GYRO_XOUT_H, 6, &rawData[0]);  // Read the six raw data registers sequentially into data array
        destination[0] = (int16_t)(((int16_t)rawData[0] << 8) | rawData[1]) ;  // Turn the MSB and LSB into a signed 16-bit value
        destination[1] = (int16_t)(((int16_t)rawData[2] << 8) | rawData[3]) ;  
        destination[2] = (int16_t)(((int16_t)rawData[4] << 8) | rawData[5]) ; 
    }

    void readMagData(int16_t * destination){
        uint8_t rawData[7];  // x/y/z gyro register data, ST2 register stored here, must read ST2 at end of data acquisition
        if(readByte(AK8963_ADDRESS, AK8963_ST1) & 0x01) { // wait for magnetometer data ready bit to be set
        readBytes(AK8963_ADDRESS, AK8963_XOUT_L, 7, &rawData[0]);  // Read the six raw data and ST2 registers sequentially into data array
        uint8_t c = rawData[6]; // End data read by reading ST2 register
        if(!(c & 0x08)) { // Check if magnetic sensor overflow set, if not then report data
            destination[0] = (int16_t)(((int16_t)rawData[1] << 8) | rawData[0]);  // Turn the MSB and LSB into a signed 16-bit value
            destination[1] = (int16_t)(((int16_t)rawData[3] << 8) | rawData[2]) ;  // Data stored as little Endian
            destination[2] = (int16_t)(((int16_t)rawData[5] << 8) | rawData[4]) ; 
            }
        }
    }    
    
    void resetMPU9250() { // reset device
        writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x80); // Write a one to bit 7 reset bit; toggle reset device
        wait(0.1);
    }
      
    void initAK8963(float * destination){ // First extract the factory calibration for each magnetometer axis
        uint8_t rawData[3];  // x/y/z gyro calibration data stored here
        writeByte(AK8963_ADDRESS, AK8963_CNTL, 0x00); // Power down magnetometer  
        wait(0.01);
        writeByte(AK8963_ADDRESS, AK8963_CNTL, 0x0F); // Enter Fuse ROM access mode
        wait(0.01);
        readBytes(AK8963_ADDRESS, AK8963_ASAX, 3, &rawData[0]);  // Read the x-, y-, and z-axis calibration values
        destination[0] =  (float)(rawData[0] - 128)/256.0f + 1.0f;   // Return x-axis sensitivity adjustment values, etc.
        destination[1] =  (float)(rawData[1] - 128)/256.0f + 1.0f;  
        destination[2] =  (float)(rawData[2] - 128)/256.0f + 1.0f; 
        writeByte(AK8963_ADDRESS, AK8963_CNTL, 0x00); // Power down magnetometer  
        wait(0.01);
        writeByte(AK8963_ADDRESS, AK8963_CNTL, Mscale << 4 | Mmode); // Set magnetometer data resolution and sample ODR
        wait(0.01);
    }

    void initMPU9250(){  
        writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x00); // Clear sleep mode bit (6), enable all sensors 
        wait(0.1); // Delay 100 ms for PLL to get established on x-axis gyro; should check for PLL ready interrupt  
        
        writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x01);  // Set clock source to be PLL with x-axis gyroscope reference, bits 2:0 = 001
        writeByte(MPU9250_ADDRESS, CONFIG, 0x03);  
     
        writeByte(MPU9250_ADDRESS, SMPLRT_DIV, 0x04);  // Use a 200 Hz rate; the same rate set in CONFIG above
        
        uint8_t c =  readByte(MPU9250_ADDRESS, GYRO_CONFIG);
        writeByte(MPU9250_ADDRESS, GYRO_CONFIG, c & ~0xE0); // Clear self-test bits [7:5] 
        writeByte(MPU9250_ADDRESS, GYRO_CONFIG, c & ~0x18); // Clear AFS bits [4:3]
        writeByte(MPU9250_ADDRESS, GYRO_CONFIG, c | Gscale << 3); // Set full scale range for the gyro
    
        c =  readByte(MPU9250_ADDRESS, ACCEL_CONFIG);
        writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, c & ~0xE0); // Clear self-test bits [7:5] 
        writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, c & ~0x18); // Clear AFS bits [4:3]
        writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, c | Ascale << 3); // Set full scale range for the accelerometer 
    
        c = readByte(MPU9250_ADDRESS, ACCEL_CONFIG2);
        writeByte(MPU9250_ADDRESS, ACCEL_CONFIG2, c & ~0x0F); // Clear accel_fchoice_b (bit 3) and A_DLPFG (bits [2:0])  
        writeByte(MPU9250_ADDRESS, ACCEL_CONFIG2, c | 0x03); // Set accelerometer rate to 1 kHz and bandwidth to 41 Hz
    
        writeByte(MPU9250_ADDRESS, INT_PIN_CFG, 0x22);    
        writeByte(MPU9250_ADDRESS, INT_ENABLE, 0x01);  // Enable data ready (bit 0) interrupt
    }

    void calibrateMPU9250(float * dest1, float * dest2){  
        uint8_t data[12]; // data array to hold accelerometer and gyro x, y, z, data
        uint16_t ii, packet_count, fifo_count;
        int32_t gyro_bias[3] = {0, 0, 0}, accel_bias[3] = {0, 0, 0};

        writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x80); // Write a one to bit 7 reset bit; toggle reset device
        wait(0.1);  

        writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x01);  
        writeByte(MPU9250_ADDRESS, PWR_MGMT_2, 0x00); 
        wait(0.2);

        writeByte(MPU9250_ADDRESS, INT_ENABLE, 0x00);   // Disable all interrupts
        writeByte(MPU9250_ADDRESS, FIFO_EN, 0x00);      // Disable FIFO
        writeByte(MPU9250_ADDRESS, PWR_MGMT_1, 0x00);   // Turn on internal clock source
        writeByte(MPU9250_ADDRESS, I2C_MST_CTRL, 0x00); // Disable I2C master
        writeByte(MPU9250_ADDRESS, USER_CTRL, 0x00);    // Disable FIFO and I2C master modes
        writeByte(MPU9250_ADDRESS, USER_CTRL, 0x0C);    // Reset FIFO and DMP
        wait(0.015);

        writeByte(MPU9250_ADDRESS, CONFIG, 0x01);      // Set low-pass filter to 188 Hz
        writeByte(MPU9250_ADDRESS, SMPLRT_DIV, 0x00);  // Set sample rate to 1 kHz
        writeByte(MPU9250_ADDRESS, GYRO_CONFIG, 0x00);  // Set gyro full-scale to 250 degrees per second, maximum sensitivity
        writeByte(MPU9250_ADDRESS, ACCEL_CONFIG, 0x00); // Set accelerometer full-scale to 2 g, maximum sensitivity
        
        uint16_t  gyrosensitivity  = 131;   // = 131 LSB/degrees/sec
        uint16_t  accelsensitivity = 16384;  // = 16384 LSB/g

        writeByte(MPU9250_ADDRESS, USER_CTRL, 0x40);   // Enable FIFO  
        writeByte(MPU9250_ADDRESS, FIFO_EN, 0x78);     // Enable gyro and accelerometer sensors for FIFO (max size 512 bytes in MPU-9250)
        wait(0.04); // accumulate 40 samples in 80 milliseconds = 480 bytes

        writeByte(MPU9250_ADDRESS, FIFO_EN, 0x00);        // Disable gyro and accelerometer sensors for FIFO
        readBytes(MPU9250_ADDRESS, FIFO_COUNTH, 2, &data[0]); // read FIFO sample count
        fifo_count = ((uint16_t)data[0] << 8) | data[1];
        packet_count = fifo_count/12;// How many sets of full gyro and accelerometer data for averaging
    
        for (ii = 0; ii < packet_count; ii++) {
            int16_t accel_temp[3] = {0, 0, 0}, gyro_temp[3] = {0, 0, 0};
            readBytes(MPU9250_ADDRESS, FIFO_R_W, 12, &data[0]); // read data for averaging
            accel_temp[0] = (int16_t) (((int16_t)data[0] << 8) | data[1]  ) ;  // Form signed 16-bit integer for each sample in FIFO
            accel_temp[1] = (int16_t) (((int16_t)data[2] << 8) | data[3]  ) ;
            accel_temp[2] = (int16_t) (((int16_t)data[4] << 8) | data[5]  ) ;    
            gyro_temp[0]  = (int16_t) (((int16_t)data[6] << 8) | data[7]  ) ;
            gyro_temp[1]  = (int16_t) (((int16_t)data[8] << 8) | data[9]  ) ;
            gyro_temp[2]  = (int16_t) (((int16_t)data[10] << 8) | data[11]) ;
            
            accel_bias[0] += (int32_t) accel_temp[0]; // Sum individual signed 16-bit biases to get accumulated signed 32-bit biases
            accel_bias[1] += (int32_t) accel_temp[1];
            accel_bias[2] += (int32_t) accel_temp[2];
            gyro_bias[0]  += (int32_t) gyro_temp[0];
            gyro_bias[1]  += (int32_t) gyro_temp[1];
            gyro_bias[2]  += (int32_t) gyro_temp[2];       
        }
        
        accel_bias[0] /= (int32_t) packet_count; // Normalize sums to get average count biases
        accel_bias[1] /= (int32_t) packet_count;
        accel_bias[2] /= (int32_t) packet_count;
        gyro_bias[0]  /= (int32_t) packet_count;
        gyro_bias[1]  /= (int32_t) packet_count;
        gyro_bias[2]  /= (int32_t) packet_count;
        
        if(accel_bias[2] > 0L) {accel_bias[2] -= (int32_t) accelsensitivity;}  // Remove gravity from the z-axis accelerometer bias calculation
        else {accel_bias[2] += (int32_t) accelsensitivity;}
     
        data[0] = (-gyro_bias[0]/4  >> 8) & 0xFF; // Divide by 4 to get 32.9 LSB per deg/s to conform to expected bias input format
        data[1] = (-gyro_bias[0]/4)       & 0xFF; // Biases are additive, so change sign on calculated average gyro biases
        data[2] = (-gyro_bias[1]/4  >> 8) & 0xFF;
        data[3] = (-gyro_bias[1]/4)       & 0xFF;
        data[4] = (-gyro_bias[2]/4  >> 8) & 0xFF;
        data[5] = (-gyro_bias[2]/4)       & 0xFF;
    
        dest1[0] = (float) gyro_bias[0]/(float) gyrosensitivity; // construct gyro bias in deg/s for later manual subtraction
        dest1[1] = (float) gyro_bias[1]/(float) gyrosensitivity;
        dest1[2] = (float) gyro_bias[2]/(float) gyrosensitivity;
    
        int32_t accel_bias_reg[3] = {0, 0, 0}; // A place to hold the factory accelerometer trim biases
        readBytes(MPU9250_ADDRESS, XA_OFFSET_H, 2, &data[0]); // Read factory accelerometer trim values
        accel_bias_reg[0] = (int16_t) ((int16_t)data[0] << 8) | data[1];
        readBytes(MPU9250_ADDRESS, YA_OFFSET_H, 2, &data[0]);
        accel_bias_reg[1] = (int16_t) ((int16_t)data[0] << 8) | data[1];
        readBytes(MPU9250_ADDRESS, ZA_OFFSET_H, 2, &data[0]);
        accel_bias_reg[2] = (int16_t) ((int16_t)data[0] << 8) | data[1];
    
        uint32_t mask = 1uL; // Define mask for temperature compensation bit 0 of lower byte of accelerometer bias registers
        uint8_t mask_bit[3] = {0, 0, 0}; // Define array to hold mask bit for each accelerometer bias axis
    
        for(ii = 0; ii < 3; ii++) {
            if(accel_bias_reg[ii] & mask) mask_bit[ii] = 0x01; // If temperature compensation bit is set, record that fact in mask_bit
        }
    
        accel_bias_reg[0] -= (accel_bias[0]/8); // Subtract calculated averaged accelerometer bias scaled to 2048 LSB/g (16 g full scale)
        accel_bias_reg[1] -= (accel_bias[1]/8);
        accel_bias_reg[2] -= (accel_bias[2]/8);
        
        data[0] = (accel_bias_reg[0] >> 8) & 0xFF;
        data[1] = (accel_bias_reg[0])      & 0xFF;
        data[1] = data[1] | mask_bit[0]; // preserve temperature compensation bit when writing back to accelerometer bias registers
        data[2] = (accel_bias_reg[1] >> 8) & 0xFF;
        data[3] = (accel_bias_reg[1])      & 0xFF;
        data[3] = data[3] | mask_bit[1]; // preserve temperature compensation bit when writing back to accelerometer bias registers
        data[4] = (accel_bias_reg[2] >> 8) & 0xFF;
        data[5] = (accel_bias_reg[2])      & 0xFF;
        data[5] = data[5] | mask_bit[2]; // preserve temperature compensation bit when writing back to accelerometer bias registers
    
        dest2[0] = (float)accel_bias[0]/(float)accelsensitivity; 
        dest2[1] = (float)accel_bias[1]/(float)accelsensitivity;
        dest2[2] = (float)accel_bias[2]/(float)accelsensitivity;
    }   

    void MadgwickQuaternionUpdate(void){
    //void MadgwickQuaternionUpdate(float deltat){
        
        /*float tmp = mx;
        mx = my;
        my = tmp;         
        az = -az;*/
        
        gx = gx*PI/180.0f;
        gy = gy*PI/180.0f;
        gz = gz*PI/180.0f;
        
        float q1 = q[0], q2 = q[1], q3 = q[2], q4 = q[3];   // short name local variable for readability
        float norm;
        float hx, hy, _2bx, _2bz;
        float s1, s2, s3, s4;
        float qDot1, qDot2, qDot3, qDot4;

        float _2q1mx;
        float _2q1my;
        float _2q1mz;
        float _2q2mx;
        float _4bx;
        float _4bz;
        float _2q1 = 2.0f * q1;
        float _2q2 = 2.0f * q2;
        float _2q3 = 2.0f * q3;
        float _2q4 = 2.0f * q4;
        float _2q1q3 = 2.0f * q1 * q3;
        float _2q3q4 = 2.0f * q3 * q4;
        float q1q1 = q1 * q1;
        float q1q2 = q1 * q2;
        float q1q3 = q1 * q3;
        float q1q4 = q1 * q4;
        float q2q2 = q2 * q2;
        float q2q3 = q2 * q3;
        float q2q4 = q2 * q4;
        float q3q3 = q3 * q3;
        float q3q4 = q3 * q4;
        float q4q4 = q4 * q4;

        norm = sqrt(ax * ax + ay * ay + az * az);
        if (norm == 0.0f) return; 
        norm = 1.0f/norm;
        ax *= norm;
        ay *= norm;
        az *= norm;

        norm = sqrt(mx * mx + my * my + mz * mz);
        if (norm == 0.0f) return; 
        norm = 1.0f/norm;
        mx *= norm;
        my *= norm;
        mz *= norm;

        _2q1mx = 2.0f * q1 * mx;
        _2q1my = 2.0f * q1 * my;
        _2q1mz = 2.0f * q1 * mz;
        _2q2mx = 2.0f * q2 * mx;
        
        hx = mx * q1q1 - _2q1my * q4 + _2q1mz * q3 + mx * q2q2 + _2q2 * my * q3 + _2q2 * mz * q4 - mx * q3q3 - mx * q4q4;
        hy = _2q1mx * q4 + my * q1q1 - _2q1mz * q2 + _2q2mx * q3 - my * q2q2 + my * q3q3 + _2q3 * mz * q4 - my * q4q4;
        _2bx = sqrt(hx * hx + hy * hy);
        _2bz = -_2q1mx * q3 + _2q1my * q2 + mz * q1q1 + _2q2mx * q4 - mz * q2q2 + _2q3 * my * q4 - mz * q3q3 + mz * q4q4;
        _4bx = 2.0f * _2bx;
        _4bz = 2.0f * _2bz;

        s1 = -_2q3 * (2.0f * q2q4 - _2q1q3 - ax) + _2q2 * (2.0f * q1q2 + _2q3q4 - ay) - _2bz * q3 * (_2bx * (0.5f - q3q3 - q4q4) + _2bz * (q2q4 - q1q3) - mx) + (-_2bx * q4 + _2bz * q2) * (_2bx * (q2q3 - q1q4) + _2bz * (q1q2 + q3q4) - my) + _2bx * q3 * (_2bx * (q1q3 + q2q4) + _2bz * (0.5f - q2q2 - q3q3) - mz);
        s2 = _2q4 * (2.0f * q2q4 - _2q1q3 - ax) + _2q1 * (2.0f * q1q2 + _2q3q4 - ay) - 4.0f * q2 * (1.0f - 2.0f * q2q2 - 2.0f * q3q3 - az) + _2bz * q4 * (_2bx * (0.5f - q3q3 - q4q4) + _2bz * (q2q4 - q1q3) - mx) + (_2bx * q3 + _2bz * q1) * (_2bx * (q2q3 - q1q4) + _2bz * (q1q2 + q3q4) - my) + (_2bx * q4 - _4bz * q2) * (_2bx * (q1q3 + q2q4) + _2bz * (0.5f - q2q2 - q3q3) - mz);
        s3 = -_2q1 * (2.0f * q2q4 - _2q1q3 - ax) + _2q4 * (2.0f * q1q2 + _2q3q4 - ay) - 4.0f * q3 * (1.0f - 2.0f * q2q2 - 2.0f * q3q3 - az) + (-_4bx * q3 - _2bz * q1) * (_2bx * (0.5f - q3q3 - q4q4) + _2bz * (q2q4 - q1q3) - mx) + (_2bx * q2 + _2bz * q4) * (_2bx * (q2q3 - q1q4) + _2bz * (q1q2 + q3q4) - my) + (_2bx * q1 - _4bz * q3) * (_2bx * (q1q3 + q2q4) + _2bz * (0.5f - q2q2 - q3q3) - mz);
        s4 = _2q2 * (2.0f * q2q4 - _2q1q3 - ax) + _2q3 * (2.0f * q1q2 + _2q3q4 - ay) + (-_4bx * q4 + _2bz * q2) * (_2bx * (0.5f - q3q3 - q4q4) + _2bz * (q2q4 - q1q3) - mx) + (-_2bx * q1 + _2bz * q3) * (_2bx * (q2q3 - q1q4) + _2bz * (q1q2 + q3q4) - my) + _2bx * q2 * (_2bx * (q1q3 + q2q4) + _2bz * (0.5f - q2q2 - q3q3) - mz);
        
        norm = sqrt(s1 * s1 + s2 * s2 + s3 * s3 + s4 * s4);    
        norm = 1.0f/norm;
        s1 *= norm;
        s2 *= norm;
        s3 *= norm;
        s4 *= norm;

        qDot1 = 0.5f * (-q2 * gx - q3 * gy - q4 * gz) - beta * s1;
        qDot2 = 0.5f * (q1 * gx + q3 * gz - q4 * gy) - beta * s2;
        qDot3 = 0.5f * (q1 * gy - q2 * gz + q4 * gx) - beta * s3;
        qDot4 = 0.5f * (q1 * gz + q2 * gy - q3 * gx) - beta * s4;

        /*q1 += qDot1 * deltat;
        q2 += qDot2 * deltat;
        q3 += qDot3 * deltat;
        q4 += qDot4 * deltat;*/
        
        q1 += qDot1 * Filter;
        q2 += qDot2 * Filter;
        q3 += qDot3 * Filter;
        q4 += qDot4 * Filter;      
        
        norm = sqrt(q1 * q1 + q2 * q2 + q3 * q3 + q4 * q4); 
        norm = 1.0f/norm;
        q[0] = q1 * norm;
        q[1] = q2 * norm;
        q[2] = q3 * norm;
        q[3] = q4 * norm;
    }
    
    int Float_to_Int(float numb){
        float n = numb -(int)numb;
        if(n >= 0 && n < 0.5) return (int)numb;
        if(n >= 0.5 && n <= 1) return (int)numb + 1;
        if(n <= 0 && n > -0.5) return (int)numb;
        if(n <= -0.5 && n >= -1) return (int)numb - 1;
        else return 0;
    }  
    
    void CalibIMU(void){
        wait(1);          
        resetMPU9250();  
        calibrateMPU9250(gyroBias, accelBias);   
        wait(2);  
        initMPU9250();    
        initAK8963(magCalibration);  
        wait(2);       
    }

    void BiasIMU(void){
        getAres(); 
        getGres(); 
        getMres();  
        magbias[0] = +470.;  
        magbias[1] = +120.;  
        magbias[2] = +125.;  
    } 
    
    void GetQuaternion(void){                
        readAccelData(accelCount);  
        ax = (float)accelCount[0]*aRes - accelBias[0];  
        ay = (float)accelCount[1]*aRes - accelBias[1];   
        az = (float)accelCount[2]*aRes - accelBias[2];  
        
        readGyroData(gyroCount);  
        gx = (float)gyroCount[0]*gRes - gyroBias[0]; 
        gy = (float)gyroCount[1]*gRes - gyroBias[1];  
        gz = (float)gyroCount[2]*gRes - gyroBias[2];   
        
        readMagData(magCount); 
        mx = (float)magCount[0]*mRes*magCalibration[0] - magbias[0];  
        my = (float)magCount[1]*mRes*magCalibration[1] - magbias[1];  
        mz = (float)magCount[2]*mRes*magCalibration[2] - magbias[2];                                       
    } 
    
    void FinalQuaternion(void){
        yaw   = atan2(2.0f * (q[1] * q[2] + q[0] * q[3]), q[0] * q[0] + q[1] * q[1] - q[2] * q[2] - q[3] * q[3]);   
        pitch = -asin(2.0f * (q[1] * q[3] - q[0] * q[2]));
        roll  = atan2(2.0f * (q[0] * q[1] + q[2] * q[3]), q[0] * q[0] - q[1] * q[1] - q[2] * q[2] + q[3] * q[3]);
        pitch *= 180.0f / PI;
        yaw   *= 180.0f / PI; 
        yaw   -= 0.9f;
        roll  *= 180.0f / PI; 
        
        /*if((q[0]*q[1] + q[2]*q[3]) == 0.5){    //north pole
            pitch = atan2(2*q[1]*q[3]-2*q[0]*q[2], 1 - 2*(q[1] * q[1]) - 2*(q[2] * q[2]));
            yaw = atan2(2*q[0]*q[3]-2*q[1]*q[2], 1 - 2*(q[0] * q[0]) - 2*(q[2] * q[2]));
        }   
        else{
            pitch = 2.0 * atan2(q[0],q[4]);
            yaw = 0;           
        }
        
        if((q[0]*q[1] + q[2]*q[3]) == -0.5){   //south pole
            pitch = atan2(2*q[1]*q[3]-2*q[0]*q[2], 1 - 2*(q[1] * q[1]) - 2*(q[2] * q[2]));
            yaw = atan2(2*q[0]*q[3]-2*q[1]*q[2], 1 - 2*(q[0] * q[0]) - 2*(q[2] * q[2]));
        }   
        else {
            pitch = -2.0 * atan2(q[0],q[4]);
            yaw = 0;        
        }                     
        roll = asin(2*q[0]*q[1] + 2*q[2]*q[3]);*/ 

        
        
        //if(roll < 180 & roll > 0) roll -= 180; 
        //else if(roll > -180 & roll < 0) roll += 180; 
        
        pitch = Float_to_Int(pitch);
        roll = Float_to_Int(roll);
        yaw = Float_to_Int(yaw);    
    }         
};
  
//Definition des variables dans la Class MPU9250, car la definition ne peut pas être faite à l'intérieur  
float MPU9250::gyroBias[3] = {0, 0, 0};
float MPU9250::accelBias[3] = {0, 0, 0}; // Bias corrections for gyro and accelerometer 
float MPU9250::magCalibration[3] = {0, 0, 0};
float MPU9250::magbias[3] = {0, 0, 0};
float MPU9250::GyroMeasError = PI * (60.0f / 180.0f);     // gyroscope measurement error in rads/s (start at 60 deg/s), then reduce after ~10 s to 3
float MPU9250::beta = sqrt(3.0f / 4.0f) * GyroMeasError;  // compute beta
float MPU9250::GyroMeasDrift = PI * (1.0f / 180.0f);      // gyroscope measurement drift in rad/s/s (start at 0.0 deg/s/s)
float MPU9250::zeta = sqrt(3.0f / 4.0f) * GyroMeasDrift;  // compute zeta, the other free parameter in the Madgwick scheme usually set to a small or zero value
float MPU9250::q[4] = {1.0f, 0.0f, 0.0f, 0.0f};           // vector to hold quaternion
float MPU9250::eInt[3] = {0.0f, 0.0f, 0.0f};              // vector to hold integral error for Mahony method
float MPU9250::Filter = 0.05;