#include "uLCD_4DGL.h"
#include "bullet.h"
#include "game_synchronizer.h"
#include "globals.h"
#include "math.h"
#include "playSound.h"

extern Serial pc;

extern Game_Synchronizer sync;

// Initialize the bullet. Don't have to do much here.
// Keep a pointer to this bullet's tank.
// Set the speed, and default the bullet to not in_flight.
Bullet::Bullet(Tank* t) {
    tank = t;
    speed = 30;
    powerups = 2;
    power = 0;
    in_flight = false;
}

// If in_flight, do nothing. Otherwise,
// set the in_flight flag, and initialize values needed for
// the trajectory calculations. (x0, y0), (vx0, vy0), time
// Hint: tank->barrel_end(...) is useful here.
void Bullet::shoot(void) {
    if (!in_flight) {
        in_flight = true;
        tank->barrel_end(&x0, &y0);
        x = x0;
        y = y0;
        vx0 = speed*cos(tank->barrel_theta);
        vy0 = speed*sin(tank->barrel_theta);
        vx = vx0;
        vy = vy0;
        bounce = 0;
        time = 0;
    }
}

void Bullet::changeSky(int SKY_COLOR2) {
    SKY_COLOR = SKY_COLOR2;
}

// power up increases explosion size
// you get 2 powerups per round
void Bullet::powerUp(void) {
    if (powerups > 0) {
        // play sound for powerup
        playSound("/sd/wavfiles/boing2.wav");
        powerups = powerups - 1;
        power = power + 1;
    }
}

// If the bullet is in flight, calculate its new position
// after a time delta dt.
void Bullet::update_position(float dt) {
    vx = vx;
    vy = vy - 9.81*dt;
    x = floor(x0 + vx*time);
    y = floor(y0 + vy*time - 0.5*9.81*time*time);
}

void boing(void const *argument) {
    playSound("/sd/wavfiles/boing2.wav");
}

int Bullet::time_step(float dt) {
    // If the bullet hasn't hit anything, 
    // redraw the bullet at its new location. 
    // If it has hit something (obstacle, tank, edge of the screen), 
    // set the in_flight flag back to false, explode the nearby area,
    // and return one of the following codes.
    //
    // return codes:
    //      BULLET_NO_COLLISION: no collision
    //      BULLET_OFF_SCREEN:   off the side of the screen
    //      Otherwise, return the color you've hit in 16bpp format.
    if (in_flight) {
        int x_old = x;
        int y_old = y;
        // old pixel
        sync.pixel(x_old, y_old, SKY_COLOR);
        sync.update();
        time = time + dt;
        update_position(dt);
        // new pixel
        int color = sync.read_pixel(x, y);
        if (x_old == x && y_old == y) {
            // no change
            sync.pixel(x, y, tank->tank_color);
            return BULLET_NO_COLLISION;
        } else if (x < 0 || x > 128 || y < 0 || y > 128) {
            // out of bounds
            in_flight = false;
            return BULLET_OFF_SCREEN;
        } else if (sync.pixel_eq(color, SKY_COLOR)) {
            // travelling in the sky
            sync.pixel(x, y, tank->tank_color);
            return BULLET_NO_COLLISION;
        } else if (!sync.pixel_eq(color, SKY_COLOR)) {
            // hit
            if (sync.pixel_eq(color, GND_COLOR)) {
                bounce = bounce + 1;
                // bounce
                if (bounce < 3) {
                    vy = vy/2.0;
                    sync.pixel(x, y, tank->tank_color);
                    return BULLET_NO_COLLISION;
                }
            }
            // hit
            in_flight = false;
            int i = 1;
            // powerup increases explosion size
            for(i = 1; i < 5 + (power * 3); i++) {
                sync.filled_rectangle(x-i, y-i, x+i, y+i, tank->tank_color);
                sync.update();
                wait(0.1);
            }
            power = 0;
            // make everything the sky afterwards
            sync.filled_rectangle(x-i-1, y-i-1, x+i-1, y+i-1, SKY_COLOR);
            sync.update();
            // play the sound
            playSound("/sd/wavfiles/splooge.wav");
            return color;
        }
    }
    return BULLET_NO_COLLISION;
}