#ifndef SmartLab_MuRata_MuRata
#define SmartLab_MuRata_MuRata

#include "Payload.h"
#include "UARTFrame.h"

#include "UARTConfig.h"
#include "DHCPConfig.h"
#include "SoftAPConfig.h"

#include "CMDCode.h"
#include "SNICCode.h"
#include "WIFICode.h"

#include "BSSType.h"
#include "CommandID.h"
#include "DHCPMode.h"
#include "HTTPContent.h"
#include "IPAddress.h"
#include "ResetCode.h"
#include "ScanType.h"
#include "SecurityMode.h"
#include "SocketSentOption.h"
#include "SubCommandID.h"
#include "WIFIInfo.h"
#include "WIFINetwork.h"
#include "WIFINetworkDetail.h"
#include "WIFIInterface.h"
#include "WIFIStatusCode.h"
#include "WPSMode.h"

#include "SSIDRecordIndication.h"
#include "TCPStatusIndication.h"
#include "SocketReceiveInidcation.h"
#include "UDPReceivedIndication.h"
#include "WIFIConnectionIndication.h"
#include "HTTPResponseIndication.h"

#include "WIFIStatusResponse.h"
#include "InitializationResponse.h"
#include "SendFromSocketResponse.h"
#include "CreateSocketResponse.h"
#include "SocketStartReceiveResponse.h"
#include "DHCPInfoResponse.h"
#include "HTTPResponse.h"

#include "mbed.h"

namespace SmartLabMuRata
{
class MuRata
{
private :
    SSIDRecordIndication _SSIDRecordIndication;
    WIFIConnectionIndication _WIFIConnectionIndication;
    TCPStatusIndication _TCPStatusIndication;
    SocketReceiveInidcation _SocketReceiveInidcation;
    UDPReceivedIndication _UDPReceivedIndication;
    HTTPResponseIndication _HTTPResponseIndication;

    WIFIStatusResponse _WIFIStatusResponse;
    InitializationResponse _InitializationResponse;
    SendFromSocketResponse _SendFromSocketResponse;
    DHCPInfoResponse _DHCPInfoResponse;
    SocketStartReceiveResponse _SocketStartReceiveResponse;
    CreateSocketResponse _CreateSocketResponse;
    HTTPResponse _HTTPResponse;

    IPAddress ip;

    Serial * serial;
    Payload _payload;
    UARTFrame _frame;

    void Send();

    bool FrameReceive();

    CreateSocketResponse * SNIC_CreateSocket(const SubCommandID subID, const bool bind = false, IPAddress * localIP = NULL, const int localPort = 0);

    /// Time out when there is no data received
    Timer timer;

public:

    static const int DEFAULT_BAUDRATE = 921600;

    // ##############################################################################
    //                                indication
    // ##############################################################################

    SSIDRecordIndication * Get_ScanResultIndication();

    WIFIConnectionIndication * Get_WiFiStatusIndication();

    /**
    * This event reports the Murata module power up reason. Murata module is ready for serial communication after this report is generated.
    * @return N0_Indication when no response is received.
    */
    ResetCode Get_PowerUpIndication();

    TCPStatusIndication * Get_TcpConnectionStatusIndication();

    SocketReceiveInidcation * Get_SocketReceiveIndication();

    UDPReceivedIndication * Get_UDPReceiveIndication();

    HTTPResponseIndication * Get_HTTPResponseIndication();

    // ##############################################################################
    //                                constructor
    // ##############################################################################

    /**
    * Constructor
    * @param tx mbed pin name for UART
    * @param rx mbed pin name for UART
    */
    MuRata(PinName tx, PinName rx);

    /**
    * Constructor
    * @param tx mbed pin name for UART
    * @param rx mbed pin name for UART
    * @param baudrate serial baud rate
    * @param bits
    * @param parity
    * @param stop_bits
    */
    MuRata(PinName tx, PinName rx, int baudrate ,int bits=8, Serial::Parity parity=SerialBase::None, int stop_bits=1);


    // ##############################################################################
    //                                   GEN
    // ##############################################################################

    /**
    * SNIC firmware has a built in version string. Use this command to retrieve the version info.
    * @returns return null when timeout
    */
    const char * GEN_GetFirmwareVersionInfo();

    /**
    * This command restores the data stored in NVM to factory default values. Any web page update is not affected by this command.
    * A soft reset will be performed automatically after the NVM has been restored.
    * Application needs to send WIFI_GET_STATUS_REQ or SNIC_GET_DHCP_INFO_REQ commands to determine the new state of the Murata module.
    */
    CMDCode GEN_RestoreNVMtoFactoryDefault();

    /**
    * This command resets the module. Application needs to send WIFI_GET_STATUS_REQ or SNIC_GET_DHCP_INFO_REQ commands to determine the new state of the module after the reset.
    */
    CMDCode GEN_SoftReset();

    /**
    * This command configures the UART interface. The specified parameters are saved into the NVM and they are used for the specified UART interface in subsequent powerups.
    */
    CMDCode GEN_UARTConfiguration(UARTConfig * config);

    // ##############################################################################
    //                                   WiFi
    // ##############################################################################

    /**
    * This command turns on Wifi on module.
    * The default country code is “US”, which is one of the startup parameters in NVM. If the WIFI_ON_REQ has no intention of changing the country code, put 0x0000 in the two-byte Country code, so that the firmware will use the country code configured in NVM.
    * The module supports both soft AP mode and STA mode at the same time. The module has reserved flash space (NVM) to store startup parameters for both the soft AP and the STA. Only STA’s parameters can be dynamically changed at run time.
    * Turning on WiFi would cause the following to happen:
    * The following operations occur using the parameters specified in the NVM if the AP mode is enabled.
    * 1. Turn on the soft AP
    * 2. Starts DNS server, DHCP server and HTTP server. The HTTP server provides a means for configuring the WLAN access parameters for the STA.
    * Turn on the STA. If the NVM has valid startup parameters, the STA will try to join the saved SSID with saved authentication information. The NVM also stores whether DHCP or static IP is used for STA. If DHCP is used, DHCP client will be started. After a successful join, STA’s IP will be configured according to the NVM.
    * By default, the soft AP is turned on to allow user to use a WiFi enabled computer to connect to the soft AP, and instructs the STA to join one of the surrounding APs. So WiFi is turned on by default and this command is not required at startup.
    */
    WIFICode WIFI_TurnOn();

    /**
    * This command turns off Wifi on module. Turning off WiFi causes the following to happen:
    * 1. Turn off the soft AP, including shutting down DNS server, DHCP server and HTTP server.
    * 2. Disconnect STA from any joined network, and close all sockets opened by application.
    */
    WIFICode WIFI_TurnOff();

    /**
    * This command turns on or off the soft AP. The WIFI_ON(OFF)_REQ controls both the soft AP and STA at the same time, while this command only controls the soft AP.
    * An example use case is, the soft AP (and its web server) is turned on at startup to configure STA to join a network and is no longer needed after the STA is connected.
    * WIFI_AP_CTRL_REQ can be used to turn the soft AP off.
    * OnOff = 0 indicates AP is to be turned off. The rest of the parameters are ignored.
    * OnOff = 1 indicates turning on soft AP using existing NVM parameters,
    * OnOff = 2 indicates turning on AP with the parameters provided. If the soft AP is already on, it is first turned off.
    * Persistency=1 indicates the soft AP’s on/off state and parameters (if OnOff = 2) will be saved in NVM. For example, if OnOff =0 and Persistency=1, the soft AP will not be turned on after a reset.
    */
    WIFICode WIFI_SoftAPControl(SoftAPConfig * config);

    /**
    * This command instructs module to associate to a network.
    */
    WIFICode WIFI_AssociateNetwork(WIFINetwork * AP);

    /**
    * This command instructs the module to disconnect from a network.
    * Upon a successful reception of the command, the module disconnects from associated network. Sockets opened by application are not closed.
    */
    WIFICode WIFI_DisconnectNetwork();

    /**
    * This command queries the WiFi status from module. This command should be called by application after startup to determine the WiFi status since the module may have joined an AP automatically based on NVM parameters (see 6.1).
    * </summary>
    */
    WIFIStatusResponse * WIFI_GetStatus(const WIFIInterface WiFiInterface);

    /**
    * This command requests the reporting of the current RSSI from module’s STA interface
    * @returns RSSI in dBm. 127 means unspecified value
    */
    int8_t WIFI_GetRSSI();

    /**
    * This command requests the module to use WPS to join the network. Two methods are supported: push button and pin-based configuration.
    * If Mode is 1, Pin value must be present. Pin value is NUL terminated ASCII string. Pin string length of 0, 4, 7, or 8 is valid. When length is 0, the module will use the WPS default pin configured in the NVM by using the SNIC monitor. When length is 8, the 8th digit must be the correct checksum of the first 7 digits. The pin checksum calculation method can be found from the Internet. When the length is 7, the module firmware will calculate the checksum automatically. When the length is 4, no checksum is required.
    * Upon a successful reception of the command, the module tries to associate to a network using the WPS configuration specified. Upon a successful completion of the join process, the SSID and authentication parameters will be saved in NVM which will be used in subsequent power up (see 6.1).
    */
    WIFICode WIFI_StartWPSProcess(const WPSMode mode, const char * pin = NULL, int pinLength = 0);

    /**
    * Upon a successful reception of the command, the module starts to scan. The response will indicate only WIFI_SUCCESS if no error. Actual scan result shall be sent from module as multiple indications defined in WIFI_SCAN_RESULT_IND
    * </summary>
    * @param scan
    * @param bss
    * @param BSSID 6 bytes MAC address of the AP or STA
    * @param channelList up to 10 array elements
    * @param SSID string for the AP or STA SSID, up to 32 bytes
    */
    WIFICode WIFI_ScanNetworks(const ScanType scan, const BSSType bss);

    // ##############################################################################
    //                                   SNIC
    // ##############################################################################

    /**
    * This command initializes the SNIC networking framework on module. TCP/UDP socket communication may be performed only after this command is called.
    * The Default receive buffer size is the default maximum size of receive buffer in the module. If 0 is specified, a system defined value (2048) will be used. If there is a Receive buffer size field in other commands, then it must be less than or equal to the Default receive buffer size. If the Receive buffer size in any of those commands is 0, the Default receive buffer size will be used.
    * @param  receiveBufferSize Upon a successful reception of the command, the module sends to the host the following response. If user specified Default receive buffer size is bigger than what the module can handle, the system defined value will be returned in the response; otherwise, user specified Default receive buffer size will be retuned. Maximum number of UDP and TCP sockets supported by module will also be returned.
    */
    InitializationResponse * SNIC_Initialization(int receiveBufferSize = 0);

    /**
    * This command closes the SNIC networking framework on module. It should cleanup resources for socket communication on module. If some sockets are not closed, this command will close all of them. No more network communication can be performed until SNIC_INIT_REQ is called.
    */
    SNICCode SNIC_Cleanup();

    /**
    * In TCP server case, Socket is the socket number returned by SNIC_TCP_CLIENT_SOCKET_IND. In TCP client case, Socket can be either from SNIC_CONNECT_TO_TCP_SERVER_RSP, or from the SNIC_TCP_CONNECTION_STATUS_IND with SNIC_CONNECTION_UP status. In UDP case, Socket is the socket number returned by SNIC_UDP_CREATE_SOCKET_REQ and it must be in connected mode.
    * A success response of this command does not guarantee the receiver receives the packet. If error occurs, a SNIC_TCP_CONNECTION_STATUS_IND with SNIC_SOCKET_CLOSED will be sent to the application in TCP case. No indication will be sent in UDP case.
    * Option is the action module will perform to the socket after the send operation. Use it when application is sure to close or shutdown the connection after sending. The effect is the same as using SNIC_CLOSE_SOCKET_REQ, but round-trip UART traffic is reduced.
    */
    SendFromSocketResponse * SNIC_SendFromSocket(const char SocketID, const SocketSentOption option, const char * payload, int offset, int length);

    /**
    * This command instructs the module to close a socket.
    */
    SNICCode SNIC_SloseSocket(const char SocketID);

    /**
    * This command queries the DHCP information for a particular interface.
    */
    DHCPInfoResponse * SNIC_GetDHCPInfo(const WIFIInterface wifiInterface);

    /**
    * This command converts a remote host name to IP address.
    * Interface number is either 0 or 1. 0 indicates STA interface. 1 indicates soft AP interface. Currently only STA interface is supported.
    * If multiple SNIC_RESOLVE_NAME_REQ’s need to be sent, it is required they be sent sequentially due to resource limitation. If the name is not resolved, it takes up to 15 seconds for the failure response to come back. While waiting for the response, host application can send other commands (except for SNIC_RESOLVE_NAME_REQ and SNIC_SEND_ARP_REQ).
    */
    IPAddress * SNIC_ResolveHostName(const char * host);

    /**
    * This command instructs module configure the mechanism for obtaining the IP address.
    * DHCP mode specifies how the address is assigned for the interface.
    *  0: interface is assigned the static IP, NetMask and Gateway IP. First IP and Last IP are not present. Any active DHCP client or server is stopped.
    *  1: STA interface uses DHCP to obtain the address. All subsequent fields are not present. STA DHCP client is started if necessary.
    *  2: only for AP interface. If the soft AP is not started or SNIC_INIT_REQ is not done, this command fails. Otherwise, this command stops the HTTP server, DNS server and DHCP server if configured, and restarts them with new parameters. It assigns IP for clients in range [First IP, Last IP] within the subnet mask. The AP itself is assigned the address within the same subnet specified by IP which must not be in the range of [First IP, Last IP]. The value of GTW IP and IP should be the same. If there are clients connected to the soft AP before this command, make sure the clients reconnect to the soft AP after this command.
    */
    SNICCode SNIC_ConfigureDHCPorStaticIP(DHCPConfig * config);

    /**
    * If the connect attempt is immediately completed, the response will contain SNIC_SUCCESS status, with the actual Receive buffer size.
    * If the connect attempt is not immediately completed, the response will have the SNIC_COMMAND_PENDING status. The Timeout value is the time (in seconds) the module will wait before aborting the connection attempt. If timeout occurs, the SNIC_TCP_CONNECTION_STATUS_IND indication with SNIC_TIMEOUT status will be sent to the application. If connection is successful before timeout, the SNIC_TCP_CONNECTION_STATUS_IND with SNIC_CONNECTION_UP status will be sent to the application. Timeout value should be non-zero.
    * @param remoteHost
    * @param port
    * @param timeout in seconds
    * @param receiveBufferSize Receive buffer size is the maximum packet size the application wants to receive per transmission. It must be less than or equal to the Default receive buffer size from SNIC_INIT_REQ in the module. If it is 0 or exceeds the system capability, the Default receive buffer size is returned.
    */
    SocketStartReceiveResponse * SNIC_ConnectTCPServer(const char SocketID, IPAddress * remoteIP, const int remotePort, const char timeout, const int receiveBufferSize = 0);

    /**
    * If Bind option is 0, the socket will not be bound, and Local IP address and Local port should not be present. Otherwise, it will be bound to Local IP address and Local port specified. 0x0 for IP or port are valid, which means system assigned. Port number 5000 is reserved for internal use.
    * the socket number must get and store separately, since the response payload may change
    * @param bing do not bing if this tcp socket is used as a client
    * @param localIP
    * @param localPort
    */
    CreateSocketResponse * SNIC_CreateTCPSocket(const bool bind = false, IPAddress * localIP = NULL, const int localPort = 0);

    /**
    * If Bind option is 0, the socket will not be bound, and Local IP address and Local port should not be present. Otherwise, it will be bound to Local IP address and Local port specified. 0x0 for IP or port are valid, which means system assigned. Port number 5000 is reserved for internal use.
    * the socket number must get and store separately, since the response payload may change
    * @param bind
    * @param localIP
    * @param localPort
    */
    CreateSocketResponse * SNIC_CreateUDPSocket(const bool bind = false, IPAddress * localIP = NULL, const int localPort = 0);

    /**
    * The Socket should have been created by command SNIC_UDP_CREATE_SOCKET_REQ. The same socket can be used in SNIC_UDP_SEND_FROM_SOCKET_REQ command, so that send and receive can be done via the same socket (port). The application is responsible to close the socket using SNIC_CLOSE_SOCKET_REQ.
    * Receive buffer size is the maximum packet size the application wants to receive per transmission. It must be less than or equal to the Default receive buffer size from SNIC_INIT_REQ in the module. If 0 or exceeds the system capability, the Default receive buffer size will be used and returned in the response.
    * After this command, the Socket can receive any UDP sender with connected mode or non-connected mode. The module will generate SNIC_UDP_RECV_IND indication for incoming data, which includes sender’s IP and port info.
    * But if this Socket is later connected to a peer UDP server by SNIC_UDP_SEND_FROM_SOCKET_REQ with Connection mode set to1, the module will generate SNIC_CONNECTION_RECV_IND indication without the sender’s IP and port info. See Section 5.19. After that, this Socket will only be able to receive from the one sender it connects to.
    */
    SocketStartReceiveResponse * SNIC_StartUDPReceive(const char SocketID, const int receiveBufferSize = 0);

    /**
    * A socket will be created for sending the packet out through the default network connection, but will be closed after the transmission. This command can be used when the application just wants to send out one packet to peer, and it also does not expect to receive any packets from peer.
    */
    SendFromSocketResponse * SNIC_SendUDPPacket(IPAddress * remoteIP, const int remotePort, const char * payload, const int offset, const int length);

    /**
    * The Socket should have been created by command SNIC_UDP_CREATE_SOCKET_REQ. If SNIC_UDP_START_RECV_REQ is not called on the socket, the application can only send out UDP packet from this socket. If SNIC_UDP_START_RECV_REQ has been called for this socket, the application can send and receive UDP packets from the socket. This implies the application can send and receive packets from the same local port. The application is responsible to close the socket using SNIC_CLOSE_SOCKET_REQ.
    * If Connection mode is 1, the module will first connect to the UDP server then send data. Since the socket is still connected after the call, application can send subsequent data using another command SNIC_SEND_FROM_SOCKET_REQ.
    * The benefit of the connected mode is that subsequent send can use SNIC_SEND_FROM_SOCKET_REQ, which does not require the receiver’s IP and port every time, and thus reduces overhead. If this socket is also used to receive by calling SNIC_UDP_START_RECV_REQ, the receive indication to the host will also omits the sender IP and port info, further reducing overhead.
    * <param name="remoteIP"></param>
    * <param name="remotePort"></param>
    * <param name="SocketID"></param>
    * <param name="connectServer"></param>
    * <param name="payload"></param>
    * <param name="offset"></param>
    * <param name="length"></param>
    */
    SendFromSocketResponse * SNIC_SendUDPFromSocket(IPAddress * remoteIP, const int remotePort, const char SocketID, const bool connectServer, const char * payload, int offset, const int length);

    /**
    * This command instructs the module to send a HTTP request packet to the remote HTTP server.
    * Post content can be binary. So even if it is text string, it should not contain NUL at the end. The most significant bit of Post content length is reserved to indicate if there is more data to send. If there is more data to send (as indicated by MSBit=1 in the content length), host application should use another API (SNIC_HTTP_MORE_REQ) to send the rest of the data until it is finished. If this bit is set to 1, then the “Transfer-Encoding” in the HTTP request will be set to “chunked” by SNIC. For GET method, the highest bit of Content length must be set to 0 (not chunked).
    * For HTTP request with chunked encoding, status code of SNIC_SUCCESS in the response only means the HTTP request has been sent. After one or more subsequent SNIC_HTTP_MORE_REQ/RSPs, the last SNIC_HTTP_MORE_RSP with HTTP status code will be sent to host containing the data from HTTP server.
    * The most significant bit of Content length is reserved to indicate if there is more response data to send to the host. If there is more data to send (Content length MSBit=1), module uses SNIC_HTTP_RSP_IND to send the rest of the data until it is finished, i.e., when this bit is 1, the host application should continue to receive SNIC_HTTP_RSP_IND, until this bit is 0.
    * The Content length is limited by the receive buffer size specified in SNIC_INIT_REQ and the system resource at that moment.
    * <param name="content"></param>
    * <param name="isHTTPS"></param>
    * <param name="chunked"></param>
    */
    HTTPResponse * SNIC_SendHTTPRequest(HTTPContent * content, const bool isHTTPS = false, const bool chunked = false);

    /**
    * This command instructs the module to send a subsequent HTTP request packet to the remote HTTP server if the initial SNIC_HTTP_REQ cannot finish the packet due to size or other consideration. It is used when the send method is POST.
    * <param name="content"></param>
    * <param name="chunked"></param>
    */
    HTTPResponse * SNIC_SendHTTPMoreRequest(HTTPContent * content, const bool chunked = false);

    /**
    * If Bind option is 0, the socket will not be bound, and Local IP address and Local port should not be present. Otherwise, it will be bound to Local IP address and Local port specified. 0x0 for IP or port are valid, which means system assigned. Port number 5000 is reserved for internal use.
    * the socket number must get and store separately, since the response payload may change
    * </summary>
    * <param name="bing">do not bing if this tcp socket is used as a client</param>
    * <param name="localIP"></param>
    * <param name="localPort"></param>
    */
    CreateSocketResponse * SNIC_CreateAdvancedTLSTCP(const bool bind, IPAddress * localIP = NULL, const int localPort = 0);

    /**
    * If Bind option is 0, the socket will not be bound, and Local IP address and Local port should not be present. Otherwise, it will be bound to Local IP address and Local port specified. 0x0 for IP or port are valid, which means system assigned. Port number 5000 is reserved for internal use.
    * the socket number must get and store separately, since the response payload may change
    * </summary>
    * <param name="bind"></param>
    * <param name="localIP"></param>
    * <param name="localPort"></param>
    */
    CreateSocketResponse * SNIC_CreateSimpleTLSTCP(const bool bind, IPAddress * localIP = NULL, const int localPort = 0);

};
}

#endif