#include "MuRata.h"

using namespace SmartLabMuRata;

MuRata::MuRata(PinName tx, PinName rx)
    : _SSIDRecordIndication(NULL),_WIFIConnectionIndication(NULL),_TCPStatusIndication(NULL),_SocketReceiveInidcation(NULL),
      _UDPReceivedIndication(NULL),_HTTPResponseIndication(NULL),
      _WIFIStatusResponse(NULL),_InitializationResponse(NULL),_SendFromSocketResponse(NULL),_DHCPInfoResponse(NULL),_SocketStartReceiveResponse(NULL),_CreateSocketResponse(NULL),_HTTPResponse(NULL)
{
    serial = new Serial(tx,rx);
    serial->format();
    serial->baud(DEFAULT_BAUDRATE );
}

MuRata::MuRata(PinName tx, PinName rx, int baudrate ,int bits, Serial::Parity parity, int stop_bits)
    : _SSIDRecordIndication(NULL),_WIFIConnectionIndication(NULL),_TCPStatusIndication(NULL),_SocketReceiveInidcation(NULL),
      _UDPReceivedIndication(NULL),_HTTPResponseIndication(NULL),
      _WIFIStatusResponse(NULL),_InitializationResponse(NULL),_SendFromSocketResponse(NULL),_DHCPInfoResponse(NULL),_SocketStartReceiveResponse(NULL),_CreateSocketResponse(NULL),_HTTPResponse(NULL)
{
    serial = new Serial(tx,rx);
    serial->format(bits, parity, stop_bits);
    serial->baud(baudrate);
}

SSIDRecordIndication * MuRata::Get_ScanResultIndication()
{
    if (FrameReceive())
        if (_payload.GetResponseFlag() == Request_Indication
                && _frame.GetCommandID() == CMD_ID_WIFI
                && _payload.GetSubCommandID() == WIFI_SCAN_RESULT_IND) {
            _SSIDRecordIndication.SetPayload(&_payload);
            return &_SSIDRecordIndication;
        }

    return NULL;
}

WIFIConnectionIndication * MuRata::Get_WiFiStatusIndication()
{
    if (FrameReceive())
        if (_payload.GetResponseFlag() == Request_Indication
                && _frame.GetCommandID() == CMD_ID_WIFI
                && _payload.GetSubCommandID() == WIFI_NETWORK_STATUS_IND) {
            _WIFIConnectionIndication.SetPayload(&_payload);
            return &_WIFIConnectionIndication;
        }

    return NULL;
}

ResetCode MuRata::Get_PowerUpIndication()
{
    if (FrameReceive())
        if (_payload.GetResponseFlag() == Request_Indication
                && _frame.GetCommandID() == CMD_ID_GEN
                && _payload.GetSubCommandID() == GEN_PWR_UP_IND)
            return (ResetCode)(_payload.GetData()[2] << 8 | _payload.GetData()[3]);

    return N0_Indication;
}

TCPStatusIndication * MuRata::Get_TcpConnectionStatusIndication()
{
    if (FrameReceive())
        if (_payload.GetResponseFlag() == Request_Indication
                && _frame.GetCommandID() == CMD_ID_SNIC
                && _payload.GetSubCommandID() == SNIC_TCP_CONNECTION_STATUS_IND) {
            _TCPStatusIndication.SetPayload(&_payload);
            return &_TCPStatusIndication;
        }

    return NULL;
}

SocketReceiveInidcation * MuRata::Get_SocketReceiveIndication()
{
    if (FrameReceive())
        if (_payload.GetResponseFlag() == Request_Indication
                && _frame.GetCommandID() == CMD_ID_SNIC
                && _payload.GetSubCommandID() == SNIC_CONNECTION_RECV_IND) {
            _SocketReceiveInidcation.SetPayload(&_payload);
            return &_SocketReceiveInidcation;
        }

    return NULL;
}

UDPReceivedIndication * MuRata::Get_UDPReceiveIndication()
{
    if (FrameReceive())
        if (_payload.GetResponseFlag() == Request_Indication
                && _frame.GetCommandID() == CMD_ID_SNIC
                && _payload.GetSubCommandID() == SNIC_UDP_RECV_IND) {
            _UDPReceivedIndication.SetPayload(&_payload);
            return &_UDPReceivedIndication;
        }

    return NULL;
}

HTTPResponseIndication * MuRata::Get_HTTPResponseIndication()
{
    if (FrameReceive())
        if (_payload.GetResponseFlag() == Request_Indication
                && _frame.GetCommandID() == CMD_ID_SNIC
                && _payload.GetSubCommandID() == SNIC_HTTP_RSP_IND) {
            _HTTPResponseIndication.SetPayload(&_payload);
            return &_HTTPResponseIndication;
        }

    return NULL;
}

void MuRata::Send()
{
    _frame.SetACKRequired(false);
    _payload.SetResponseFlag(Request_Indication);
    serial->putc(UARTFrame::SOM);

    serial->putc(_frame.GetL0() | 0x80);
    serial->putc(_frame.GetL1() | 0x80 | (_frame.GetACKRequired() ? 0x40 : 0x00));

    serial->putc(_frame.GetCommandID() | 0x80);

    for (int i = 0; i < _payload.GetPosition(); i++)
        serial->putc(_payload.GetData()[i]);

    serial->putc(_frame.GetChecksum() | 0x80);

    serial->putc(UARTFrame::EOM);
}

bool MuRata::FrameReceive()
{
    timer.reset();
    timer.start();

    while (serial->readable() <= 0) {
        if (timer.read_ms() > 1000) {
            timer.stop();
            return false;
        }
    }
    timer.stop();

    int value = serial->getc();
    while (value != UARTFrame::SOM)
        value = serial->getc();

    _frame.SetL0(serial->getc());
    _frame.SetL1(serial->getc());

    _frame.SetCommandID(serial->getc());

    int _size = _frame.GetPayloadLength();

    _payload.Allocate(_size);
    while (_payload.GetPosition() < _size)
        _payload.SetContent(serial->getc());

    _frame.SetChecksum(serial->getc());

    if (serial->getc() == UARTFrame::EOM && _frame.VerifyChecksum())
        return true;
    else return false;
}

const char * MuRata::GEN_GetFirmwareVersionInfo()
{
    _payload.Rewind();
    _payload.SetSubCommandID(GEN_FW_VER_GET_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);

    _frame.SetCommandID(CMD_ID_GEN);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive()) {
        if (_frame.GetCommandID() == CMD_ID_GEN && _payload.GetSubCommandID() == GEN_FW_VER_GET_REQ) {
            if (_payload.GetData()[2] == GEN_SUCCESS) {
                // set the null for the string
                _payload.GetData()[4 + _payload.GetData()[3]] = 0x00;
                return _payload.GetData() + 4;
            }
        }
    }
    return NULL;
}

CMDCode MuRata::GEN_RestoreNVMtoFactoryDefault()
{
    _payload.Rewind();
    _payload.SetSubCommandID(GEN_RESTORE_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);

    _frame.SetCommandID(CMD_ID_GEN);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_GEN && _payload.GetSubCommandID() == GEN_RESTORE_REQ)
            return (CMDCode)_payload.GetData()[2];

    return GEN_NORESPONSE;
}

CMDCode MuRata::GEN_SoftReset()
{
    _payload.Rewind();
    _payload.SetSubCommandID(GEN_RESET_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);

    _frame.SetCommandID(CMD_ID_GEN);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_GEN && _payload.GetSubCommandID() == GEN_RESET_REQ)
            return (CMDCode)_payload.GetData()[2];

    return GEN_NORESPONSE;
}

CMDCode MuRata::GEN_UARTConfiguration(UARTConfig * config)
{
    _payload.Rewind();
    _payload.SetSubCommandID(GEN_UART_CFG_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(config->GetValue(), 0, 6);

    _frame.SetCommandID(CMD_ID_GEN);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_GEN && _payload.GetSubCommandID() == GEN_UART_CFG_REQ)
            return (CMDCode)_payload.GetData()[2];

    return GEN_NORESPONSE;
}


WIFICode MuRata::WIFI_TurnOn()
{
    _payload.Rewind();
    _payload.SetSubCommandID(WIFI_ON_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    /*
     * Country code is a 2-character ASCII string. E.g., “US” = the United States. For the complete list, see Appendix A. The default country code is “US”, which is one of the startup parameters in NVM. If the WIFI_ON_REQ has no intention of changing the country code, put 0x0000 in the two-byte Country code, so that the firmware will use the country code configured in NVM.
     */
    _payload.SetContent(0x00);
    _payload.SetContent(0x00);

    _frame.SetCommandID(CMD_ID_WIFI);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_WIFI && _payload.GetSubCommandID() == WIFI_ON_REQ)
            return (WIFICode)_payload.GetData()[2];

    return WIFI_NORESPONSE;
}

WIFICode MuRata::WIFI_TurnOff()
{
    _payload.Rewind();
    _payload.SetSubCommandID(WIFI_OFF_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);

    _frame.SetCommandID(CMD_ID_WIFI);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_WIFI && _payload.GetSubCommandID() == WIFI_OFF_REQ)
            return (WIFICode)_payload.GetData()[2];

    return WIFI_NORESPONSE;
}

/*
 * Parameters are as follows:
 * UINT8 Request Sequence
 * UINT8 Onoff
 * UINT8 Persistency
 * UINT8 SSID [up to 33]
 * UINT8 Channel
 * UINT8 Security mode
 * UINT8 Security key length (0-64)
 * … Security key [ ]
 * OnOff = 0 indicates AP is to be turned off. The rest of the parameters are ignored.
 * OnOff = 1 indicates turning on soft AP using existing NVM parameters,
 * OnOff = 2 indicates turning on AP with the parameters provided. If the soft AP is already on, it is first turned off.
 * Persistency=1 indicates the soft AP’s on/off state and parameters (if OnOff = 2) will be saved in NVM. For example, if OnOff =0 and Persistency=1, the soft AP will not be turned on after a reset.
 */
WIFICode MuRata::WIFI_SoftAPControl(SoftAPConfig * config)
{
    _payload.Rewind();
    _payload.SetSubCommandID(WIFI_AP_CTRL_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);

    _payload.SetContent(config->GetOnOffStatus());
    _payload.SetContent(config->GetPersistency());
    if (config->GetOnOffStatus() == SoftAPConfig::ON_PARAMETERS) {
        const char * ssid = config->GetSSID();
        int length = strlen(ssid);
        _payload.SetContent(ssid, 0 , length);
        _payload.SetContent(0x00);
    }
    _payload.SetContent(config->GetChannel());
    _payload.SetContent(config->GetSecurityMode());

    const char * key = config->GetSecurityKey();
    int keyLength = 0;
    if (key != NULL)
        keyLength = strlen(key);

    _payload.SetContent(keyLength);
    if (config->GetSecurityMode() != WIFI_SECURITY_OPEN && keyLength > 0)
        _payload.SetContent(key, 0 , keyLength);

    _frame.SetCommandID(CMD_ID_WIFI);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_WIFI && _payload.GetSubCommandID() == WIFI_AP_CTRL_REQ)
            return (WIFICode)_payload.GetData()[2];

    return WIFI_NORESPONSE;
}

WIFICode MuRata::WIFI_AssociateNetwork(WIFINetwork * AP)
{
    _payload.Rewind();
    _payload.SetSubCommandID(WIFI_JOIN_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(AP->GetSSID(), 0, strlen(AP->GetSSID()));
    _payload.SetContent(0x00);

    _payload.SetContent(AP->GetSecurityMode());

    int keyLength = strlen(AP->GetSecurityKey());
    _payload.SetContent(keyLength);
    if (keyLength > 0)
        _payload.SetContent(AP->GetSecurityKey(), 0, keyLength);

    if (AP->GetBSSID() != NULL) {
        _payload.SetContent(AP->GetChannel());
        _payload.SetContent(AP->GetBSSID(), 0, 6);
    }

    _frame.SetCommandID(CMD_ID_WIFI);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_WIFI && _payload.GetSubCommandID() == WIFI_JOIN_REQ)
            return (WIFICode)_payload.GetData()[2];

    return WIFI_NORESPONSE;
}

WIFICode MuRata::WIFI_DisconnectNetwork()
{
    _payload.Rewind();
    _payload.SetSubCommandID(WIFI_DISCONNECT_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);

    _frame.SetCommandID(CMD_ID_WIFI);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_WIFI && _payload.GetSubCommandID() == WIFI_DISCONNECT_REQ)
            return (WIFICode)_payload.GetData()[2];

    return WIFI_NORESPONSE;
}
WIFIStatusResponse * MuRata::WIFI_GetStatus(const WIFIInterface WiFiInterface)
{
    _payload.Rewind();
    _payload.SetSubCommandID(WIFI_GET_STATUS_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(WiFiInterface);

    _frame.SetCommandID(CMD_ID_WIFI);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_WIFI && _payload.GetSubCommandID() == WIFI_GET_STATUS_REQ) {
            _WIFIStatusResponse.SetPayload(&_payload);
            return &_WIFIStatusResponse;
        }

    return NULL;
}

int8_t MuRata::WIFI_GetRSSI()
{
    _payload.Rewind();
    _payload.SetSubCommandID(WIFI_GET_STA_RSSI_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);

    _frame.SetCommandID(CMD_ID_WIFI);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_WIFI && _payload.GetSubCommandID() == WIFI_GET_STA_RSSI_REQ)
            return _payload.GetData()[2];

    return 127;
}

WIFICode MuRata::WIFI_StartWPSProcess(const WPSMode mode, const char * pin, int pinLength)
{
    _payload.Rewind();
    _payload.SetSubCommandID(WIFI_WPS_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(mode);

    if (mode == Pin) {
        if (Pin == NULL)
            return WIFI_NORESPONSE;

        _payload.SetContent(pin, 0 , pinLength);
        _payload.SetContent(0x00);
    }

    _frame.SetCommandID(CMD_ID_WIFI);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_WIFI && _payload.GetSubCommandID() == WIFI_WPS_REQ)
            return (WIFICode)_payload.GetData()[2];

    return WIFI_NORESPONSE;
}

WIFICode MuRata::WIFI_ScanNetworks(const ScanType scan, const BSSType bss)
{
    /*
     * This command instructs the module to scan available networks. Parameters are as follows:
     * UINT8 Request Sequence
     * UINT8 Scan Type
     * UINT8 BSS Type
     * UINT8 BSSID [6]
     * UINT8 Channel list []
     * UINT8 SSID[]
     * BSSID, Channel List, and SSID are optional fields. All 0’s for BSSID, Channel list or SSID indicates it is not present.
     * - Scan Type: 0 = Active scan, 1= Passive scan
     * - BSS Type: 0 = Infrastructure, 1 = ad hoc, 2 = any
     * - BSSID: 6 bytes MAC address of the AP or STA. 6 bytes of 0’s indicates it is not present.
     * - Channel list: 0 terminated array, up to 10 array elements.
     * - SSID: 0 terminated string for the AP or STA SSID, up to 33 bytes including NUL-termination.
     */

    _payload.Rewind();
    _payload.SetSubCommandID(WIFI_SCAN_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(scan);
    _payload.SetContent(bss);

    for (int i = 0; i < 8; i++)
        _payload.SetContent(0x00);

    _frame.SetCommandID(CMD_ID_WIFI);
    _frame.SetPayload(&_payload);

    Send();

    if (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_WIFI && _payload.GetSubCommandID() == WIFI_SCAN_REQ)
            return (WIFICode)_payload.GetData()[2];

    return WIFI_NORESPONSE;
}

InitializationResponse * MuRata::SNIC_Initialization(int receiveBufferSize)
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_INIT_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(receiveBufferSize >> 8);
    _payload.SetContent(receiveBufferSize);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_INIT_REQ) {
            _InitializationResponse.SetPayload(&_payload);
            return &_InitializationResponse;
        }

    return NULL;
}

SNICCode MuRata::SNIC_Cleanup()
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_CLEANUP_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_CLEANUP_REQ)
            return (SNICCode)_payload.GetData()[2];

    return SNIC_NORESPONSE;

}


SendFromSocketResponse * MuRata::SNIC_SendFromSocket(const char SocketID, const SocketSentOption option, const char * payload, int offset, int length)
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_SEND_FROM_SOCKET_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(SocketID);
    _payload.SetContent(option);
    _payload.SetContent(length >> 8);
    _payload.SetContent(length);
    _payload.SetContent(payload, offset, length);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_SEND_FROM_SOCKET_REQ) {
            _SendFromSocketResponse.SetPayload(&_payload);
            return &_SendFromSocketResponse;
        }

    return NULL;
}

SNICCode MuRata::SNIC_SloseSocket(const char SocketID)
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_CLOSE_SOCKET_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(SocketID);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_CLOSE_SOCKET_REQ)
            return (SNICCode)_payload.GetData()[2];

    return SNIC_NORESPONSE;
}

DHCPInfoResponse * MuRata::SNIC_GetDHCPInfo(const WIFIInterface wifiInterface)
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_GET_DHCP_INFO_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(wifiInterface);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_GET_DHCP_INFO_REQ) {
            _DHCPInfoResponse.SetPayload(&_payload);
            return &_DHCPInfoResponse;
        }

    return NULL;
}

IPAddress * MuRata::SNIC_ResolveHostName(const char * host)
{
    int hostLength = strlen(host);

    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_RESOLVE_NAME_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(STA);
    _payload.SetContent(hostLength);
    _payload.SetContent(host, 0, hostLength);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_RESOLVE_NAME_REQ)
            if ((SNICCode)_payload.GetData()[2] == SNIC_SUCCESS) {
                ip.SetValue(_payload.GetData(), 3);
                return &ip;
            }

    return NULL;
}

SNICCode MuRata::SNIC_ConfigureDHCPorStaticIP(DHCPConfig * config)
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_IP_CONFIG_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(config->GetInterface());
    _payload.SetContent(config->GetDHCPMode());

    if (config->GetDHCPMode() != dynamic_IP) {
        _payload.SetContent(config->GetLocalIP()->GetValue(), 0, 4);
        _payload.SetContent(config->GetNetmask()->GetValue(), 0, 4);
        _payload.SetContent(config->GetGatewayIP()->GetValue(), 0, 4);
    }

    if (config->GetDHCPMode() == soft_AP) {
        _payload.SetContent(config->GetIPRangeFirst()->GetValue(), 0, 4);
        _payload.SetContent(config->GetIPRangeLast()->GetValue(), 0, 4);
    }

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_IP_CONFIG_REQ)
            return (SNICCode)_payload.GetData()[2];

    return SNIC_NORESPONSE;
}

SocketStartReceiveResponse * MuRata::SNIC_ConnectTCPServer(const char SocketID, IPAddress * remoteIP, const int remotePort, const char timeout, const int receiveBufferSize)
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_TCP_CONNECT_TO_SERVER_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(SocketID);
    _payload.SetContent(remoteIP->GetValue(), 0, 4);
    _payload.SetContent((remotePort >> 8));
    _payload.SetContent(remotePort);
    _payload.SetContent((receiveBufferSize >> 8));
    _payload.SetContent(receiveBufferSize);
    _payload.SetContent(timeout);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_TCP_CONNECT_TO_SERVER_REQ) {
            _SocketStartReceiveResponse.SetPayload(&_payload);
            return &_SocketStartReceiveResponse;
        }

    return NULL;
}

CreateSocketResponse * MuRata::SNIC_CreateTCPSocket(const bool bind, IPAddress * localIP, const int localPort)
{
    return SNIC_CreateSocket(SNIC_TCP_CREATE_SOCKET_REQ, bind, localIP, localPort);
}

CreateSocketResponse * MuRata::SNIC_CreateUDPSocket(const bool bind, IPAddress * localIP, const int localPort)
{
    return SNIC_CreateSocket(SNIC_UDP_CREATE_SOCKET_REQ, bind, localIP, localPort);
}

CreateSocketResponse * MuRata::SNIC_CreateSocket(const SubCommandID subID, const bool bind, IPAddress * localIP, const int localPort)
{
    _payload.Rewind();
    _payload.SetSubCommandID(subID);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(bind ? 0x01 : 0x00);

    if (bind) {
        if (localIP != NULL)
            _payload.SetContent(localIP->GetValue(), 0, 4);
        else {
            _payload.SetContent(0x00);
            _payload.SetContent(0x00);
            _payload.SetContent(0x00);
            _payload.SetContent(0x00);
        }

        _payload.SetContent(localPort >> 8);
        _payload.SetContent(localPort);
    }

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == subID) {
            _CreateSocketResponse.SetPayload(&_payload);
            return &_CreateSocketResponse;
        }

    return NULL;
}

SocketStartReceiveResponse * MuRata::SNIC_StartUDPReceive(const char SocketID, const int receiveBufferSize)
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_UDP_START_RECV_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(SocketID);
    _payload.SetContent(receiveBufferSize >> 8);
    _payload.SetContent(receiveBufferSize);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_UDP_START_RECV_REQ) {
            _SocketStartReceiveResponse.SetPayload(&_payload);
            return &_SocketStartReceiveResponse;
        }

    return NULL;
}

SendFromSocketResponse * MuRata::SNIC_SendUDPPacket(IPAddress * remoteIP, const int remotePort, const char * payload, int offset, int length)
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_UDP_SIMPLE_SEND_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(remoteIP->GetValue(), 0 ,4);
    _payload.SetContent(remotePort >> 8);
    _payload.SetContent(remotePort);
    _payload.SetContent(length >> 8);
    _payload.SetContent(length);
    _payload.SetContent(payload, offset, length);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_UDP_SIMPLE_SEND_REQ) {
            _SendFromSocketResponse.SetPayload(&_payload);
            return &_SendFromSocketResponse;
        }

    return NULL;
}

SendFromSocketResponse * MuRata::SNIC_SendUDPFromSocket(IPAddress * remoteIP, const int remotePort, const char SocketID, const bool connectServer, const char * payload, int offset, int length)
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_UDP_SEND_FROM_SOCKET_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(remoteIP->GetValue(),0, 4);
    _payload.SetContent(remotePort >> 8);
    _payload.SetContent(remotePort);
    _payload.SetContent(SocketID);
    _payload.SetContent(connectServer ? 0x01 : 0x00);
    _payload.SetContent(length >> 8);
    _payload.SetContent(length);
    _payload.SetContent(payload, offset, length);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_UDP_SEND_FROM_SOCKET_REQ) {
            _SendFromSocketResponse.SetPayload(&_payload);
            return &_SendFromSocketResponse;
        }

    return NULL;
}

HTTPResponse * MuRata::SNIC_SendHTTPRequest(HTTPContent * content, const bool isHTTPS, const bool chunked)
{
    SubCommandID _id = (isHTTPS == true) ? SNIC_HTTPS_REQ: SNIC_HTTP_REQ;

    _payload.Rewind();
    _payload.SetSubCommandID(_id);
    _payload.SetFrameID(_payload.GetFrameID() + 1);
    _payload.SetContent(content->GetRemotePort() >> 8);
    _payload.SetContent(content->GetRemotePort());
    _payload.SetContent(content->GetMethod());
    _payload.SetContent(content->GetTimeout());

    _payload.SetContent(content->GetRemoteHost(), 0, strlen(content->GetRemoteHost()));
    _payload.SetContent(0x00);

    _payload.SetContent(content->GetURI(), 0, strlen(content->GetURI()));
    _payload.SetContent(0x00);

    _payload.SetContent(content->GetContentType(), 0, strlen(content->GetContentType()));
    _payload.SetContent(0x00);

    string _others;
    content->GetOtherHeaders(&_others);
    _payload.SetContent(_others.c_str(), 0, _others.length());
    _payload.SetContent(0x00);

    while (content->GetMethod() == POST) {
        int length = content->GetContentLength();

        char msb = length >> 8;
        if (chunked)
            msb |= 0x80;
        else msb &= 0x7F;

        _payload.SetContent(msb);
        _payload.SetContent(length);

        if (length > 0)
            _payload.SetContent(content->GetBody(), 0, length);
    }

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == _id) {
            _HTTPResponse.SetPayload(&_payload);
            return &_HTTPResponse;
        }

    return NULL;
}

HTTPResponse * MuRata::SNIC_SendHTTPMoreRequest(HTTPContent * content, const bool chunked)
{
    _payload.Rewind();
    _payload.SetSubCommandID(SNIC_HTTP_MORE_REQ);
    _payload.SetFrameID(_payload.GetFrameID() + 1);

    int length = content->GetContentLength();
    char msb = length;
    if (chunked)
        msb |= 0x80;
    else msb &= 0x7F;

    _payload.SetContent(msb);
    _payload.SetContent(length);

    if (length > 0)
        _payload.SetContent(content->GetBody(), 0, length);

    _frame.SetCommandID(CMD_ID_SNIC);
    _frame.SetPayload(&_payload);

    Send();

    if (chunked)
        return NULL;

    while (FrameReceive())
        if (_frame.GetCommandID() == CMD_ID_SNIC && _payload.GetSubCommandID() == SNIC_HTTP_MORE_REQ) {
            _HTTPResponse.SetPayload(&_payload);
            return &_HTTPResponse;
        }

    return NULL;
}

CreateSocketResponse * MuRata::SNIC_CreateAdvancedTLSTCP(const bool bind, IPAddress * localIP, const int localPort)
{
    return SNIC_CreateSocket(SNIC_TCP_CREATE_ADV_TLS_SOCKET_REQ, bind, localIP, localPort);
}

CreateSocketResponse * MuRata::SNIC_CreateSimpleTLSTCP(const bool bind, IPAddress * localIP, const int localPort)
{
    return SNIC_CreateSocket(SNIC_TCP_CREAET_SIMPLE_TLS_SOCKET_REQ, bind, localIP, localPort);
}