#ifndef UK_AC_HERTS_SMARTLAB_XBEE_CoreAPI
#define UK_AC_HERTS_SMARTLAB_XBEE_CoreAPI

#include "APIFrame.h"
#include "ISerial.h"

#include "ATCommandRequest.h"
#include "CreateSourceRouteRequest.h"
#include "IOCDetectionConfigRequest.h"
#include "PinConfigurationRequest.h"
#include "RemoteATCommandRequest.h"
#include "RemoteIODetectionConfigRequest.h"
#include "RemotePinConfigurationRequest.h"
#include "XBeeTx16Request.h"
#include "XBeeTx64Request.h"
#include "ZigBeeExplicitTxRequest.h"
#include "ZigBeeTxRequest.h"

#include "XBeeRx64Indicator.h"
#include "XBeeRx16Indicator.h"
#include "XBeeRx64IOSampleIndicator.h"
#include "XBeeRx16IOSampleIndicator.h"
#include "XBeeTxStatusIndicator.h"
#include "ATCommandIndicator.h"
#include "ModemStatusIndicator.h"
#include "ZigBeeTxStatusIndicator.h"
#include "ZigBeeRxIndicator.h"
#include "ZigBeeExplicitRxIndicator.h"
#include "ZigBeeIOSampleIndicator.h"
#include "SensorReadIndicator.h"
#include "NodeIdentificationIndicator.h"
#include "RemoteCommandIndicator.h"
#include "RouteRecordIndicator.h"
#include "ManyToOneRouteIndicator.h"

/**
* The core API class which responseable for processing frame data, but not the serial operation.
*/
class CoreAPI
{
private:
    static const unsigned char KEY = 0x7E;
    static const unsigned char ESCAPED = 0x7D;
    static const unsigned char XON = 0x11;
    static const unsigned char XOFF = 0x13;
    static const unsigned int INITIAL_FRAME_LENGTH = 10;

    ISerial * serial;
    bool isEscapeMode;
    bool isRunning;
    bool isChecksum;
    Timer timer;

    unsigned char waitFrameID;

    APIFrame * msg;
    XBeeRx64Indicator xBeeRx64Indicator;
    XBeeRx16Indicator xBeeRx16Indicator;
    XBeeRx64IOSampleIndicator xBeeRx64IOSampleIndicator;
    XBeeRx16IOSampleIndicator xBeeRx16IOSampleIndicator;
    XBeeTxStatusIndicator xBeeTxStatusIndicator;
    ATCommandIndicator aTCommandIndicator;
    ModemStatusIndicator modemStatusIndicator;
    ZigBeeTxStatusIndicator zigBeeTxStatusIndicator;
    ZigBeeRxIndicator zigBeeRxIndicator;
    ZigBeeExplicitRxIndicator zigBeeExplicitRxIndicator;
    ZigBeeIOSampleIndicator zigBeeIOSampleIndicator;
    SensorReadIndicator sensorReadIndicator;
    NodeIdentificationIndicator nodeIdentificationIndicator;
    RemoteCommandIndicator remoteCommandIndicator;
    RouteRecordIndicator routeRecordIndicator;
    ManyToOneRouteIndicator manyToOneRouteIndicator;

protected:

    /** Read one byte payload, which allready handle the escape char, if less than 0 means error occured
    * @returns if less than 0 means error occured.
    */
    int readByte();

    /** Write one byte to the payload, which allready handle the escape char.
    * @param data one byte [0x00-0xFF]
    */
    void writeByte(unsigned  char data);

    /// Processing API frame.
    void packetProcess();

    /// Get the next avaliable API frame length.
    int getLength();

    /// Read the next avaliable API frame data.
    void readPayLoad(int length);

public:
    CoreAPI(ISerial * serial, bool escape);

    ~CoreAPI();

    /** Set whether to verify checksum during receiving, default is not verify.
    *
    * @param isCheck true only to process API frame when checksum matches.
    *                false ignore the checksum.
    */
    void setVerifyChecksum(bool isCheck);

    /// Start send and process response, must call this method before starting processing data.
    void start();

    /// Stop the serial port.
    void stop();

    /** A general function to send frame out
    *
    * @param request any API frame
    */
    void send(APIFrame * request);

    /** Read the next avaliable API frame, and the type of fram can be retrieved from getFrameType().
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    APIFrame * getResponse();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    XBeeRx64Indicator * getXBeeRx64();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    XBeeRx16Indicator * getXBeeRx16();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    XBeeRx64IOSampleIndicator * getXBeeRx64IOSample();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    XBeeRx16IOSampleIndicator * getXBeeRx16IOSample();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    XBeeTxStatusIndicator * getXBeeTxStatus();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    ATCommandIndicator * getATCommand();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    ModemStatusIndicator * getModemStatus();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    ZigBeeTxStatusIndicator * getZigBeeTxStatus();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    ZigBeeRxIndicator * getZigBeeRx();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    ZigBeeExplicitRxIndicator * getZigBeeExplicitRx();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    ZigBeeIOSampleIndicator * getZigBeeIOSample();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    SensorReadIndicator * getSensorRead();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    NodeIdentificationIndicator * getNodeIdentification();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    RemoteCommandIndicator * getRemoteCommand();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    RouteRecordIndicator * getRouteRecord();

    /** Read the next avaliable API frame.
    *
    * @returns a API frame, NULL means data not avaliable.
    */
    ManyToOneRouteIndicator * getManyToOneRoute();
    
    XBeeTxStatusIndicator * sendXBeeTx16(Address * remoteAddress, OptionsBase * option, const unsigned char * payload, int offset, int length);

    XBeeTxStatusIndicator * sendXBeeTx64(Address * remoteAddress, OptionsBase * option, const unsigned char * payload, int offset, int length);

    ATCommandIndicator * sendATCommand(const char * command, bool applyChange, const unsigned  char * parameter = NULL, int offset = 0, int length = 0);

    RemoteCommandIndicator * sendRemoteATCommand(Address * remoteAddress, const char * command, OptionsBase * transmitOptions, const unsigned  char * parameter = NULL, int parameterOffset = 0, int parameterLength = 0);

    ZigBeeTxStatusIndicator * sendZigBeeTx(Address * remoteAddress, OptionsBase * option, const unsigned  char * payload, int offset, int length);

    ZigBeeTxStatusIndicator * sendZigBeeExplicitTx(ExplicitAddress * remoteAddress, OptionsBase * option, const unsigned  char * payload, int offset, int length);

    /**
    * @param function
    * DISABLED = 0x00,
    * RESERVED_FOR_PIN_SPECIFIC_ALTERNATE_FUNCTIONALITIES = 0x01,
    * ANALOG_INPUT_SINGLE_ENDED = 0x02,
    * DIGITAL_INPUT_MONITORED = 0x03,
    * DIGITAL_OUTPUT_DEFAULT_LOW = 0x04,
    * DIGITAL_OUTPUT_DEFAULT_HIGH = 0x05,
    * ALTERNATE_FUNCTIONALITIES_WHERE_APPLICABLE = 0x06//0x06~0x09
    */
    ATCommandIndicator * setPinFunction(Pin * pin, unsigned  char function);

    ATCommandIndicator * setIODetection(Pin ** pins, int size);

    /**
    * @param function
    * DISABLED = 0x00,
    * RESERVED_FOR_PIN_SPECIFIC_ALTERNATE_FUNCTIONALITIES = 0x01,
    * ANALOG_INPUT_SINGLE_ENDED = 0x02,
    * DIGITAL_INPUT_MONITORED = 0x03,
    * DIGITAL_OUTPUT_DEFAULT_LOW = 0x04,
    * DIGITAL_OUTPUT_DEFAULT_HIGH = 0x05,
    * ALTERNATE_FUNCTIONALITIES_WHERE_APPLICABLE = 0x06//0x06~0x09
    */
    RemoteCommandIndicator * setRemotePinFunction(Address * remoteAddress, Pin * pin, unsigned char function);

    RemoteCommandIndicator * setRemoteIODetection(Address * remoteAddress, Pin ** pins, int size);

    /// <summary>
    /// The command will immediately return an "OK" response. The data will follow in the normal API format for DIO data event.
    /// </summary>
    /// <returns>true if the command is "OK", false if no IO is enabled.</returns>
    bool forceXBeeLocalIOSample();

    /// <summary>
    /// Return 1 IO sample from the local module.
    /// </summary>
    /// <returns></returns>
    IOSamples * forceZigBeeLocalIOSample();

    /// <summary>
    /// Return 1 IO sample only, Samples before TX (IT) does not affect.
    /// </summary>
    /// <param name="remote"Remote address of the device></param>
    /// <returns></returns>
    IOSamples * forceXBeeRemoteIOSample(Address * remote);

    /// <summary>
    /// Return 1 IO sample only.
    /// </summary>
    /// <param name="remote">Remote address of the device</param>
    /// <returns></returns>
    IOSamples * forceZigBeeRemoteIOSample(Address * remote);
};

#endif
