#ifndef UK_AC_HERTS_SMARTLAB_XBEE_Address
#define UK_AC_HERTS_SMARTLAB_XBEE_Address

#include "mbed.h"

/// XBee address class for both S1 and S2 hardware
class Address
{
protected:
    /// total 10 bytes (IEEE 64 bits + 16 bits networ address)
    unsigned char value[10];
public:
    /// Get the ZigBee broadcast address.
    static Address * BROADCAST_ZIGBEE;
    
    /// Get the XBee broadcast address.
    static Address * BROADCAST_XBEE;

    /**
    * Create empty address : 0x00000000 0x00000000 0x0000
    * this is the default ZigBee Coordinatior
    */
    Address();

    /**
    * Create address from existing byte aray value : 8 bytes of ieee + 2 bytes network
    * @param address64 8 bytes of IEEE address
    * @param NET16 value 2 bytes network address
    */
    Address(const unsigned char * address64, const unsigned char * NET16);

    /** Create address from existing byte array value ( 8 bytes of IEEE address + 2 bytes 16 bit network address)
    * @param addr 10 bytes leong : (0-7 bytes for IEEE address, 8-9 bytes for 16 bit network address)
    */
    Address(const unsigned char * addr);

    /** Create address from number value.
    * @param serialNumberHigh first four byts of IEEE address and this can be retrieved by 'SH' command (eg. 0x0013A200)
    * @param serialNumberLow last four byts of IEEE address and this can be retrieved by 'SL' command.
    * @param networkAddress a dynamic allocated value for XBee module, 0x0000 is reserved for the coordinator, and this can be retrieved by 'MY' command.
    */
    Address(long serialNumberHigh, long serialNumberLow, int networkAddress);
    
    /// Get the first 4 bytes of IEEE address
    unsigned long getSerialNumberHigh();

    /// Get the last 4 bytes of IEEE address
    unsigned long getSerialNumberLow();
    
    /// Get the 16 bit network address
    unsigned int getNetworkAddress();

    /// Set the first 4 bytes of IEEE address
    void setSerialNumberHigh(long SerialNumberHigh);

    /// Set the last 4 bytes of IEEE address
    void setSerialNumberLow(long SerialNumberLow);

    /// Set the 16 bit network address
    void setNetworkAddress(int NetworkAddress);

    /** Convert the device address to 10 bytes array.
    * @returns IEEE 64 bit address follow by 16 bit network address
    */
    const unsigned  char * getAddressValue();
    
    /// Compare two XBee addresses and check if both point to the same device.
    friend bool operator ==(const Address &a,const Address &b);

    /// Compare two XBee addresses and check if both point to the same device.
    friend bool operator !=(const Address &a,const Address &b);
};

#endif