/*
Copyright (c) 2013, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef XBEE_DATA_TYPES_H
#define XBEE_DATA_TYPES_H

#include "mbed.h"

/**
 * class for XBee 64-bit address data type
 */
class XBeeAddress64 {
public:
    XBeeAddress64();
    XBeeAddress64(const char *address);
    XBeeAddress64(uint64_t address);
    XBeeAddress64(uint32_t high, uint32_t low);

    operator uint64_t();
    bool operator ==(XBeeAddress64& address);
    bool operator ==(const char* address);
    bool operator ==(uint64_t address);
    operator char *();
    
    char operator [](int index) const { 
        return address64[index]; 
    };

    char& operator [](int index) { 
        return address64[index]; 
    };
    
    friend class XBee;
    
private:
    char address64[8];
    char buf[18];
    
    char *raw_address() { return (char *) address64; }
};

const XBeeAddress64 COORDINATOR(0, 0);
const XBeeAddress64 BROADCAST(0, 0xFFFF);

/**
 * class for XBee 16-bit address data type
 */
class XBeeAddress16 {
public:
    XBeeAddress16();
    XBeeAddress16(const char *address);
    XBeeAddress16(uint16_t address);
    XBeeAddress16(char high, char low);

    operator uint16_t();
    bool operator ==(XBeeAddress16& address);
    bool operator ==(const char* address);
    bool operator ==(uint16_t address);
    operator char *();
    
    char operator [](int index) const { 
        return address16[index]; 
    };
    char& operator [](int index) { 
        return address16[index]; 
    };

    friend class XBee;
    
private:
    char address16[2];
    char buf[5];
    
    char *raw_address() { return (char *) address16; }
};

/**
 * class for XBee data type
 */
class XBeeData {
public:
    XBeeData(int capacity);
    ~XBeeData();
    
    int length() {
        return size;
    }

    operator char *() const {
         return data;
    }
    
    char operator [](int index) const { 
         return data[index]; 
    };

    char& operator [](int index) { 
         return data[index]; 
    };

    friend class XBee;
    friend class XBeeCommandData;
    friend class XBeeReceivedData;
    friend class XBeeRawData;
    friend class XBeeNodeIdentificaion;
  
private:
    char *data;
    int capacity;
    int size;
    
    char *raw_address() { return (char *) data; }
};

/**
 * class for XBee command data type
 */
class XBeeCommandData : public XBeeData {
public:
    XBeeCommandData(int capacity = 20) : XBeeData(capacity) {}
};

/**
 * class for XBee received data type
 */
class XBeeReceivedData : public XBeeData {
public:
    XBeeReceivedData(int capacity = 128) : XBeeData(capacity) {}
};

/**
 * class for XBee raw data type
 */
class XBeeRawData : public XBeeData {
public:
    XBeeRawData(int capacity = 128) : XBeeData(capacity) {}    
};

/**
 * class for XBee AT command data type
 */
class XBeeATCommand : public XBeeData {
public:
    XBeeATCommand() : XBeeData(3) {}
};

/**
 * class for XBee Node Identifier
 */
class XBeeNodeIdentifier: public XBeeData {
public:
    XBeeNodeIdentifier() : XBeeData(20) {}
};

/**
 * class for bit array data type
 */
class BitArray {
public:
    BitArray(int mask, int values);
    int operator [](int i) const;
    
private:
    int mask;
    int values;

    int valueAt(int i) const;
};

/**
 * class for int array data type
 */
class IntArray {
public:
    IntArray(int mask, const char *value);
    int operator [](int i) const;

private:
    int mask;
    const char *values;

    int valueAt(int i) const;
};

/**
 * class for IOSample data type
 */
class IOSample {
public:
    IOSample(const char *data);

    operator char *();

    BitArray dio;
    IntArray ad;
private:
    char buf[128];
};

/**
 * class for uint8 data type
 */
class XBeeUint8 {
public:
    XBeeUint8(char data = 0, char base = 10) {
        this->data = data;
    }
    
    operator int() {
        return data;
    }
    
    operator char *() {
        sprintf(buf, base == 10 ? "%d" : "%02X", data);
        return buf;
    } 
    
    friend class XBee;
    friend class XBeeFrameID;
    friend class XBeeRetryCount;
    friend class XBeeStatus;
    friend class XBeeDeliveryStatus;
    friend class XBeeDiscoveryStatus;
    friend class XBeeReceiveOptions;
    friend class XBeeDeviceType;
    friend class XBeeSourceEvent;
 
private:
    char data;
    char base;
    char buf[4];
    
    char *raw_address() {
         return &data;
    }
};

/**
 * class for XBee frame ID data type
 */
class XBeeFrameID : public XBeeUint8 {
public:
    XBeeFrameID(char id = 0) : XBeeUint8(id) {}
};

/**
 * class for XBee retry count data type
 */
class XBeeRetryCount : public XBeeUint8 {
public:
    XBeeRetryCount(char count = 0) : XBeeUint8(count) {}
};

/**
 * class for XBee status data type
 */
class XBeeStatus : public XBeeUint8 {
public:
    XBeeStatus(char data = 0) : XBeeUint8(data, 16) {}
};

/**
 * class for XBee delivery status data type
 */
class XBeeDeliveryStatus : public XBeeUint8 {
public:
    XBeeDeliveryStatus(char data = 0) : XBeeUint8(data, 16) {}
};

/**
 * class for XBee discovery status data type
 */
class XBeeDiscoveryStatus : public XBeeUint8 {
public:
    XBeeDiscoveryStatus(char data = 0) : XBeeUint8(data, 16) {}
};

/**
 * class for XBee receive status data type
 */
class XBeeReceiveOptions : public XBeeUint8 {
public:
    XBeeReceiveOptions(char data = 0) : XBeeUint8(data, 16) {}
};

/**
 * class for XBee device type
 */
class XBeeDeviceType : public XBeeUint8 {
public:
    XBeeDeviceType(char data = 0) : XBeeUint8(data, 16) {}
};

/**
 * class for XBee Source Event
 */
class XBeeSourceEvent : public XBeeUint8 {
public:
    XBeeSourceEvent(char data = 0) : XBeeUint8(data, 16) {}
};
#endif
