/*
Copyright (c) 2013, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef XBEE_H
#define XBEE_H

#include "mbed.h"
#include "XBeeDataTypes.h"
#include "rtos.h"

const char ESCAPE = 0x7D;
const char PREAMBLE = 0x7E;
const int BUFSIZE = 512;

/**
 * class for XBee module API mode interface
 */
class XBee : public Serial {
public:

    /**
     * Frame type declaration for XBee API frames
     */
    enum FrameType {
        /**
        * Empty data
        */
        None = 0,
        
        /**
         *  AT Command Response API frame
         */
        ATCommandResponse,
        
        /**
         * Modem Status API frame
         */
        ModemStatus,
        
        /**
         * ZigBee Transmit Status API frame
         */
        ZigBeeTransmitStatus,
        
        /**
         * ZigBee Receive Packet API frame
         */
        ZigBeeReceivePacket,

        /**
         *  ZigBee Explicit Rx Indicator API frame
         */
        ZigBeeExplicitRxIndicator,

        /**
         * ZigBee I/O Data Sample Rx Indicator API frame
         */
        ZigBeeIODataSampleRxIndicator,

        /**
         * XBee Sensor Read Indicator API frame
         */
        XBeeSensorReadIndicator,

        /**
         * Node Identification Indicator API frame
         */
        NodeIdentificationIndicator,

        /**
         * Remote Command Response API frame
         */
        RemoteCommandResponse,

        /**
         * Unknown API frame
         */
        Other
    };
    
    /**
     * Value type declarations for retrieving frame data contents
     */
    enum ValueType {
        /**
         * Frame ID
         */
        FrameID,
        
        /**
         *AT command name
         */
        ATCommand,
        
        /**
         * Status
         */       
        Status,
        
        /**
         * Command data
         */
        CommandData,
        
        /**
         * 16 bit address
         */
        Address16,
        
        /**
         * 64 bit address
         */
        Address64,

        /**
         * 16 bit remote address
         */
        RemoteAddress16,
        
        /**
         * 64 bit remote address
         */
        RemoteAddress64,

        /**
         * 16 bit parent address
         */
        ParentAddress16,
        
        /**
         * Retry count
         */
        RetryCount,
        
        /**
         * Delivery status
         */
        DeliveryStatus,
        
        /**
         * Discovery status
         */
        DiscoveryStatus,
        
        /**
         * Receive option
         */
        ReceiveOptions,
        
        /**
         * Received data
         */
        ReceivedData,
        
        /**
         * Node Identification String
         */
        NIString,
        
        /**
         * Device Type
         */
        DeviceType,        
        
        /**
         * Source Event
         */
        SourceEvent,        
        
        /**
         * Raw data
         */
        RawData
    };

    /**
     * creates an XBee interface object.
     *
     * @param ser Serial object through which XBee module is connected to mbed
     * @param api API mode either 1 or 2 (use escape; default)
     */
    XBee(Serial& ser, int api = 2);

    /**
     * creates an XBee interface object.
     *
     * @param tx TX pin connected to XBee
     * @param rx RX pin connected to XBee
     * @param api API mode either 1 or 2 (use escape; default)
     */
    XBee(PinName tx, PinName rx, int api = 2);

    /**
     * creates an XBee interface object.
     *
     * @param ser Serial object through which XBee module is connected to mbed
     * @param mon alternate Serial object for monitoring (use serial ports other than USBTX/USBRX)
     * @param api API mode either 1 or 2 (use escape; default)
     */
    XBee(Serial& ser, Serial& mon, int api = 2);

    /**
     * creates an XBee interface object.
     *
     * @param tx TX pin connected to XBee
     * @param rx RX pin connected to XBee
     * @param mon alternate Serial object for monitoring (use serial ports other than USBTX/USBRX)
     * @param api API mode either 1 or 2 (use escape; default)
     */
    XBee(PinName tx, PinName rx, Serial& mon, int api = 2);
    
    /**
     * initializes XBee module.
     *
     * issues VR command to test XBee modem connection
     *
     * @returns true if initialization succeeded, false otherwise
     */
    bool init(float timeout = 15.0);

    /**
     * sets destination addresses.
     *
     * @param address64 XBeeAddress64 type address
     * @param address16 XBeeAddress16 type address (optional)
     */
    void setDestination(XBeeAddress64 address64, XBeeAddress16 address16 = 0xFFFE);
    
    /**
     * sets destination addresses.
     *
     * @param address64 64-bit destination address in uint64_t
     * @param address16 16-bit destination address in uint16_t
     */
    void setDestination(uint64_t address64, uint16_t address16 = 0xFFFE);

    /**
      * sets destination addresses.
      *
      * @param address64 64-bit destination address in bytes (big endian)
      * @param address16 16-bit destination address in bytes
      */
    void setDestination(char address64[], char address16[]);
    
    /**
     * sends an AT command.
     *
     * @param command AT command char string
     * @param param parameter to the AT command
     */
    void sendCommand(const char *command, int8_t param, bool queue = false);
    void sendCommand(const char *command, int16_t param, bool queue = false);
    void sendCommand(const char *command, int32_t param, bool queue = false);
    void sendCommand(const char *command, int64_t param, bool queue = false);
    void sendCommand(const char *command, uint8_t param, bool queue = false);
    void sendCommand(const char *command, uint16_t param, bool queue = false);
    void sendCommand(const char *command, uint32_t param, bool queue = false);
    void sendCommand(const char *command, uint64_t param, bool queue = false);
    void sendCommand(const char *command, const char *param, bool queue = false);
    
    /**
     * sends an AT command.
     *
     * @param command AT command char string
     * @param param parameter to the AT command (pointer to byte array)
     * @param param_length parameter length in bytes
     * @param queue true if command paramter is to be queued
     */
    void sendCommand(const char *command, const uint8_t *param = 0, int param_length = 0, bool queue = false);

    /**
     * sends a remote AT command.
     *
     * sends an AT command to the XBee(s) at the destination address
     *
     * @param command AT command char string
     * @param param parameter to the AT command (pointer to byte array)
     * @param param_length parameter length in bytes
     * @param options remote command options
     */
    void sendRemoteCommand(const char *command, int8_t param);
    void sendRemoteCommand(const char *command, int16_t param);
    void sendRemoteCommand(const char *command, int32_t param);
    void sendRemoteCommand(const char *command, int64_t param);
    void sendRemoteCommand(const char *command, uint8_t param);
    void sendRemoteCommand(const char *command, uint16_t param);
    void sendRemoteCommand(const char *command, uint32_t param);
    void sendRemoteCommand(const char *command, uint64_t param);
    void sendRemoteCommand(const char *command, const char *param);
    
    /**
     * sends a remote AT command.
     *
     * sends an AT command to the XBee(s) at the destination address
     *
     * @param command AT command char string
     * @param param parameter to the AT command (pointer to byte array)
     * @param param_length parameter length in bytes
     * @param options remote command options
     */
    void sendRemoteCommand(const char *command, const uint8_t *param = 0, int param_length = 0, char options = 0x02);

    /**
     * executes an AT command and gets the result.
     *
     * @param command AT command char string
     * @param param parameter to the AT command
     *
     * @returns pointer to the command result, if the result is a number (char, short, long, int64_t),
     *          the address to the number will be returned; otherwise the address to the byte array
     *          containing the command response will be returned.
     */
    void *executeCommand(const char *command, int8_t param);
    void *executeCommand(const char *command, int16_t param);
    void *executeCommand(const char *command, int32_t param);
    void *executeCommand(const char *command, int64_t param);
    void *executeCommand(const char *command, uint8_t param);
    void *executeCommand(const char *command, uint16_t param);
    void *executeCommand(const char *command, uint32_t param);
    void *executeCommand(const char *command, uint64_t param);
    void *executeCommand(const char *command, const char *param);

    /**
     * executes an AT command and gets the result.
     *
     * @param command AT command char string
     * @param param parameter to the AT command (pointer to byte array)
     * @param param_length parameter length in bytes
     *
     * @returns pointer to the command result, if the result is a number (char, short, long, long long),
     *          the address to the number will be returned; otherwise the address to the byte array
     *          containing the command response will be returned.
     */
    void *executeCommand(const char *command, const uint8_t *param = 0, int laram_length = 0);

    /**
     * sends data to the XBee(s) at the destination address.
     *
     * @param data address to the data (byte array)
     * @param length data length in bytes
     */
    bool send(const char *data, int length);

    /**
     * sets send confirmation timeout
     *
     * @param sendConfirmation maximum waiting time for receiving transmit status
     *
     */
    void setSendConfirmation(float sendConfirmation);

    /**
     * sends data to the destination using printf format.
     *
     * @param format printf format string, followed by corresponding arguments
     *
     * @returns the number of charancters sent, or negative if error occurred
     */
    int printf(const char *format, ...);

    /**
     * receives data frame from the XBee module.
     *
     * @param timeout seconds bofer time out
     *
     * @returns FrameType of the received frame data
     */
    FrameType receive(float timeout = 3.0);

    /**
     * scan received data
     *
     * @param data XBeeDataType data to be scanned
     *
     * @param true if scan succeeded, false otherwise
     */
    bool scan(XBeeFrameID& id);
    bool scan(XBeeRetryCount& count);
    bool scan(XBeeStatus& status);
    bool scan(XBeeDeliveryStatus& status);
    bool scan(XBeeDiscoveryStatus& status);
    bool scan(XBeeReceiveOptions& options);
    bool scan(XBeeDeviceType& device);
    bool scan(XBeeSourceEvent& event);
    bool scan(XBeeAddress64& address64);
    bool scan(XBeeAddress16& address16);
    bool scan(XBeeATCommand& command);
    bool scan(XBeeCommandData& data);
    bool scan(XBeeReceivedData& data);
    bool scan(XBeeNodeIdentifier& ni);
    bool scan(XBeeRawData& data);
    
    /**
     * scan received data according to the specified format.
     *
     * @param type ValueType of the data to be scanned
     * @param value pointer to the byte array to store the scanned value
     * @param maxlength max size of the value in bytes
     * @param length pointer to an int to receive the actual data length
     *
     * @param true if scan succeeded, false otherwise
     */
    bool scan(ValueType type, char *value, int maxlength = 1, int *length = 0);

    /**
     * gets the XBee firmware version.
     *
     * @returns XBee firmwre version in int (unsigned short value)
     */
    int getFirmwareVersion();

    /**
     * gets the current frame ID.
     *
     * @returns frame ID number being used in the next send request
     */
    char getFrameID();

    /**
    * displays received data in dump format.
    */
    void dump();

    /**
     * displays the internal data fields and receive buffer in dump format.
     */
    void dumpAll();

    /**
     * operator overloading for testing XBee modem connection status.
     *
     * @returns false if XBee modem connection has an error
     */
    operator bool();

private:
    Serial mon;
    Timer timer;
    Semaphore sem;
    int api;
    volatile int cur, in, out, received, free;
    char frame_id;
    float sendConfirmation;
    char destination64[8];
    char destination16[2];
    char buf[BUFSIZE];

    void send(char c);
    void send2(char c);
    void sendFrame(const char *frame, int length);
    int createTxRequest(char frame_id, const char *data, int data_length, char *buf, int buf_size);
    int createAtRequest(char frame_id, const char *command, const uint8_t *param, int param_length, bool queue, char *buf, int buf_size);
    int createRemoteAtRequest(char frame_id, const char *command, const uint8_t *param, int param_length, char options, char *buf, int buf_size);
    bool send(const char *data, int length, float timeout);
    int seekFor(FrameType type, char id, float timeout);
    FrameType getFrameType(char c);
    bool scan(int i, ValueType type, char *value, int maxlength = 1, int *length = 0);
    void flush();
    void rxISR();
    void dump(const char *data, int length);
    void dumpIOSample(const char *data, int length);
    void copy(char *toBuf, int fromIndex, int length);
};

#endif