/*
Copyright (c) 2013, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "XBee.h"

#define INDEX(n)    ((n) % BUFSIZE)
#define SIZE(b, i)  (b[i] << 8 | b[INDEX(i + 1)])
#define min(x, y)   ((x) <= (y) ? (x) : (y))

bool XBee::scan(XBeeFrameID& id) {
    return scan(XBee::FrameID, id.raw_address(), 1);
}

bool XBee::scan(XBeeRetryCount& count) {
    return scan(XBee::RetryCount, count.raw_address(), 1);
}

bool XBee::scan(XBeeStatus& status) {
    return scan(XBee::Status, status.raw_address(), 1);
}

bool XBee::scan(XBeeDeliveryStatus& status) {
    return scan(XBee::DeliveryStatus, status.raw_address(), 1);
}

bool XBee::scan(XBeeDiscoveryStatus& status) {
    return scan(XBee::DiscoveryStatus, status.raw_address(), 1);
}

bool XBee::scan(XBeeReceiveOptions& options) {
    return scan(XBee::ReceiveOptions, options.raw_address(), 1);
}

bool XBee::scan(XBeeDeviceType& device) {
    return scan(XBee::DeviceType, device.raw_address(), 1);
}

bool XBee::scan(XBeeSourceEvent& event) {
    return scan(XBee::SourceEvent, event.raw_address(), 1);
}

bool XBee::scan(XBeeAddress64& address64) {
    return scan(XBee::Address64, address64.raw_address(), 8);
}

bool XBee::scan(XBeeAddress16& address16) {
    return scan(XBee::Address16, address16.raw_address(), 2);
}

bool XBee::scan(XBeeATCommand& command) {
    return scan(XBee::ATCommand, command.raw_address(), 3);
}

bool XBee::scan(XBeeCommandData& data) {
    return scan(XBee::CommandData, data.raw_address(), data.capacity, &data.size);
}

bool XBee::scan(XBeeReceivedData& data) {
    return scan(XBee::ReceivedData, data.raw_address(), data.capacity, &data.size);
}

bool XBee::scan(XBeeNodeIdentifier& ni) {
    return scan(XBee::NIString, ni.raw_address(), ni.capacity, &ni.size);
}

bool XBee::scan(XBeeRawData& data) {
    return scan(XBee::RawData, data.raw_address(), data.capacity, &data.size);
}

bool XBee::scan(ValueType type, char *value, int maxlength, int *length) {
    if (received != out || (in == out && free != 0))
        return false;

    return scan(out, type, value, maxlength, length);
}

bool XBee::scan(int index, ValueType type, char *value, int maxlength, int *length) {
    int from = 0;
    int len = 1;
    FrameType frameType = getFrameType(buf[INDEX(index + 2)]);

    if (frameType == (FrameType) None)
        return false;

    switch (type) {
        case FrameID:
            from = 1;
            break;
        case ATCommand:
            switch (frameType) {
                case ATCommandResponse:
                    from = 2;
                    break;
                case RemoteCommandResponse:
                    from = 12;
                    break;
            }
            len = 2;
            break;
        case Status:
            switch (frameType) {
                case ATCommandResponse:
                    from = 4;
                    break;
                case ModemStatus:
                    from = 1;
                    break;
                case RemoteCommandResponse:
                    from = 14;
                    break;
            }
            break;
        case CommandData:
            switch (frameType) {
                case ATCommandResponse:
                    from = 5;
                    len = SIZE(buf, index) - 5;
                    break;
                case RemoteCommandResponse:
                    from = 15;
                    len = SIZE(buf, index) - 15;
                    break;
            }
            break;
        case Address16:
            switch (frameType) {
                case ZigBeeTransmitStatus:
                    from = 2;
                    break;
                case ZigBeeReceivePacket:
                case ZigBeeIODataSampleRxIndicator:
                case NodeIdentificationIndicator:
                    from = 9;
                    break;
                case RemoteCommandResponse:
                    from = 10;
                    break;
            }
            len = 2;
            break;
        case Address64:
            switch (frameType) {
                case ZigBeeReceivePacket:
                case ZigBeeIODataSampleRxIndicator:
                case NodeIdentificationIndicator:
                    from = 1;
                    break;
                case RemoteCommandResponse:
                    from = 2;
                    break;
            }
            len = 8;
            break;
        case RemoteAddress16:
            from = 12;
            len = 2;
            break;
        case RemoteAddress64:
            from = 14;
            len = 8;
            break;
        case ParentAddress16:
            from = SIZE(buf, index) - 8;
            len = 2;
            break;
        case RetryCount:
            from = 4;
            break;
        case DeliveryStatus:
            from = 5;
            break;
        case DiscoveryStatus:
            from = 6;
            break;
        case ReceiveOptions:
            from = 11;
            break;
        case ReceivedData:
            from = 12;
            len = SIZE(buf, index) - 12;
            break;
        case NIString:
            from = 22;
            len = SIZE(buf, index) - (22 + 8);
            break;
        case DeviceType:       
            from = SIZE(buf, index) - 6;
            break;
        case SourceEvent:
            from = SIZE(buf, index) - 5;
            break;              
        case RawData:
            from = 1;
            len = SIZE(buf, index) - 1;
            break;
    }

    if (from > 0) {
        copy(value, INDEX(index + 2 + from), min(len, maxlength));
        if (length) *length = len;
        return true;
    } else
        return false;
}