/*
Copyright (c) 2013, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "XBee.h"

#define LOCK()      NVIC_DisableIRQ(UARTx_IRQn[_serial.index])
#define UNLOCK()    NVIC_EnableIRQ(UARTx_IRQn[_serial.index])
#define INDEX(n)    ((n) % BUFSIZE)
#define SIZE(b, i)  (b[i] << 8 | b[INDEX(i + 1)])

const IRQn_Type UARTx_IRQn[] = {UART0_IRQn, UART1_IRQn, UART2_IRQn, UART3_IRQn};

XBee::FrameType XBee::receive(float timeout) {
    timer.reset();
    flush();

    while (true) {
        LOCK();
        if (out != in || free == 0) {
            FrameType type = getFrameType(buf[INDEX(out + 2)]);
            if (type != None) {
                received = out;
                UNLOCK();
                return type;
            }
            int size = SIZE(buf, out);
            out = INDEX(out + 2 + size);
            free += (2 + size);
            UNLOCK();
            continue;
        } else if (timeout <= timer.read()) {
            UNLOCK();
            return None;
        } else {
            UNLOCK();
            sem.wait((int) ((timeout - timer.read()) * 1000));
        }
    }
}

/*
 * function seekFor() -
 *
 * searches buf[] in range between the index out and in for a frame of specified type and id.
 * if the frame is found before timeout, returns the index of the packet, otherwise -1.
 *
 */
int XBee::seekFor(FrameType type, char id, float timeout) {
    timer.reset();
    LOCK();

    while (out != in && getFrameType(buf[INDEX(out + 2)]) == None) {
        int size = SIZE(buf, out);
        out = INDEX(out + 2 + size);
        free += 2 + size;
    }

    int index = out;
    while (true) {
        if (index != in) {
            if (getFrameType(buf[INDEX(index + 2)]) == type && (id == 0 || buf[INDEX(index + 3)] == id)) {
                UNLOCK();
                return index;
            }
            int size = SIZE(buf, index);
            index = INDEX(index + 2 + size);
            continue;
        } else if (timeout <= timer.read()) {
            UNLOCK();
            return -1;
        } else {
            UNLOCK();
            sem.wait((int) ((timeout - timer.read()) * 1000));
            LOCK();
        }
    }
}

XBee::FrameType XBee::getFrameType(char c) {
    switch (c) {
        case 0x00:
            return None;
        case 0x88:
            return ATCommandResponse;
        case 0x8A:
            return ModemStatus;
        case 0x8B:
            return ZigBeeTransmitStatus;
        case 0x90:
            return ZigBeeReceivePacket;
        case 0x91:
            return ZigBeeExplicitRxIndicator;
        case 0x92:
            return ZigBeeIODataSampleRxIndicator;
        case 0x94:
            return XBeeSensorReadIndicator;
        case 0x95:
            return NodeIdentificationIndicator;
        case 0x97:
            return RemoteCommandResponse;
        default:
            return Other;
    }
}

void XBee::flush() {
    LOCK();
    if (received == out) {
        do {
            int size = SIZE(buf, out);
            out = INDEX(out + 2 + size);
            free += 2 + size;
        } while (out != in && getFrameType(buf[INDEX(out + 2)]) == None);
    }
    UNLOCK();
}

void XBee::rxISR() {
    static enum {UNKNOWN, LENGTH1, LENGTH2, DATA, SUMCHECK} state = UNKNOWN;
    static bool escaped = false;
    int c = -1;
    
    switch (_serial.index) {
        case 0: {// USBTX, USBRX
            uint32_t UART_0_IIR = LPC_UART0->IIR;
            if (readable()) c = LPC_UART0->RBR;
        }
        break;
        case 1: {// p13/14
            uint32_t UART_1_IIR = LPC_UART1->IIR;
            if (readable()) c = LPC_UART1->RBR;
        }
        break;
        case 2: {// p28/27
            uint32_t UART_2_IIR = LPC_UART2->IIR;
            if (readable()) c = LPC_UART2->RBR;
        }
        break;
        case 3: {// p9/10
            uint32_t UART_3_IIR = LPC_UART3->IIR;
            if (readable()) c = LPC_UART3->RBR;
        }
        break;
    }

    if (c != -1) {
        if (api == 2) {
            if (escaped) {
                c ^= 0x20;
                escaped = false;
            } else if (c == ESCAPE) {
                escaped = true;
                return;
            }
        }

        switch (state) {
            case LENGTH1:
                cur = in;
                buf[cur] = c;
                state = LENGTH2;
                break;
                
            case LENGTH2:
                if ((buf[cur] << 8 | c) + 2 < BUFSIZE) {
                    state = DATA;
                    while (free < (buf[cur] << 8 | c) + 2) {
                        int size = SIZE(buf, out);
                        out = INDEX(out + size + 2);
                        free += (size + 2);
                    }
                    buf[INDEX(cur + 1)] = c;
                    cur = INDEX(cur + 2);
                } else
                    state = UNKNOWN;
                break;
                
            case DATA:
                buf[cur] = c;
                cur = INDEX(cur + 1);
                if (cur == INDEX(in + 2 + SIZE(buf, in)))
                    state = SUMCHECK;
                break;
                
            case SUMCHECK:
                for (int i = INDEX(in + 2); i != cur; i = INDEX(i + 1)) {
                    c += buf[i];
                }
                if ((c & 255) == 255) {
                    in = cur;
                    free = in <= out ? out - in : BUFSIZE + out - in; // maybe in == out, then free == 0, but != BUFSIZE
                }
                state = UNKNOWN;
                sem.release();
                break;
                
            default:
                if (c == PREAMBLE)
                    state = LENGTH1;
        }
    }
}