/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef XBEE_H
#define XBEE_H

#include "mbed.h"
#include "XBeeDataTypes.h"

#define min(x, y)       ((x) < (y) ? (x) : (y))
#define INDEX(n)        ((n) % BUFSIZE)
#define SIZE(b, i)      (b[i] << 8 | b[INDEX(i + 1)])

const int BUFSIZE = 512;
const char ESCAPE = 0x7D;
const char PREAMBLE = 0x7E;

/**
 * class for XBee module API mode interface
 */
class XBee : public Serial {
public:

    /**
     * Frame type declaration for XBee API frames
     */
    enum FrameType {
        /**
        * Empty data
        */
        None = 0,
        
        /**
         *  AT Command Response API frame
         */
        ATCommandResponse,
        
        /**
         * Modem Status API frame
         */
        ModemStatus,
        
        /**
         * ZigBee Transmit Status API frame
         */
        ZigBeeTransmitStatus,
        
        /**
         * ZigBee Receive Packet API frame
         */
        ZigBeeReceivePacket,

        /**
         *  ZigBee Explicit Rx Indicator API frame
         */
        ZigBeeExplicitRxIndicator,

        /**
         * ZigBee I/O Data Sample Rx Indicator API frame
         */
        ZigBeeIODataSampleRxIndicator,

        /**
         * XBee Sensor Read Indicator API frame
         */
        XBeeSensorReadIndicator,

        /**
         * Node Identification Indicator API frame
         */
        NodeIdentificationIndicator,

        /**
         * Remote Command Response API frame
         */
        RemoteCommandResponse,

        /**
         * Unknown API frame
         */
        Other
    };
    
    /**
     * Value type declarations for retrieving frame data contents
     */
    enum ValueType {
        /**
         * Frame ID
         */
        FrameID,
        
        /**
         *AT command name
         */
        ATCommand,
        
        /**
         * Status
         */       
        Status,
        
        /**
         * Command data
         */
        CommandData,
        
        /**
         * 16 bit address
         */
        Address16,
        
        /**
         * 64 bit address
         */
        Address64,
        
        /**
         * Retry count
         */
        RetryCount,
        
        /**
         * Delivery status
         */
        DeliveryStatus,
        
        /**
         * Discovery status
         */
        DiscoveryStatus,
        
        /**
         * Receive option
         */
        ReceiveOptions,
        
        /**
         * Received data
         */
        ReceivedData,
        
        /**
         * Raw data
         */
        RawData
    };


    /**
     * creates an XBee interface object.
     *
     * @param ser Serial object through which XBee module is connected to mbed
     * @param apiMode API mode either 1 or 2 (use escape; default)
     * @param debug display debugging (I/O state) information through LEDs
     */
    XBee(Serial& ser, int apiMode = 2, bool debug = false);


    /**
     * creates an XBee interface object.
     *
     * @param tx TX pin connected to XBee
     * @param rx RX pin connected to XBee
     * @param apiMode API mode either 1 or 2 (use escape; default)
     * @param debug display debugging (I/O state) information through LEDs
     */
    XBee(PinName tx, PinName rx, int apiMode = 2, bool debug = false);

    /**
     * creates an XBee interface object.
     *
     * @param ser Serial object through which XBee module is connected to mbed
     * @param mon alternate Serial object for monitoring (use serial ports other than USBTX/USBRX)
     * @param apiMode API mode either 1 or 2 (use escape; default)
     * @param debug display debugging (I/O state) information through LEDs
     */
    XBee(Serial& ser, Serial& mon, int apiMode = 2, bool debug = false);

    /**
     * creates an XBee interface object.
     *
     * @param tx TX pin connected to XBee
     * @param rx RX pin connected to XBee
     * @param mon alternate Serial object for monitoring (use serial ports other than USBTX/USBRX)
     * @param apiMode API mode either 1 or 2 (use escape; default)
     * @param debug display debugging (I/O state) information through LEDs
     */
    XBee(PinName tx, PinName rx, Serial& mon, int apiMode = 2, bool debug = false);
    
    /**
     * initializes XBee module.
     *
     * issues VR command to test XBee modem connection
     *
     * @returns true if initialization succeeded, false otherwise
     */
    bool init(float timeout = 15.0);

    /**
     * sets destination addresses.
     *
     * @param address64 XBeeAddress64 type address
     * @param address16 XBeeAddress16 type address (optional)
     */
    void setDestination(XBeeAddress64 address64, XBeeAddress16 address16 = 0xFFFE);
    
    /**
     * sets destination addresses.
     *
     * @param address64 64-bit destination address in uint64_t
     * @param address16 16-bit destination address in uint16_t
     */
    void setDestination(uint64_t address64, uint16_t address16 = 0xFFFE);

    /**
      * sets destination addresses.
      *
      * @param address64 64-bit destination address in bytes (big endian)
      * @param address16 16-bit destination address in bytes
      */
    void setDestination(char address64[], char address16[]);
    
    /**
     * sends an AT command.
     *
     * @param command AT command char string
     * @param param parameter to the AT command
     */
    void sendCommand(const char *command, int8_t param, bool queue = false);
    void sendCommand(const char *command, int16_t param, bool queue = false);
    void sendCommand(const char *command, int32_t param, bool queue = false);
    void sendCommand(const char *command, int64_t param, bool queue = false);
    void sendCommand(const char *command, uint8_t param, bool queue = false);
    void sendCommand(const char *command, uint16_t param, bool queue = false);
    void sendCommand(const char *command, uint32_t param, bool queue = false);
    void sendCommand(const char *command, uint64_t param, bool queue = false);
    void sendCommand(const char *command, const char *param, bool queue = false);
    
    /**
     * sends an AT command.
     *
     * @param command AT command char string
     * @param param parameter to the AT command (pointer to byte array)
     * @param param_length parameter length in bytes
     * @param queue true if command paramter is to be queued
     */
    void sendCommand(const char *command, const uint8_t *param = 0, int param_length = 0, bool queue = false);

    /**
     * sends a remote AT command.
     *
     * sends an AT command to the XBee(s) at the destination address
     *
     * @param command AT command char string
     * @param param parameter to the AT command (pointer to byte array)
     * @param param_length parameter length in bytes
     * @param options remote command options
     */
    void sendRemoteCommand(const char *command, int8_t param);
    void sendRemoteCommand(const char *command, int16_t param);
    void sendRemoteCommand(const char *command, int32_t param);
    void sendRemoteCommand(const char *command, int64_t param);
    void sendRemoteCommand(const char *command, uint8_t param);
    void sendRemoteCommand(const char *command, uint16_t param);
    void sendRemoteCommand(const char *command, uint32_t param);
    void sendRemoteCommand(const char *command, uint64_t param);
    void sendRemoteCommand(const char *command, const char *param);
    
    /**
     * sends a remote AT command.
     *
     * sends an AT command to the XBee(s) at the destination address
     *
     * @param command AT command char string
     * @param param parameter to the AT command (pointer to byte array)
     * @param param_length parameter length in bytes
     * @param options remote command options
     */
    void sendRemoteCommand(const char *command, const uint8_t *param = 0, int param_length = 0, char options = 0x02);

    /**
     * executes an AT command and gets the result.
     *
     * @param command AT command char string
     * @param param parameter to the AT command
     *
     * @returns pointer to the command result, if the result is a number (char, short, long, int64_t),
     *          the address to the number will be returned; otherwise the address to the byte array
     *          containing the command response will be returned.
     */
    void *executeCommand(const char *command, int8_t param);
    void *executeCommand(const char *command, int16_t param);
    void *executeCommand(const char *command, int32_t param);
    void *executeCommand(const char *command, int64_t param);
    void *executeCommand(const char *command, uint8_t param);
    void *executeCommand(const char *command, uint16_t param);
    void *executeCommand(const char *command, uint32_t param);
    void *executeCommand(const char *command, uint64_t param);
    void *executeCommand(const char *command, const char *param);

    /**
     * executes an AT command and gets the result.
     *
     * @param command AT command char string
     * @param param parameter to the AT command (pointer to byte array)
     * @param param_length parameter length in bytes
     *
     * @returns pointer to the command result, if the result is a number (char, short, long, long long),
     *          the address to the number will be returned; otherwise the address to the byte array
     *          containing the command response will be returned.
     */
    void *executeCommand(const char *command, const uint8_t *param = 0, int laram_length = 0);

    /**
     * sends data to the XBee(s) at the destination address.
     *
     * @param data address to the data (byte array)
     * @param length data length in bytes
     * @param confirm if true, checks the transmission status frame and returns the result
     */
    bool send(const char *data, int length, bool confirm);

    /**
     * sends data and confirm the transmit status.
     *
     * @param data address to the data (byte array)
     * @param length data length in bytes
     *
     * @returns true if sent successfully, false otherwise (either timeout or send error occurred)
     */
    bool sendConfirm(const char *data, int length);

    /**
     * sends data to the destination using printf format.
     *
     * @param format printf format string, followed by corresponding arguments
     *
     * @returns the number of charancters sent, or negative if error occurred
     */
    int printf(const char *format, ...);

    /**
     * receives data frame from the XBee module.
     *
     * @param timeout seconds bofer time out
     *
     * @returns FrameType of the received frame data
     */
    FrameType receive(float timeout = 3.0);

    /**
     * scan received data
     *
     * @param data XBeeDataType data to be scanned
     *
     * @param true if scan succeeded, false otherwise
     */
    bool scan(XBeeFrameID& id) { return scan(XBee::FrameID, id.raw_address(), 1); }
    bool scan(XBeeRetryCount& count) { return scan(XBee::RetryCount, count.raw_address(), 1); }
    bool scan(XBeeStatus& status) { return scan(XBee::Status, status.raw_address(), 1); }
    bool scan(XBeeDeliveryStatus& status) { return scan(XBee::DeliveryStatus, status.raw_address(), 1); }
    bool scan(XBeeDiscoveryStatus& status) { return scan(XBee::DiscoveryStatus, status.raw_address(), 1); }
    bool scan(XBeeReceiveOptions& options) { return scan(XBee::ReceiveOptions, options.raw_address(), 1); }
    bool scan(XBeeAddress64& address64) { return scan(XBee::Address64, address64.raw_address(), 8); }
    bool scan(XBeeAddress16& address16) { return scan(XBee::Address16, address16.raw_address(), 2); }
    bool scan(XBeeATCommand& command) { return scan(XBee::ATCommand, command.raw_address(), 3); }
    bool scan(XBeeCommandData& data) { return scan(XBee::CommandData, data.raw_address(), data.capacity, &data.size); }
    bool scan(XBeeReceivedData& data) { return scan(XBee::ReceivedData, data.raw_address(), data.capacity, &data.size); }
    bool scan(XBeeRawData& data) { return scan(XBee::RawData, data.raw_address(), data.capacity, &data.size); }
    
    /**
     * scan received data according to the specified format.
     *
     * @param type ValueType of the data to be scanned
     * @param value pointer to the byte array to store the scanned value
     * @param maxlength max size of the value in bytes
     * @param length pointer to an int to receive the actual data length
     *
     * @param true if scan succeeded, false otherwise
     */
    bool scan(ValueType type, char *value, int maxlength = 1, int *length = 0);

    /**
     * gets the XBee firmware version.
     *
     * @returns XBee firmwre version in int (unsigned short value)
     */
    int getFirmwareVersion();

    /**
     * gets the current frame ID.
     *
     * @returns frame ID number being used in the next send request
     */
    char getFrameID();

    /**
    * sets to run in debug mode.
    *
    * @param debug true to display debugging information
    */
    void setDebug(bool debug);

    /**
    * displays received data in dump format.
    */
    void dump();

    /**
     * displays the internal data fields and receive buffer in dump format.
     */
    void dumpAll();

    /**
     * operator overloading for testing XBee modem connection status.
     *
     * @returns false if XBee modem connection has an error
     */
    operator bool();

private:
    Serial mon;
    BusOut leds;
    int apiMode;
    volatile enum {UNKNOWN, LENGTH1, LENGTH2, DATA, SUMCHECK} state;
    volatile int cur, in, out, received, free;
    char frame_id;
    char destination64[8];
    char destination16[2];
    char buf[BUFSIZE];
    bool debug;

    void send(char c);
    void send2(char c);
    void sendFrame(const char *frame, int length);
    int createTxRequest(char frame_id, const char *data, int data_length, char *buf, int buf_size);
    int createAtRequest(char frame_id, const char *command, const uint8_t *param, int param_length, bool queue, char *buf, int buf_size);
    int createRemoteAtRequest(char frame_id, const char *command, const uint8_t *param, int param_length, char options, char *buf, int buf_size);
    int seekFor(FrameType type, float timeout);
    FrameType getFrameType(char c);
    bool scan(int i, ValueType type, char *value, int maxlength = 1, int *length = 0);
    void flush();
    void rxInterruptHandler();
    void rxInterruptHandler2();

    void dump(const char *data, int length);
    void dumpIOSample(const char *data, int length);
    void copy(char *toBuf, int fromIndex, int length);
};

#endif