/*
Copyright (c) 2012, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

$Id: XBeeDataTypes.h,v 1.2 2012/03/14 06:31:54 yamaguchi Exp yamaguchi $
*/

#ifndef XBEE_DATA_TYPES_H
#define XBEE_DATA_TYPES_H

#include "mbed.h"

class XBeeAddress64 {
public:
    XBeeAddress64();
    XBeeAddress64(const char *address);
    XBeeAddress64(uint64_t address);
    XBeeAddress64(uint32_t high, uint32_t low);

    operator uint64_t();
    bool operator==(XBeeAddress64& address);
    bool operator==(const char* address);
    bool operator==(uint64_t address);
    operator char *();
    
    char operator [](int index) const { 
        return address64[index]; 
    };

    char& operator [](int index) { 
        return address64[index]; 
    };
    
    friend class XBee;
    
private:
    char address64[8];
    char buf[18];
    
    char *raw_address() { return (char *) address64; }
};

const XBeeAddress64 COORDINATOR(0, 0);
const XBeeAddress64 BROADCAST(0, 0xFFFF);

class XBeeAddress16 {
public:
    XBeeAddress16();
    XBeeAddress16(const char *address);
    XBeeAddress16(uint16_t address);
    XBeeAddress16(char high, char low);

    operator uint16_t();
    bool operator==(XBeeAddress16& address);
    bool operator==(const char* address);
    bool operator==(uint16_t address);
    operator char *();
    
    char operator [](int index) const { 
        return address16[index]; 
    };
    char& operator [](int index) { 
        return address16[index]; 
    };

    friend class XBee;
    
private:
    char address16[2];
    char buf[5];
    
    char *raw_address() { return (char *) address16; }
};

class XBeeData {
public:
    XBeeData(int capacity);
    ~XBeeData();
    
    int length() {
        return size;
    }

    operator char *() const {
         return data;
    }
    
    char operator [](int index) const { 
         return data[index]; 
    };

    char& operator [](int index) { 
         return data[index]; 
    };

    friend class XBee;
    friend class XBeeCommandData;
    friend class XBeeReceivedData;
    friend class XBeeRawData;
  
private:
    char *data;
    int capacity;
    int size;
    
    char *raw_address() { return (char *) data; }
};

class XBeeCommandData : public XBeeData {
public:
    XBeeCommandData(int capacity = 20) : XBeeData(capacity) {}
};

class XBeeReceivedData : public XBeeData {
public:
    XBeeReceivedData(int capacity = 128) : XBeeData(capacity) {}
};

class XBeeRawData : public XBeeData {
public:
    XBeeRawData(int capacity = 128) : XBeeData(capacity) {}    
};

class XBeeATCommand : public XBeeData {
public:
    XBeeATCommand() : XBeeData(3) {}
};

class BitArray {
public:
    BitArray(int mask, int values);
    int operator [](int i) const;
    
private:
    int mask;
    int values;

    int valueAt(int i) const;
};

class IntArray {
public:
    IntArray(int mask, const char *value);
    int operator [](int i) const;

private:
    int mask;
    const char *values;

    int valueAt(int i) const;
};

class IOSample {
public:
    IOSample(const char *data);

    operator char *();

    BitArray dio;
    IntArray ad;
private:
    char buf[128];
};

class XBeeUint8 {
public:
    XBeeUint8(char data = 0, char base = 10) {
        this->data = data;
    }
    
    operator int() {
        return data;
    }
    
    operator char *() {
        sprintf(buf, base == 10 ? "%d" : "%02X", data);
        return buf;
    } 
    
    friend class XBee;
    friend class XBeeFrameID;
    friend class XBeeRetryCount;
    friend class XBeeStatus;
    friend class XBeeDeliveryStatus;
    friend class XBeeDiscoveryStatus;
    friend class XBeeReceiveOptions;
 
private:
    char data;
    char base;
    char buf[4];
    
    char *raw_address() {
         return &data;
    }
};

class XBeeFrameID : public XBeeUint8 {
public:
    XBeeFrameID(char id = 0) : XBeeUint8(id) {}
};

class XBeeRetryCount : public XBeeUint8 {
public:
    XBeeRetryCount(char count = 0) : XBeeUint8(count) {}
};

class XBeeStatus : public XBeeUint8 {
public:
    XBeeStatus(char data = 0) : XBeeUint8(data, 16) {}
};

class XBeeDeliveryStatus : public XBeeUint8 {
public:
    XBeeDeliveryStatus(char data = 0) : XBeeUint8(data, 16) {}
};

class XBeeDiscoveryStatus : public XBeeUint8 {
public:
    XBeeDiscoveryStatus(char data = 0) : XBeeUint8(data, 16) {}
};

class XBeeReceiveOptions : public XBeeUint8 {
public:
    XBeeReceiveOptions(char data = 0) : XBeeUint8(data, 16) {}
};

#endif
