/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "XBee.h"

XBee::XBee(Serial& ser, int apiMode, bool debug)
        : Serial(ser), mon(USBTX, USBRX), leds(LED1, LED2, LED3, LED4), apiMode(apiMode),
        state(UNKNOWN), in(0), out(0), received(-1), free(BUFSIZE), frame_id(1), debug(debug) {
    memset(buf, 0, BUFSIZE);
    attach(this, apiMode == 1 ? &XBee::rxInterruptHandler : &XBee::rxInterruptHandler2, Serial::RxIrq);
}

XBee::XBee(PinName tx, PinName rx, int apiMode, bool debug)
        : Serial(tx, rx), mon(USBTX, USBRX), leds(LED1, LED2, LED3, LED4), apiMode(apiMode),
        state(UNKNOWN), in(0), out(0), received(-1), free(BUFSIZE), frame_id(1), debug(debug) {
    memset(buf, 0, BUFSIZE);
    attach(this, apiMode == 1 ? &XBee::rxInterruptHandler : &XBee::rxInterruptHandler2, Serial::RxIrq);
}

XBee::XBee(Serial& ser, Serial& mon, int apiMode, bool debug)
        : Serial(ser), mon(mon), leds(LED1, LED2, LED3, LED4), apiMode(apiMode),
        state(UNKNOWN), in(0), out(0), received(-1), free(BUFSIZE), frame_id(1), debug(debug) {
    memset(buf, 0, BUFSIZE);
    attach(this, apiMode == 1 ? &XBee::rxInterruptHandler : &XBee::rxInterruptHandler2, Serial::RxIrq);
}

XBee::XBee(PinName tx, PinName rx, Serial& mon, int apiMode, bool debug)
        : Serial(tx, rx), mon(mon), leds(LED1, LED2, LED3, LED4), apiMode(apiMode),
        state(UNKNOWN), in(0), out(0), received(-1), free(BUFSIZE), frame_id(1), debug(debug) {
    memset(buf, 0, BUFSIZE);
    attach(this, apiMode == 1 ? &XBee::rxInterruptHandler : &XBee::rxInterruptHandler2, Serial::RxIrq);
}

bool XBee::init(float timeout) {
    while (readable()) getc();
    while (timeout > 0 && getFirmwareVersion() == -1) {
        timeout -= 1;
        wait(0.8);
    }

    switch (getFirmwareVersion() & 0xFF00) {
        case 0x2100:
            setDestination(0xFFFFULL, 0xFFFE);
            return true;

        case 0x2300:
        case 0x2900:
            setDestination(0x00ULL);
            return true;

        default:
            return false;
    }
}

void XBee::setDestination(XBeeAddress64 address64, XBeeAddress16 address16) {
    setDestination(address64.raw_address(), address16.raw_address());
}

void XBee::setDestination(uint64_t address64, uint16_t address16) {
    for (int i = 0; i < 8; i++, address64 >>= 8)
        destination64[7 - i] = address64 & 255;
    destination16[0] = (address16 >> 8) & 255;
    destination16[1] = address16 & 255;
}

void XBee::setDestination(char address64[], char address16[]) {
    memcpy(destination64, address64, 8);
    destination16[0] = address16[0];
    destination16[1] = address16[1];
}

XBee::operator bool() {
    return getFirmwareVersion() != -1;
}

char XBee::getFrameID() {
    return frame_id;
}

void *XBee::executeCommand(const char *command, int8_t param) {
    return executeCommand(command, (uint64_t) param);
}

void *XBee::executeCommand(const char *command, int16_t param) {
    return executeCommand(command, (uint64_t) param);
}

void *XBee::executeCommand(const char *command, int32_t param) {
    return executeCommand(command, (uint64_t) param);
}

void *XBee::executeCommand(const char *command, int64_t param) {
    return executeCommand(command, (uint64_t) param);
}

void *XBee::executeCommand(const char *command, uint8_t param) {
    return executeCommand(command, (uint64_t) param);
}

void *XBee::executeCommand(const char *command, uint16_t param) {
    return executeCommand(command, (uint64_t) param);
}

void *XBee::executeCommand(const char *command, uint32_t param) {
    return executeCommand(command, (uint64_t) param);
}

void *XBee::executeCommand(const char *command, uint64_t param) {
    uint8_t param_buf[8] = {
        (uint8_t) (param >> 56), (uint8_t) (param >> 48),
        (uint8_t) (param >> 40), (uint8_t) (param >> 32),
        (uint8_t) (param >> 24), (uint8_t) (param >> 16),
        (uint8_t) (param >> 8), (uint8_t) param
    };

    int n = 0;
    while (n < 7 && param_buf[n] == 0) n++;

    return executeCommand(command, param_buf + n, 8 - n);
}

void *XBee::executeCommand(const char *command, const char *param) {
    return executeCommand(command, (uint8_t *) param, strlen(param));
}

void *XBee::executeCommand(const char *command, const uint8_t *param, int param_length) {
    static char data[21];
    bool succeeded = false;
    char id = getFrameID();

    sendCommand(command, param, param_length);
    int index = seekFor(XBee::ATCommandResponse, 0.1);
    if (index != -1) {
        char id2, status;
        if (scan(index, FrameID, &id2) && id == id2 && scan(index, Status, &status) && status == 0) {
            memset(data, 0, sizeof(data));
            if ((command[0] == 'N' && command[1] == 'I') ||
                    (command[0] == 'I' && command[1] == 'S')) {
                if (scan(index, CommandData, data, sizeof(data)))
                    succeeded = true;
            } else {
                char buf2[8];
                int length;
                if (scan(index, CommandData, buf2, sizeof(buf2), &length) && length <= sizeof(buf2)) {
                    for (int i = 0; i < length; i++)
                        data[i] = buf2[length - i - 1];
                    succeeded = true;
                }
            }
        }
        buf[INDEX(index + 2)] = 0x00;
    }

    return succeeded ? data : 0;
}

int XBee::getFirmwareVersion() {
    static int firmwareVersion = -1;

    if (firmwareVersion == -1) {
        unsigned short *data = (unsigned short *) executeCommand("VR");
        if (data) firmwareVersion = *data; //data[0] << 8 | data[1];
    }
    return firmwareVersion;
}

void XBee::copy(char *toBuf, int fromIndex, int length) {
    int length1 = min(BUFSIZE - fromIndex, length);
    memcpy(toBuf, &buf[fromIndex], length1);
    if (length1 < length)
        memcpy(toBuf + length1, &buf[0], length - length1);
}

void XBee::setDebug(bool debug) {
    this->debug = debug;
}