/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef PACHUBE_CLIENT_H
#define PACHUBE_CLIENT_H

#include "SimpleSocket.h"
#include "Utils.h"

/**
 * Client API for accessing Pachube
 */
class PachubeClient {
public:
    /**
     * creates an PachubeClient object
     *
     * @param feedid Feed ID
     * @param apikey API Key
     * @param verbose if true display debug info
     */
    PachubeClient(int feedid, char *apikey, bool verbose = false)
            : feedid(feedid), length(0), verbose(verbose) {
        strcpy(this->apikey, apikey);
    }

    /**
     * creates an PachubeClient object
     *
     * @param filename name of the configuration file
     * @param verbose if true display debug info
     */
    static PachubeClient create(char *filename, bool verbose = false) {
        int feedid = 0;
        char apikey[64] = {};

        if (filename) {
            char path[32];
            LocalFileSystem local("local");
            sprintf(path, "/local/%s", filename);
            if (FILE *fp = fopen(path, "r")) {
                Utils::fgetValues(fp, "feed-id:%d", &feedid);
                Utils::fgetValues(fp, "api-key:%s", apikey);
                fclose(fp);
                if (verbose) {
                    printf("feed-id = %d\n", feedid);
                    printf("api-key = %s\n", apikey);
                }
            }
        }

        return PachubeClient(feedid, apikey, verbose);
    }

    /**
     * adds data to the Pachube datastream
     *
     * @param id Datastream ID
     * @param value value to add
     */
    void add(int id, float value) {
        length += sprintf(&data[length], "%d,%f\n", id, value);
    }

    /**
     * adds data to the Pachube datastream
     *
     * @param id Datastream ID
     * @param value value to add
     */
    void add(char *id, float value) {
        length += sprintf(&data[length], "%s,%f\n", id, value);
    }

    /**
     * updates datastreams
     *
     * @returns true if succeeded
     */
    bool update() {
        bool result = false;

        ClientSocket client("api.pachube.com", 80);
        if (client) {
            const char *request = "PUT /v2/feeds/%d.csv HTTP/1.1\r\n"
                                  "Host: api.pachube.com\r\n"
                                  "X-PachubeApiKey: %s\r\n"
                                  "Content-Type: text/csv\r\n"
                                  "Content-Length: %d\r\n"
                                  "Connection: close\r\n\r\n";

            client.printf(request, feedid, apikey, length);
            client.write(data, length);
            if (verbose) {
                printf(request, feedid, apikey, length);
                printf("%s", data);
            }

            while (client) {
                if (client.available()) {
                    while (client.available()) {
                        char response[128] = {};
                        client.read(response, sizeof(response) - 1);
                        if (!result && strncmp(response, "HTTP/1.1 200", 12) == 0)
                            result = true;
                        if (verbose)
                            printf("%s", response);
                    }
                    client.close();
                }
            }
        }
        length = 0;
        return result;
    }

private:
    int feedid;
    int length;
    bool verbose;
    char apikey[64];
    char data[64];
};

#endif