/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef STATION_H
#define STATION_H

#include "SimpleSocket.h"
#include <map>
#include <string>

/**
 * A simple command/response network interface for mbed
 */
class Station {
public:
    /**
     * creates a Station object
     *
     * @param port port number
     */
    Station(int port) : server(port) {}

    ~Station() {}

    /**
     * adds a command handler
     *
     * @param command name of the command
     * @param handler pointer to the handler of the command
     */
    void addHandler(string command, void (*handler)(void)) {
        handlerMap.insert(pair<string, FunctionPointer>(command, FunctionPointer(handler)));
    }
    
    /**
     * adds a command handler
     *
     * @param command name of the command
     * @param tptr pointer to the object to call the handler on
     * @param handler pointer to the handler of the command
     */
    template <typename T> void addHandler(string command, T* tptr, void (T::*handler)(void)) {
        handlerMap.insert(pair<string, FunctionPointer>(command, FunctionPointer(tptr, handler)));
    }

    /**
     * sets the buffer for communication between caller and callee
     *
     * @param buf pointer to the communication buffer
     * @param length length of the buffer
     */
    void setBuffer(char *buf, int length) {
        this->buf = buf;
        this->length = length;
    }

    /**
     * Check for any client and handle its request. This function is expected to be called repeatedly in main loop.
     */
    void handleClient() {
        ClientSocket client = server.accept(0);
        while (client) {
            if (client.available()) {
                memset(buf, '\0', length);
                int len = client.read(buf, this->length - 1);
                char name[16] = {};
                sscanf(buf, "%s", name);
                callHandler(string(name));
                if (buf[0]) {
                    client.write(buf, strlen(buf));
                }
                client.close();
            }
        }
    }

private:
    ServerSocket server;
    map<string, FunctionPointer> handlerMap;
    char *buf;
    int length;

    void callHandler(string name) {
        for (map<string, FunctionPointer>::iterator it = handlerMap.begin(); it != handlerMap.end(); it++) {
            if (name.compare(it->first) == 0) {
                (it->second).call();
                break;
            }
        }
    }
};

#endif