/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "EthernetNetIf.h"
#include "Utils.h"

/**
 * A wrapper class for EthernetNetIf to create an instance from a config file
 */
class Ether {
public:
    /**
     * creates Ether instance
     *
     * @param verbose if true display debug info
     */
    Ether(bool verbose = false)
            : active(false), verbose(verbose) {
    }

    /**
     * creates Ether instance
     *
     * @param ip IP address
     * @param subnet subnet mask
     * @param gateway gateway address
     * @param dns DNS server address
     * @param verbose if true display debug info
     */
    Ether(IpAddr ip, IpAddr subnet, IpAddr gateway, IpAddr dns, bool verbose = false)
            : eth(ip, subnet, gateway, dns), active(false), verbose(verbose) {
    }

    /**
     * creates Ether instance
     *
     * @param ip IP address
     * @param subnet subnet mask
     * @param gateway gateway address
     * @param dns DNS server address
     * @param verbose if true display debug info
     */
    static Ether create(IpAddr ip, IpAddr subnet, IpAddr gateway, IpAddr dns, bool verbose = false) {
        return Ether(ip, subnet, gateway, dns, verbose);
    }

    /**
     * creates Ether instance
     *
     * @param filename configuration file containing IP address, subnet mask, gateway and DNS address
     * @param verbose if true display debug info
     */
    static Ether create(char *filename = 0, bool verbose = false) {
        if (filename) {
            char path[32];
            LocalFileSystem local("local");
            sprintf(path, "/local/%s", filename);
            if (FILE *fp = fopen(path, "r")) {
                char b1, b2, b3, b4;
                IpAddr ip = Utils::fgetValues(fp, "ip-address:%hhu.%hhu.%hhu.%hhu", &b1, &b2, &b3, &b4) ? IpAddr(b1, b2, b3, b4) : IpAddr();
                IpAddr subnet = Utils::fgetValues(fp, "subnet-mask:%hhu.%hhu.%hhu.%hhu", &b1, &b2, &b3, &b4) ? IpAddr(b1, b2, b3, b4) : IpAddr();
                IpAddr gateway = Utils::fgetValues(fp, "gateway-address:%hhu.%hhu.%hhu.%hhu", &b1, &b2, &b3, &b4) ? IpAddr(b1, b2, b3, b4) : IpAddr();
                IpAddr dns = Utils::fgetValues(fp, "dns-address:%hhu.%hhu.%hhu.%hhu", &b1, &b2, &b3, &b4) ? IpAddr(b1, b2, b3, b4) : IpAddr();
                fclose(fp);
                if (verbose) {
                    printf("ip-address:%hhu.%hhu.%hhu.%hhu\n", ip[0], ip[1], ip[2], ip[3]);
                    printf("subnet-mask:%hhu.%hhu.%hhu.%hhu\n", subnet[0], subnet[1], subnet[2], subnet[3]);
                    printf("gateway-address:%hhu.%hhu.%hhu.%hhu\n", gateway[0], gateway[1], gateway[2], gateway[3]);
                    printf("dns-address:%hhu.%hhu.%hhu.%hhu\n", dns[0], dns[1], dns[2], dns[3]);
                }

                return Ether(ip, subnet, gateway, dns, verbose);
            }
        }

        return Ether();
    }

    /**
     * sets up Ethernet interface
     */
    void setup() {
        EthernetErr err = eth.setup();
        active = err == ETH_OK;
        if (verbose) {
            printf("EthernetNetIf.setup() = %d\n", err);
        }
    }

    /**
     * returns Ethernet status
     *
     * @returns true if setup succeeded
     */
    bool isActive() {
        return active;
    }

    /**
     * returns its IP address in bytes
     *
     * @returns IP address in byte array
     */
    char *getIpAddress() {
        static char ip[] = {eth.getIp()[0], eth.getIp()[1], eth.getIp()[2], eth.getIp()[3]};
        return ip;
    }

private:
    EthernetNetIf eth;
    bool active;
    bool verbose;
};