/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef SUPERTWEET_H
#define SUPERTWEET_H

#include "SimpleSocket.h"
#include "Utils.h"

/**
 * Client API for accessing SuperTweet gateway
 */
class SuperTweetClient {
public:
    /**
     * creates a SuperTweetClient object
     *
     * @param user user account
     * @param password password (in plain text)
     * @param messageTemplate message template for the tweet message
     * @param verbose if true display debug info
     */
    SuperTweetClient(char *user, char *password, char *messageTemplate, bool verbose = false)
            : verbose(verbose) {
        char buf[64];
        sprintf(buf, "%s:%s", user, password);
        Utils::encodeBase64(buf, credential);
        strcpy(this->messageTemplate, messageTemplate);
    }

    /**
     * creates a SuperTweetCient object from a config file
     *
     * @param filename name of the configuration file
     * @param verbose if true display debug info
     */
    static SuperTweetClient create(char *filename, bool verbose = false) {
        char user[16], password[32], messageTemplate[420];

        if (filename) {
            char path[32];
            LocalFileSystem local("local");
            sprintf(path, "/local/%s", filename);
            if (FILE *fp = fopen(path, "r")) {
                Utils::fgetValues(fp, "user:%s", user);
                Utils::fgetValues(fp, "password:%s", password);
                Utils::fgetValues(fp, "message:%[^\r\n]", messageTemplate);
                fclose(fp);

                if (verbose) {
                    printf("user = %s\n", user);
                    printf("password = %s\n", password);
                    printf("messageTemplate = %s\n", messageTemplate);
                }
            }
        }

        return SuperTweetClient(user, password, messageTemplate, verbose);
    }

    /**
     * tweets message
     *
     * @param format template format string, if empty string specified, messageTemplate specified in the constructor is to be used
     * @param verbose if true display debug info
     */
    bool tweet(char *format, ...) {
        va_list argv;
        va_start(argv, format);

        bool result = false;

        if (format == 0 || format[0] == 0)
            format = &messageTemplate[0];

        ClientSocket client("api.supertweet.net", 80);
        if (client) {
            char message[420], message2[420 * 3];
            vsprintf(message, format, argv);
            int length = Utils::encodeFormUrl(message, message2);

            const char *request = "POST /1/statuses/update.xml HTTP/1.1\r\n"
                                  "Host: api.supertweet.net\r\n"
                                  "Authorization: Basic %s\r\n"
                                  "Content-Length: %d\r\n"
                                  "Content-Type: application/x-www-form-urlencoded\r\n"
                                  "\r\n"
                                  "status=";
            client.printf(request, credential, length + 7);
            client.write(message2, length);

            if (verbose) {
                printf(request, credential, length + 7);
                printf("%s", message2);
            }

            while (client) {
                if (client.available()) {
                    while (client.available()) {
                        char response[128] = {};
                        client.read(response, sizeof(response) - 1);
                        if (!result && strncmp(response, "HTTP/1.1 200", 12) == 0)
                            result = true;
                        if (verbose)
                            printf("%s", response);
                    }
                    client.close();
                }
            }
        }
        return result;
    }

private:
    bool verbose;
    char credential[64];
    char messageTemplate[420];
};

#endif