/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "SimpleSocket.h"
#include "Utils.h"

/**
 * NTP Client interface to adjust RTC
 */
class NTPClient {
public:
    /**
     * creates an NTPClient object
     *
     * @param server NTP server host name
     * @param timeout max wait time for connection
     */
    NTPClient(char *server = "pool.ntp.org", float timeout = 3.0) : timeout(timeout) {
        strncpy(this->server, server, sizeof(this->server));
    }

    /**
     * creates an NTPClient object from a config file
     *
     * @param filename name for the configuration file containing server host name and timeout
     * @param verbose if true display debug info
     */
    static NTPClient create(char *filename, bool verbose = false) {
        char server[32] = "pool.ntp.org";
        float timeout = 3.0;

        if (filename) {
            char path[32];
            LocalFileSystem local("local");
            sprintf(path, "/local/%s", filename);
            if (FILE *fp = fopen(path, "r")) {
                Utils::fgetValues(fp, "ntp-server:%s", server);
                Utils::fgetValues(fp, "ntp-timeout:%f", &timeout);
                fclose(fp);
                if (verbose) {
                    printf("ntp-server:%s\n", server);
                    printf("ntp-timeout:%2.1f\n", timeout);
                }
            }
        }

        return NTPClient(server, timeout);
    }

    /**
     * adjusts RTC
     *
     * @returns true if succeeded
     */
    bool setTime() {
        Timer timer;
        timer.start();
        while (timer.read() < timeout) {
            DatagramSocket datagram;
            char buf[48] = {0x23}; // 00100011 LI(0), Version(4), Mode(3: Client)
            datagram.write(buf, sizeof(buf));
            datagram.send(server, 123);

            if (datagram.receive(0, timeout) > 0) {
                if (datagram.read(buf, sizeof(buf))) {
                    unsigned long seconds = 0;
                    for (int i = 40; i <= 43; i++)
                        seconds = (seconds << 8) | buf[i];
                    set_time(time_t(seconds - 2208988800ULL));
                    return true;
                }
            }
        }

        return false;
    }

private:
    char server[32];
    float timeout;
};