/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef GMCOUNTER_H
#define GMCOUNTER_H

#include "mbed.h"
#include "Utils.h"

/**
 * class for Geiger Mueller Counter
 */
class GMCounter {
public:
    /**
     * Constructor
     *
     * @param gmcPin pin for GMC input
     * @param buzzerPin pin for buzzer output
     * @param ledPin pin for LED output
     * @param cpm2usv conversion ratio of CPM to uSv
     * @param buzzerEnabled if true enable buzzer
     * @param ledEnabled if true enable LED
     */
    GMCounter(PinName gmcPin, PinName buzzerPin, PinName ledPin, float cpm2usv = 1.0F / 120, bool buzzerEnabled = true, bool ledEnabled = true)
            : interrupt(gmcPin), buzzer(buzzerPin), led(ledPin), cpm2usv(cpm2usv), buzzerEnabled(buzzerEnabled), ledEnabled(ledEnabled),
            counter(0), index(0), index60(0), elapsed(0) {
        memset(count, 0, sizeof(count));
        memset(count60, 0, sizeof(count60));
        ticker.attach(this, &GMCounter::tickerHandler, 1);
        ticker60.attach(this, &GMCounter::ticker60Handler, 60);
        interrupt.rise(this, &GMCounter::interruptHandler);
    }

    /**
     * creates an GMCounter object
     *
     * @param gmcPin pin for GMC input
     * @param buzzerPin pin for buzzer output
     * @param ledPin pin for LED output
     * @param filename name of the config file
     * @param verbose if true display debug info
     *
     * @returns GMCounter object
     */
    static GMCounter create(PinName gmcPin, PinName buzzerPin, PinName ledPin, char *filename, bool verbose = false) {
        bool buzzerEnabled = true, ledEnabled = true;
        float cpm2usv = 1.0F / 120;

        if (filename) {
            char path[32];
            LocalFileSystem local("local");
            sprintf(path, "/local/%s", filename);
            if (FILE *fp = fopen(path, "r")) {
                Utils::fgetValues(fp, "set-buzzer:%d", &buzzerEnabled);
                Utils::fgetValues(fp, "set-led:%d", &ledEnabled);
                Utils::fgetValues(fp, "cpm-to-usv:%f", &cpm2usv);
                fclose(fp);
                if (verbose) {
                    printf("set-buzzer:%d\n", buzzerEnabled);
                    printf("set-led:%d\n", ledEnabled);
                    printf("cpm-to-usv:%f\n", cpm2usv);
                }
            }
        }

        return GMCounter(gmcPin, buzzerPin, ledPin, cpm2usv, buzzerEnabled, ledEnabled);
    }

    /**
     * gets CPM (Counts Per Minute) value
     *
     * @returns counts during the last 60 seconds
     */
    int getCPM() {
        // make sure that no index updated while reading the array entries
        // if there was (interrupt happened!), retry reading.
        int i, cpm;
        do {
            i = index;
            cpm = count[(i - 1 + 61) % 61] - count[i];
        } while (i != index);

        return cpm;
    }

    /**
     * gets average CPM (Counts Per Minute)
     *
     * @returns average CPM during the last 60 minutes
     */
    float getAverageCPM() {
        // make sure that no index60 updated while reading the array entries
        // if there was (interrupt happened!), retry reading.
        int i, total;
        do {
            i = index60;
            total = count60[(i - 1 + 61) % 61] - count60[i];
        } while (i != index60);

        return elapsed == 0 ? 0 : elapsed < 60.0 ? total / elapsed : total / 60.0;
    }

    /**
     * Returns radiation
     *
     * @returns radiation in uSv
     */
    float getRadiation() {
        return getCPM() * cpm2usv;
    }

    /**
     * Returns average radiation
     *
     * @returns average radiation during last 60 minutes in uSv
     */
    float getAverageRadiation() {
        return getAverageCPM() * cpm2usv;
    }

    /**
     * returns buzzer status
     *
     * @returns buzzer status
     */
    bool getBuzzer() {
        return buzzerEnabled;
    }

    /**
     * sets buzzer enabled or disabled
     */
    void setBuzzer(bool enable = true) {
        buzzerEnabled = enable;
    }

    /**
     * returns LED status
     *
     * @returns LED status
     */
    bool getLED() {
        return ledEnabled;
    }

    /**
     * sets LED enabled or disabled
     */
    void setLED(bool enable = true) {
        ledEnabled = enable;
    }

    /**
     * sets CPM to uSv conversion ration
     *
     * @param cpm2usv conversion ratio uSv/CPM
     */
    void setConversionRatio(float cpm2usv) {
        this->cpm2usv = cpm2usv;
    }

private:
    InterruptIn interrupt;
    DigitalOut buzzer;
    DigitalOut led;
    Timeout timeout;
    Ticker ticker, ticker60;
    float cpm2usv;
    bool buzzerEnabled;
    bool ledEnabled;
    unsigned int counter;
    unsigned int count[61];
    unsigned int count60[61];
    int index;
    int index60;
    float elapsed;

    void interruptHandler() {
        counter++;
        if (buzzerEnabled) buzzer = 1;
        if (ledEnabled) led = 1;
        timeout.attach(this, &GMCounter::timeoutHandler, 0.03);
    }

    void tickerHandler() {
        count[index] = counter;
        index = (index + 1) % 61;
    }

    void ticker60Handler() {
        if (elapsed < 60) elapsed++;
        count60[index60] = counter;
        index60 = (index60 + 1) % 61;
    }

    void timeoutHandler() {
        if (buzzerEnabled) buzzer = 0;
        if (ledEnabled) led = 0;
    }
};

#endif