/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef SCP1000_H
#define SCP1000_H

#include "mbed.h"

/**
 * SCP1000 pressure/temperature sensor
 */
class SCP1000 {
public:

    /***/
    enum OperationMode {
        CANCEL = 0x00,
        HIGH_SPEED_MODE = 0x09,
        HIGH_RESOLUTION_MODE = 0x0A,
        ULTRA_LOW_POWER_MODE = 0x0B,
        LOW_POWER = 0x0C,
        SELF_TEST = 0x0F,
    };
    
    /**
     * creates an SCP1000 interface with specified pins
     *
     * @param PinName mosi SPI MOSI pin
     * @param PinName miso SPI MISO pin
     * @param PinName sclk SPI SCLK pin
     * @param PinName cs SPI CS pin
     * @param OperationMode mode Initial operation mode
     */
    SCP1000(PinName mosi, PinName miso, PinName sclk, PinName cs, OperationMode mode = HIGH_RESOLUTION_MODE);

    ~SCP1000() {};

    /**
     * reads the revision number
     *
     * @returns revision number
     */
    int revision();

    /**
     * checks if data is ready
     *
     * @returns true if data is ready, false otherwise
     */
    bool isReady();

    /**
     * performs software reset
     */
    void reset();

    /**
     * sets the operation mode
     */
    void setOperationMode(OperationMode mode);

    /**
     * performs self test
     *
     * @returns the result of the test (true if succeeded, false otherwise)
     */
    bool performSelfTest();

    /**
     * reads the current pressure
     *
     * @param waitForDataReady if true, it will wait until data becomes ready
     *
     * @returns pressure in hectopascal (hPa)
     */
    float readPressure(bool waitForDataReady = true);

    /**
     * reads the current temperature
     *
     * @param waitForDataReady if true, it will wait until data becomes ready
     *
     * @returns temperature in Celsius
     */
    float readTemperature(bool waitForDataReady = true);

private:
    enum RegisterName {
        REVID = 0x00,         // ASIC revision number
        OPERATION = 0x03,     // Operation register
        OPSTATUS = 0x04,      // Operation status
        RSTR = 0x06,          // ASIC software reset
        STATUS = 0x07,        // ASIC top-level status
        DATARD8 = 0x1F,       // Pressure output data (MSB)
        DATARD16 = 0x20,      // Pressure output data (LSB, 16 bits)
        TEMPOUT = 0x21,       // Temperature output data (16 bits)
    };

    enum Command {
        READ = 0x00,          // SCP1000 READ command
        WRITE = 0x02,         // SCP1000 WRITE command
    };

    SPI spi;
    DigitalOut selectPin;

    int readRegister(RegisterName registerName, bool twoBytes = 0);
    void writeRegister(RegisterName registerName, char value);
};

#endif