/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "SCP1000.h"

SCP1000::SCP1000(PinName mosi, PinName miso, PinName sclk, PinName cs, OperationMode mode)
        : spi(mosi, miso, sclk), selectPin(cs) {
    selectPin = 1;
    spi.frequency(500000);
    spi.format(8, 0);   // frame bits = 8, mode = 0
    reset();
    setOperationMode(mode);
}

int SCP1000::revision() {
    return readRegister(REVID);
}

bool SCP1000::isReady() {
    const char DRDY = 0x20; // Data Ready
    
    return readRegister(STATUS) & DRDY;
}

void SCP1000::setOperationMode(OperationMode mode) {
    writeRegister(OPERATION, CANCEL);
    while (readRegister(OPSTATUS)) // wait until current operation is canncelled
        wait(0.01);
    if (mode != CANCEL)
        writeRegister(OPERATION, mode);
}

void SCP1000::reset() {
    writeRegister(RSTR, 0x01);
    wait(0.15);
}

bool SCP1000::performSelfTest() {
    setOperationMode(SELF_TEST);
    wait(1); // wait enough time for the test to completes (> 0.25secs)
    return readRegister(DATARD8) & 1;
}

float SCP1000::readPressure(bool waitForReady) {
    if (waitForReady)
        while (!isReady())
            wait(0.01);

    // Pressure value is stored in 19-bit unsigned int (unit: 0.25 Pa)
    long value = readRegister(DATARD8) & 0x07;
    value <<= 16;
    value |= readRegister(DATARD16, true) & 0xFFFF;

    return value / 400.0;
}

float SCP1000::readTemperature(bool waitForDataReady) {
    if (waitForDataReady)
        while (!isReady())
            wait(0.01);

    return ((short)(readRegister(TEMPOUT, true) << 3) >> 3) / 20.0;
}

int SCP1000::readRegister(RegisterName registerName, bool twoBytes) {
    selectPin = 0;
    spi.write((registerName << 2) | READ);
    int result = spi.write(0x00);
    if (twoBytes) {
        result <<= 8;
        result |= spi.write(0x00);
    }
    selectPin = 1;

    return result;
}

void SCP1000::writeRegister(RegisterName registerName, char value) {
    selectPin = 0;
    spi.write((registerName << 2) | WRITE);
    spi.write(value);
    selectPin = 1;
}