#ifndef MMA7361L_H
#define MMA7361L_H

#include "mbed.h"

/**
 * MMA7361L accelerometer
 */
class MMA7361L {
public:
    /**
     * Creates an MMA7361L accelerometer interface, connected to the specified pins
     *
     * @param xoutPin xout pin
     * @param youtPin yout pin
     * @param zoutPin zout pin
     * @param zeroGDetectPin 0G detect pin
     * @param gSelectPin select 1.5G/6G pin
     * @param nSleepPin ~sleep pin
     *
     */
    MMA7361L(PinName xoutPin, PinName youtPin, PinName zoutPin, PinName zeroGDetectPin = NC, PinName gSelectPin = NC, PinName sleepPin = NC);

    /**
    * Scale enumeration declaration
    */
    enum Scale {

        /**
         * 1.5G mode
         */
        SCALE_1_5G,

        /**
         * 6G mode
         */
        SCALE_6G
    };

    /**
     * Gets the current total acceleration
     *
     * @param forceRead true to read the device, false to use cache whenever appropriate
     * @returns total acceleration in g
     */
    float getAccel(bool forceRead = false);

    /**
     * Gets the current acceleration along the X axis
     *
     * @param forceRead true to read the device, false to use cache whenever appropriate
     * @returns acceleration along the X axis
     */
    float getAccelX(bool forceRead = false);

    /**
     * Gets the current acceleration along the Y axis
     *
     * @param forceRead true to read the device, false to use cache whenever appropriate
     * @returns acceleration along the Y axis
     */
    float getAccelY(bool forceRead = false);

    /**
     * Gets the current acceleration along the Z axis
     *
     * @param forceRead true to read the device, false to use cache whenever appropriate
     * @returns acceleration along the Z axis
     */
    float getAccelZ(bool forceRead = false);

    /**
     * Computes the inclination of the X axis
     *
     * @param forceRead true to read the device, false to use cache whenever appropriate
     * @returns the inclination of the X axis
     */
    float getTiltX(bool forceRead = false);

    /**
     * Computes the inclination of the Y axis
     *
     * @param forceRead true to read the device, false to use cache whenever appropriate
     * @returns the inclination of the Y axis
     */
    float getTiltY(bool forceRead = false);

    /**
     * Computes the inclination of the Z axis
     *
     * @param forceRead true to read the device, false to use cache whenever appropriate
     * @returns the inclination of the Z axis
     */
    float getTiltZ(bool forceRead = false);

    /**
     * Specifies the scale to use
     *
     * @param scale SCALE_1_5G (for 1.5G) or SCALE_6G (for 6G)
     */
    void setScale(Scale scale);

    /**
     * Sets sleep mode
     *
     * @param on true to activate sleep mode, false to resume normal operation
     */
    void setSleep(bool on);

    /**
     * Tests whether 0G is detected
     *
     * @returns true if 0G is detected, false otherwise
     */
    bool zeroGDetected();

    /**
     * Sets fucntion to be called when 0G is detected
     *
     * @param func pointer to a function called when 0G is detected, 0 to reset as none
     */
    void setZeroGDetectListener(void (*func)(void));

    /**
     * Sets member fucntion to be called when 0G is detected
     *
     * @param t pointer to the object to call the member function on
     * @param func pointer to the member function to be called when 0G is detected, 0 to reset as none
     */
    template<typename T>
    void setZeroGDetectListener(T* t, void (T::*func)(void));

    /**
    * Prints instance info for debugging
    */
    void printInfo();

    /**
     * Sets calibration info
     *
     * @param scale scale for calibration
     * @param minX min X value when the X axis is vertical and stable under 1g
     * @param maxX max X value when the X axis is vertical and stable under 1g
     * @param minY min Y value when the Y axis is vertical and stable under 1g
     * @param maxY max Y value when the Y axis is vertical and stable under 1g
     * @param minZ min Z value when the Z axis is vertical and stable under 1g
     * @param maxZ max Z value when the Z axis is vertical and stable under 1g
     */

    void calibrate(Scale scale, float minX, float maxX, float minY, float maxY, float minZ, float maxZ);

private:
    enum {
        ACCEL = 1 << 0,
        ACCEL_X = 1 << 1,
        ACCEL_Y = 1 << 2,
        ACCEL_Z = 1 << 3,
        TILT_X = 1 << 4,
        TILT_Y = 1 << 5,
        TILT_Z = 1 << 6
    };

    AnalogIn xout, yout, zout;
    InterruptIn zeroGDetect;
    DigitalOut gSelect;
    DigitalOut nSleep;
    bool zeroGDetectEnabled;
    bool gSelectEnabled;
    bool sleepEnabled;
    Scale scale;
    float ratio;
    float accelX, accelY, accelZ;
    int flags;
    struct Calibration {
        float minX, maxX;
        float minY, maxY;
        float minZ, maxZ;
    } calib[2];

    void prepare(int type, bool forceRead);
    float calibrate(float value, int type, Scale scale);
};

#endif