#ifndef ACCELEROMETER_H
#define ACCELEROMETER_H

#include "mbed.h"

/**
 *  MMA7260Q Accelerometer
 */
class Accelerometer {
public:
    /**
     *  Creates an MMA7260Q accelerometer interface, connected to the specified pins
     *
     *  @param xPin X pin
     *  @param yPin Y pin
     *  @param zPin Z pin
     *  @param gs1Pin 1.5G/2G/4G/6G select pin
     *  @param gs2Pin 1.5G/2G/4G/6G select pin
     *  @param slpPin Sleep pin
     *
     */
    Accelerometer(PinName xPin, PinName yPin, PinName zPin, PinName gs1Pin, PinName gs2Pin, PinName slpPin);
    
    /***/
    enum Scale {
        SCALE_1_5G /** 1.5G mode */,
        SCALE_2G   /** 2G mode */,
        SCALE_4G   /** 4G mode */,
        SCALE_6G   /** 6G mode */
    };

    /**
     *  get the current total acceleration
     *
     *  @returns total acceleration in g
     */
    float getAccel();

    /**
     *  get the current acceleration along the X axis
     *
     *  @returns acceleration along the X axis
     */
    float getAccelX();

    /**
    *  get the current acceleration along the Y axis
    *
    *  @returns acceleration along the Y axis
    */
    float getAccelY();

    /**
    *  get the current acceleration along the Z axis
    *
    *  @returns acceleration along the Z axis
    */
    float getAccelZ();

    /* Function: getTiltX
    *   compute the inclination of the X axis
    *
    *   @returns the inclination of the X axis
    */
    float getTiltX();

    /* Function: getTiltY
    *   compute the inclination of the Y axis
    *
    * Returns:
    *   the inclination of the Y axis
    */
    float getTiltY();

    /* Function: getTiltZ
    *   compute the inclination of the Z axis
    *
    * Returns:
    *   the inclination of the Z axis
    */
    float getTiltZ();

    /* Function: setScale
    *   specify the scale to use
    *
    * Variables:
    *   scale - SCALE_1_5G (for 1.5G) or SCALE_6G (for 6G)
    */
    void setScale(Scale scale);

    /* Function: setSleep
     *   activate sleep mode
     *
     * Variables:
     *   on - true to activate sleep mode, false to resume normal operation
     */
    void setSleep(bool on);

private:
    AnalogIn xout;
    AnalogIn yout;
    AnalogIn zout;
    DigitalOut gs1;
    DigitalOut gs2;
    DigitalOut sleep;
    float scale;
};

#endif